"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createVm = exports.createVmTemplate = void 0;

var _lodash = require("lodash");

var _templates = require("../utils/templates");

var _selectors = require("../utils/selectors");

var _models = require("../models");

var _constants = require("../constants");

var _constants2 = require("../components/Wizard/CreateVmWizard/constants");

var _selectors2 = require("./selectors");

var _cloudInit = require("./cloudInit");

var _clone = require("./clone");

var _utils = require("../utils");

var _vmBuilder = require("./vmBuilder");

var _vmwareProviderPod = require("../components/Wizard/CreateVmWizard/providers/vmwareProviderPod");

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const FALLBACK_DISK = {
  disk: {
    bus: 'virtio'
  }
};

const enhancedCreate = (k8sCreate, kind, data, aditionalObjects = [], ...rest) => k8sCreate(kind, data, ...rest).catch(error => // eslint-disable-next-line prefer-promise-reject-errors
Promise.reject({
  error,
  message: error.message,
  objects: [data, ...aditionalObjects]
}));

const createVmTemplate = async (k8sCreate, templates, basicSettings, networks, storage, persistentVolumeClaims) => {
  const create = enhancedCreate.bind(undefined, k8sCreate);

  const getSetting = param => {
    switch (param) {
      case _constants2.NAME_KEY:
        return `\${${_constants.TEMPLATE_PARAM_VM_NAME}}`;

      case _constants2.DESCRIPTION_KEY:
        return undefined;

      default:
        return (0, _selectors2.settingsValue)(basicSettings, param);
    }
  };

  const template = getModifiedVmTemplate(templates, basicSettings, getSetting, networks, storage, persistentVolumeClaims);
  const vmTemplate = createTemplateObject(_selectors2.settingsValue.bind(undefined, basicSettings), (0, _selectors2.selectVm)(template.objects), template);
  let bootDataVolume;

  if ((0, _selectors2.settingsValue)(basicSettings, _constants2.PROVISION_SOURCE_TYPE_KEY) === _constants.PROVISION_SOURCE_URL) {
    const bootStorage = storage.find(s => s.isBootable);
    const vm = (0, _selectors2.selectVm)(template.objects);
    const bootVolume = (0, _vmBuilder.getVolumes)(vm).find(v => v.name === bootStorage.name);
    const dataVolumeTemplates = (0, _vmBuilder.getDataVolumeTemplates)(vm);
    bootDataVolume = dataVolumeTemplates.find(t => (0, _selectors.getName)(t) === bootVolume.dataVolume.name);
    const newDataVolumeTemplates = dataVolumeTemplates.filter(t => (0, _selectors.getName)(t) !== bootVolume.dataVolume.name);
    vm.spec.dataVolumeTemplates = newDataVolumeTemplates;
    const newDataVolumeName = (0, _utils.generateDiskName)((0, _selectors2.settingsValue)(basicSettings, _constants2.NAME_KEY), bootStorage.name);
    bootVolume.dataVolume.name = newDataVolumeName;
    bootDataVolume.metadata.name = newDataVolumeName;
    bootDataVolume.metadata.namespace = (0, _selectors2.settingsValue)(basicSettings, _constants2.NAMESPACE_KEY);
    bootDataVolume.apiVersion = `${_models.DataVolumeModel.apiGroup}/${_models.DataVolumeModel.apiVersion}`;
    bootDataVolume.kind = _models.DataVolumeModel.kind;
  }

  const templateResult = await create(_models.TemplateModel, vmTemplate);
  const results = [templateResult];

  if (bootDataVolume) {
    bootDataVolume.metadata.ownerReferences = [{
      apiVersion: templateResult.apiVersion,
      blockOwnerDeletion: true,
      controller: true,
      kind: templateResult.kind,
      name: (0, _selectors.getName)(templateResult),
      uid: templateResult.metadata.uid
    }];
    const dvResult = await create(_models.DataVolumeModel, bootDataVolume);
    results.push(dvResult);
  }

  return results;
};

exports.createVmTemplate = createVmTemplate;

const createVm = async (k8sCreate, templates, basicSettings, networks, storage, persistentVolumeClaims) => {
  const create = enhancedCreate.bind(undefined, k8sCreate);

  const getSetting = _selectors2.settingsValue.bind(undefined, basicSettings);

  const template = getModifiedVmTemplate(templates, basicSettings, getSetting, networks, storage, persistentVolumeClaims); // ProcessedTemplates endpoit will reject the request if user cannot post to the namespace
  // common-templates are stored in openshift namespace, default user can read but cannot post

  const postTemplate = (0, _lodash.cloneDeep)(template);
  postTemplate.metadata.namespace = (0, _selectors2.settingsValue)(basicSettings, _constants2.NAMESPACE_KEY);
  const processedTemplate = await create(_models.ProcessedTemplatesModel, postTemplate);
  const vm = (0, _selectors2.selectVm)(processedTemplate.objects);
  addMetadata(vm, template, getSetting);
  const namespace = getSetting(_constants2.NAMESPACE_KEY);

  if (namespace) {
    vm.metadata.namespace = namespace;
  }

  const results = [];
  const importProviderSecret = getImportProviderSecret(getSetting);

  if (importProviderSecret) {
    results.push((await k8sCreate(_models.SecretModel, importProviderSecret)));
  }

  results.push((await k8sCreate(_models.VirtualMachineModel, vm)));
  return results;
};

exports.createVm = createVm;

const getImportProviderSecret = getSetting => {
  if (getSetting(_constants2.PROVIDER_VMWARE_USER_PWD_REMEMBER_KEY) && getSetting(_constants2.PROVIDER_VMWARE_VCENTER_KEY) && getSetting(_constants2.PROVISION_SOURCE_TYPE_KEY) === _constants.PROVISION_SOURCE_IMPORT) {
    const url = getSetting(_constants2.PROVIDER_VMWARE_URL_KEY);
    const username = getSetting(_constants2.PROVIDER_VMWARE_USER_NAME_KEY);
    const password = (0, _lodash.get)(getSetting(_constants2.PROVIDER_VMWARE_USER_PWD_AND_CHECK_KEY), _constants2.PROVIDER_VMWARE_USER_PWD_KEY);
    const namespace = getSetting(_constants2.NAMESPACE_KEY);
    return (0, _vmwareProviderPod.getImportProviderSecretObject)({
      url,
      username,
      password,
      namespace
    });
  }

  return null;
};

const getModifiedVmTemplate = (templates, basicSettings, getSetting, networks, storage, persistentVolumeClaims) => {
  const template = resolveTemplate(templates, basicSettings, getSetting);
  const vm = (0, _selectors2.selectVm)(template.objects);
  modifyVmObject(vm, template, getSetting, networks, storage, persistentVolumeClaims);
  return template;
};

const createTemplateObject = (getSetting, vm, template) => {
  const vmTemplate = {
    apiVersion: `${_models.TemplateModel.apiGroup}/${_models.TemplateModel.apiVersion}`,
    kind: _models.TemplateModel.kind,
    metadata: {
      name: getSetting(_constants2.NAME_KEY),
      namespace: getSetting(_constants2.NAMESPACE_KEY)
    },
    objects: [vm],
    parameters: [{
      name: _constants.TEMPLATE_PARAM_VM_NAME,
      description: _constants.TEMPLATE_PARAM_VM_NAME_DESC
    }]
  };
  const description = getSetting(_constants2.DESCRIPTION_KEY);

  if (description) {
    (0, _vmBuilder.addAnnotation)(vmTemplate, 'description', description);
  }

  addMetadata(vmTemplate, template, getSetting);
  (0, _vmBuilder.addLabel)(vmTemplate, [_constants.TEMPLATE_TYPE_LABEL], _constants.TEMPLATE_TYPE_VM);
  return vmTemplate;
};

const resolveTemplate = (templates, basicSettings, getSetting) => {
  let chosenTemplate;

  if (getSetting(_constants2.USER_TEMPLATE_KEY)) {
    chosenTemplate = templates.find(template => template.metadata.name === getSetting(_constants2.USER_TEMPLATE_KEY));

    if (!chosenTemplate) {
      return null;
    }

    chosenTemplate = (0, _lodash.cloneDeep)(chosenTemplate);
  } else {
    const baseTemplates = (0, _templates.getTemplatesWithLabels)((0, _templates.getTemplate)(templates, _constants.TEMPLATE_TYPE_BASE), [(0, _selectors2.getOsLabel)(basicSettings), (0, _selectors2.getWorkloadLabel)(basicSettings), (0, _selectors2.getFlavorLabel)(basicSettings)]);

    if (baseTemplates.length === 0) {
      return null;
    }

    chosenTemplate = (0, _lodash.cloneDeep)(baseTemplates[0]);
  }

  setParameterValue(chosenTemplate, _constants.TEMPLATE_PARAM_VM_NAME, getSetting(_constants2.NAME_KEY)); // no more required parameters

  chosenTemplate.parameters.forEach(param => {
    if (param.name !== _constants.TEMPLATE_PARAM_VM_NAME && param.required) {
      delete param.required;
    }
  }); // make sure api version is correct

  chosenTemplate.apiVersion = _constants.TEMPLATE_API_VERSION;
  return chosenTemplate;
};

const addMetadata = (vm, template, getSetting) => {
  const flavor = getSetting(_constants2.FLAVOR_KEY);
  (0, _vmBuilder.addLabel)(vm, `${_constants.TEMPLATE_FLAVOR_LABEL}/${flavor}`, 'true');
  const os = getSetting(_constants2.OPERATING_SYSTEM_KEY);
  (0, _vmBuilder.addLabel)(vm, `${_constants.TEMPLATE_OS_LABEL}/${os.id}`, 'true');
  const workload = getSetting(_constants2.WORKLOAD_PROFILE_KEY);
  (0, _vmBuilder.addLabel)(vm, `${_constants.TEMPLATE_WORKLOAD_LABEL}/${workload}`, 'true');
  (0, _vmBuilder.addLabel)(vm, _constants.LABEL_USED_TEMPLATE_NAME, (0, _selectors.getName)(template));
  (0, _vmBuilder.addLabel)(vm, _constants.LABEL_USED_TEMPLATE_NAMESPACE, (0, _selectors.getNamespace)(template));
  (0, _vmBuilder.addTemplateLabel)(vm, _constants.TEMPLATE_VM_NAME_LABEL, vm.metadata.name); // for pairing service-vm (like for RDP)

  (0, _vmBuilder.addAnnotation)(vm, `${_constants.TEMPLATE_OS_NAME_ANNOTATION}/${os.id}`, os.name);
};

const modifyVmObject = (vm, template, getSetting, networks, storages, persistentVolumeClaims) => {
  setFlavor(vm, getSetting);
  addNetworks(vm, template, getSetting, networks);
  vm.spec.running = getSetting(_constants2.START_VM_KEY, false);
  const description = getSetting(_constants2.DESCRIPTION_KEY);

  if (description) {
    (0, _vmBuilder.addAnnotation)(vm, 'description', description);
  }

  addStorages(vm, template, storages, getSetting, persistentVolumeClaims);
};

const setFlavor = (vm, getSetting) => {
  if (getSetting(_constants2.FLAVOR_KEY) === _constants.CUSTOM_FLAVOR) {
    vm.spec.template.spec.domain.cpu.cores = parseInt(getSetting(_constants2.CPU_KEY), 10);
    vm.spec.template.spec.domain.resources.requests.memory = `${getSetting(_constants2.MEMORY_KEY)}G`;
  }
};

const setParameterValue = (template, paramName, paramValue) => {
  const parameter = template.parameters.find(param => param.name === paramName);
  parameter.value = paramValue;
};

const addNetworks = (vm, template, getSetting, networks) => {
  const defaultInterface = getDefaultInterface(vm, template) || {
    bridge: {}
  };
  (0, _vmBuilder.removeInterfacesAndNetworks)(vm);

  if (!networks.find(network => network.network === _constants.POD_NETWORK)) {
    (0, _vmBuilder.getDevices)(vm).autoattachPodInterface = false;
  }

  networks.forEach(network => {
    (0, _vmBuilder.addInterface)(vm, defaultInterface, network);
    (0, _vmBuilder.addNetwork)(vm, network);
  });

  if (getSetting(_constants2.PROVISION_SOURCE_TYPE_KEY) === _constants.PROVISION_SOURCE_PXE) {
    (0, _vmBuilder.addAnnotation)(vm, _constants.ANNOTATION_PXE_INTERFACE, networks.find(network => network.isBootable).name);
    const startOnCreation = getSetting(_constants2.START_VM_KEY, false);
    (0, _vmBuilder.addAnnotation)(vm, _constants.ANNOTATION_FIRST_BOOT, `${!startOnCreation}`);
  }
};

const addCloudInit = (vm, defaultDisk, getSetting) => {
  if (getSetting(_constants2.CLOUD_INIT_KEY)) {
    const existingCloudInitVolume = (0, _selectors.getCloudInitVolume)(vm);
    const cloudInit = new _cloudInit.CloudInit({
      volume: existingCloudInitVolume
    });

    if (getSetting(_constants2.USE_CLOUD_INIT_CUSTOM_SCRIPT_KEY)) {
      cloudInit.setUserData(getSetting(_constants2.CLOUD_INIT_CUSTOM_SCRIPT_KEY));
    } else {
      cloudInit.setPredefinedUserData({
        hostname: getSetting(_constants2.HOST_NAME_KEY),
        sshAuthorizedKeys: getSetting(_constants2.AUTHKEYS_KEY)
      });
    }

    const _cloudInit$build = cloudInit.build(),
          volume = _cloudInit$build.volume,
          disk = _cloudInit$build.disk;

    if (volume !== existingCloudInitVolume) {
      (0, _vmBuilder.addDisk)(vm, defaultDisk, disk, getSetting);
      (0, _vmBuilder.addVolume)(vm, volume);
    }
  }
};

const addStorages = (vm, template, storages, getSetting, persistentVolumeClaims) => {
  const defaultDisk = getDefaultDisk(vm, template);
  (0, _vmBuilder.removeDisksAndVolumes)(vm);

  if (storages) {
    storages.forEach(storage => {
      switch (storage.storageType) {
        case _constants2.STORAGE_TYPE_PVC:
          addPvcVolume(vm, storage, getSetting, persistentVolumeClaims);
          break;

        case _constants2.STORAGE_TYPE_DATAVOLUME:
          addDataVolumeVolume(vm, storage, getSetting);
          break;

        case _constants2.STORAGE_TYPE_CONTAINER:
          (0, _vmBuilder.addContainerVolume)(vm, storage, getSetting);
          break;

        default:
          if (storage.templateStorage) {
            (0, _vmBuilder.addVolume)(vm, storage.templateStorage.volume);
          }

      }

      (0, _vmBuilder.addDisk)(vm, defaultDisk, storage, getSetting);
    });
  }

  addCloudInit(vm, defaultDisk, getSetting);
};

const addPvcVolume = (vm, storage, getSetting, persistentVolumeClaims) => {
  const pvc = persistentVolumeClaims.find(p => (0, _selectors.getName)(p) === storage.name && (0, _selectors.getNamespace)(p) === getSetting(_constants2.NAMESPACE_KEY));
  const dvTemplate = (0, _clone.addTemplateClone)(vm, storage.name, getSetting(_constants2.NAMESPACE_KEY), (0, _selectors.getPvcAccessModes)(pvc), (0, _selectors.getPvcStorageSize)(pvc), (0, _selectors.getPvcStorageClassName)(pvc), getSetting(_constants2.NAME_KEY));
  (0, _vmBuilder.addDataVolume)(vm, {
    name: storage.name,
    dvName: (0, _selectors.getName)(dvTemplate)
  });
};

const addDataVolumeVolume = (vm, storage, getSetting) => {
  const dvStorage = _objectSpread({}, storage);

  if (dvStorage.templateStorage) {
    if (dvStorage.templateStorage.dataVolume) {
      const dataVolume = dvStorage.templateStorage.dataVolume;
      const dataVolumeTemplate = (0, _clone.addTemplateClone)(vm, (0, _selectors.getName)(dataVolume), (0, _selectors.getNamespace)(dataVolume), (0, _selectors.getDataVolumeAccessModes)(dataVolume), (0, _selectors.getDataVolumeStorageSize)(dataVolume), // TODO should take storage specified by user on storage page
      (0, _selectors.getDataVolumeStorageClassName)(dataVolume), getSetting(_constants2.NAME_KEY));
      dvStorage.dvName = (0, _selectors.getName)(dataVolumeTemplate);
    } else if (dvStorage.templateStorage.dataVolumeTemplate) {
      const dataVolumeTemplate = dvStorage.templateStorage.dataVolumeTemplate;
      dataVolumeTemplate.metadata.name = (0, _utils.generateDiskName)(getSetting(_constants2.NAME_KEY), storage.name, false);
      (0, _vmBuilder.addDataVolumeTemplate)(vm, dataVolumeTemplate);
      dvStorage.dvName = dataVolumeTemplate.metadata.name;
    }
  } else {
    const source = getSetting(_constants2.PROVISION_SOURCE_TYPE_KEY) === _constants.PROVISION_SOURCE_URL && storage.isBootable ? {
      type: _constants2.DATA_VOLUME_SOURCE_URL,
      url: getSetting(_constants2.IMAGE_URL_KEY)
    } : {
      type: _constants2.DATA_VOLUME_SOURCE_BLANK
    };
    dvStorage.dvName = (0, _utils.generateDiskName)(getSetting(_constants2.NAME_KEY), storage.name);
    const dataVolumeSpec = (0, _vmBuilder.getDataVolumeTemplateSpec)(dvStorage, source);
    (0, _vmBuilder.addDataVolumeTemplate)(vm, dataVolumeSpec);
  }

  (0, _vmBuilder.addDataVolume)(vm, dvStorage);
};

const getDefaultDisk = (vm, template) => {
  const defaultDiskName = (0, _selectors2.getTemplateAnnotations)(template, _constants.ANNOTATION_DEFAULT_DISK);
  return (0, _vmBuilder.getDevice)(vm, 'disks', defaultDiskName) || FALLBACK_DISK;
};

const getDefaultInterface = (vm, template) => {
  const defaultInterfaceName = (0, _selectors2.getTemplateAnnotations)(template, _constants.ANNOTATION_DEFAULT_NETWORK);
  return (0, _vmBuilder.getDevice)(vm, 'interfaces', defaultInterfaceName);
};