"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BasicSettingsTab = exports.onCloseBasic = exports.getFormFields = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _lodash = require("lodash");

var _Form = require("../../Form");

var _selectors = require("../../../utils/selectors");

var _templates = require("../../../utils/templates");

var _validations = require("../../../utils/validations");

var _strings = require("./strings");

var _selectors2 = require("../../../k8s/selectors");

var _constants = require("../../../constants");

var _constants2 = require("./constants");

var _providers = require("./providers");

var _models = require("../../../models");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _slicedToArray(arr, i) { return _arrayWithHoles(arr) || _iterableToArrayLimit(arr, i) || _nonIterableRest(); }

function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance"); }

function _iterableToArrayLimit(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"] != null) _i["return"](); } finally { if (_d) throw _e; } } return _arr; }

function _arrayWithHoles(arr) { if (Array.isArray(arr)) return arr; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const getProvisionSourceHelp = basicSettings => {
  const provisionSource = (0, _selectors2.settingsValue)(basicSettings, _constants2.PROVISION_SOURCE_TYPE_KEY);

  switch (provisionSource) {
    case _constants.PROVISION_SOURCE_URL:
      return _strings.HELP_PROVISION_SOURCE_URL;

    case _constants.PROVISION_SOURCE_PXE:
      return _strings.HELP_PROVISION_SOURCE_PXE;

    case _constants.PROVISION_SOURCE_CONTAINER:
      return _strings.HELP_PROVISION_SOURCE_CONTAINER;

    case _constants.PROVISION_SOURCE_IMPORT:
      return _strings.HELP_PROVISION_SOURCE_IMPORT;

    default:
      return null;
  }
};

const getFormFields = (basicSettings, namespaces, templates, selectedNamespace, createTemplate, WithResources, k8sCreate, k8sGet, k8sPatch) => {
  const userTemplate = (0, _lodash.get)(basicSettings[_constants2.USER_TEMPLATE_KEY], 'value');
  const workloadProfiles = (0, _selectors2.getWorkloadProfiles)(basicSettings, templates, userTemplate);
  const operatingSystems = (0, _selectors2.getOperatingSystems)(basicSettings, templates, userTemplate);
  const flavors = (0, _selectors2.getFlavors)(basicSettings, templates, userTemplate);
  const imageSources = [_constants.PROVISION_SOURCE_PXE, _constants.PROVISION_SOURCE_URL, _constants.PROVISION_SOURCE_CONTAINER];
  const userTemplateNames = (0, _templates.getTemplate)(templates, _constants.TEMPLATE_TYPE_VM).map(_selectors.getName);
  userTemplateNames.push(_strings.NO_TEMPLATE);
  let namespaceDropdown;
  let startVmCheckbox;
  let userTemplateDropdown;
  let providersSection = {};

  if (!selectedNamespace) {
    namespaceDropdown = {
      id: 'namespace-dropdown',
      title: 'Namespace',
      type: _Form.DROPDOWN,
      defaultValue: '--- Select Namespace ---',
      choices: namespaces.map(_selectors.getName),
      required: true
    };
  }

  if (!createTemplate) {
    startVmCheckbox = {
      id: 'start-vm',
      title: 'Start virtual machine on creation',
      type: _Form.CHECKBOX,
      noBottom: true
    };
    userTemplateDropdown = {
      id: 'template-dropdown',
      title: 'Template',
      type: _Form.DROPDOWN,
      defaultValue: '--- Select Template ---',
      choices: userTemplateNames
    };
    providersSection = (0, _providers.importProviders)(basicSettings, WithResources, k8sCreate, k8sGet, k8sPatch);
    imageSources.push(_constants.PROVISION_SOURCE_IMPORT);
  }

  return _objectSpread({
    [_constants2.NAME_KEY]: {
      id: 'vm-name',
      title: 'Name',
      required: true,
      validate: settings => (0, _validations.validateDNS1123SubdomainValue)((0, _selectors2.settingsValue)(settings, _constants2.NAME_KEY))
    },
    [_constants2.DESCRIPTION_KEY]: {
      id: 'vm-description',
      title: 'Description',
      type: _Form.TEXT_AREA
    },
    [_constants2.NAMESPACE_KEY]: namespaceDropdown,
    [_constants2.USER_TEMPLATE_KEY]: userTemplateDropdown,
    [_constants2.PROVISION_SOURCE_TYPE_KEY]: {
      id: 'image-source-type-dropdown',
      title: 'Provision Source',
      type: _Form.DROPDOWN,
      defaultValue: '--- Select Provision Source ---',
      choices: imageSources,
      required: true,
      disabled: userTemplate !== undefined,
      help: getProvisionSourceHelp(basicSettings)
    },
    [_constants2.CONTAINER_IMAGE_KEY]: {
      id: 'provision-source-container',
      title: 'Container Image',
      required: true,
      isVisible: basicVmSettings => (0, _selectors2.isImageSourceType)(basicVmSettings, _constants.PROVISION_SOURCE_CONTAINER),
      disabled: userTemplate !== undefined,
      validate: settings => (0, _validations.validateContainer)((0, _selectors2.settingsValue)(settings, _constants2.CONTAINER_IMAGE_KEY))
    },
    [_constants2.IMAGE_URL_KEY]: {
      id: 'provision-source-url',
      title: 'URL',
      required: true,
      isVisible: basicVmSettings => (0, _selectors2.isImageSourceType)(basicVmSettings, _constants.PROVISION_SOURCE_URL),
      disabled: userTemplate !== undefined,
      validate: settings => (0, _validations.validateURL)((0, _selectors2.settingsValue)(settings, _constants2.IMAGE_URL_KEY))
    }
  }, providersSection, {
    [_constants2.OPERATING_SYSTEM_KEY]: {
      id: 'operating-system-dropdown',
      title: 'Operating System',
      type: _Form.DROPDOWN,
      defaultValue: '--- Select Operating System ---',
      choices: operatingSystems,
      required: true,
      disabled: userTemplate !== undefined,
      help: _strings.HELP_OS
    },
    [_constants2.FLAVOR_KEY]: {
      id: 'flavor-dropdown',
      title: 'Flavor',
      type: _Form.DROPDOWN,
      defaultValue: '--- Select Flavor ---',
      choices: flavors,
      required: true,
      disabled: userTemplate !== undefined && flavors.length === 1,
      help: _strings.HELP_FLAVOR
    },
    [_constants2.MEMORY_KEY]: {
      id: 'resources-memory',
      title: 'Memory (GB)',
      type: _Form.POSITIVE_NUMBER,
      required: true,
      isVisible: basicVmSettings => (0, _selectors2.isFlavorType)(basicVmSettings, _constants.CUSTOM_FLAVOR),
      help: _strings.HELP_MEMORY
    },
    [_constants2.CPU_KEY]: {
      id: 'resources-cpu',
      title: 'CPUs',
      type: _Form.POSITIVE_NUMBER,
      required: true,
      isVisible: basicVmSettings => (0, _selectors2.isFlavorType)(basicVmSettings, _constants.CUSTOM_FLAVOR),
      help: _strings.HELP_CPU
    },
    [_constants2.WORKLOAD_PROFILE_KEY]: {
      id: 'workload-profile-dropdown',
      title: 'Workload Profile',
      type: _Form.DROPDOWN,
      defaultValue: '--- Select Workload Profile ---',
      choices: workloadProfiles,
      required: true,
      help: _strings.HELP_WORKLOAD,
      disabled: userTemplate !== undefined
    },
    [_constants2.START_VM_KEY]: startVmCheckbox,
    [_constants2.CLOUD_INIT_KEY]: {
      id: 'use-cloud-init',
      title: 'Use cloud-init',
      type: _Form.CHECKBOX
    },
    [_constants2.USE_CLOUD_INIT_CUSTOM_SCRIPT_KEY]: {
      id: 'use-cloud-init-custom-script',
      title: 'Use custom script',
      type: _Form.CHECKBOX,
      isVisible: basicVmSettings => (0, _selectors2.settingsValue)(basicVmSettings, _constants2.CLOUD_INIT_KEY, false)
    },
    [_constants2.HOST_NAME_KEY]: {
      id: 'cloud-init-hostname',
      title: 'Hostname',
      isVisible: basicVmSettings => (0, _selectors2.settingsValue)(basicVmSettings, _constants2.CLOUD_INIT_KEY, false) && !(0, _selectors2.settingsValue)(basicVmSettings, _constants2.USE_CLOUD_INIT_CUSTOM_SCRIPT_KEY, false)
    },
    [_constants2.AUTHKEYS_KEY]: {
      id: 'cloud-init-ssh',
      title: 'Authenticated SSH Keys',
      type: _Form.TEXT_AREA,
      isVisible: basicVmSettings => (0, _selectors2.settingsValue)(basicVmSettings, _constants2.CLOUD_INIT_KEY, false) && !(0, _selectors2.settingsValue)(basicVmSettings, _constants2.USE_CLOUD_INIT_CUSTOM_SCRIPT_KEY, false)
    },
    [_constants2.CLOUD_INIT_CUSTOM_SCRIPT_KEY]: {
      id: 'cloud-init-custom-script',
      title: 'Custom Script',
      type: _Form.TEXT_AREA,
      className: 'kubevirt-create-vm-wizard__custom-cloud-script-textarea',
      isVisible: basicVmSettings => (0, _selectors2.settingsValue)(basicVmSettings, _constants2.CLOUD_INIT_KEY, false) && (0, _selectors2.settingsValue)(basicVmSettings, _constants2.USE_CLOUD_INIT_CUSTOM_SCRIPT_KEY, false)
    }
  });
};

exports.getFormFields = getFormFields;

const asValueObject = (value, validation) => ({
  value,
  validation
}); // TODO: To avoid race conditions as the basicSettings tab is bound to the CreateVmWizard state at the render-time, it would
//  be better to retrieve basicSettings at the time of its actual use - to align behavior with the setState().
// The onChange() bellow should be called with just the diff, not whole copy of the stepData.


const publish = ({
  basicSettings,
  templates,
  onChange,
  dataVolumes
}, value, target, formValid, formFields) => {
  let newBasicSettings;

  if (target === _constants2.BATCH_CHANGES_KEY) {
    // the "value" is an array of pairs { value, target }
    const difference = value.value.reduce((map, obj) => {
      map[obj.target] = {
        value: obj.value
      }; // validation will be set in a next step

      return map;
    }, {});
    newBasicSettings = _objectSpread({}, basicSettings, difference);
    value.value.forEach(pair => {
      // set field validation
      newBasicSettings[pair.target].validation = (0, _Form.getFieldValidation)(formFields[pair.target], pair.value, newBasicSettings);
    });
    formValid = (0, _Form.validateForm)(formFields, newBasicSettings);
  } else {
    newBasicSettings = _objectSpread({}, basicSettings, {
      [target]: value
    });

    if (target === _constants2.USER_TEMPLATE_KEY) {
      if (value.value === _strings.NO_TEMPLATE) {
        newBasicSettings[target] = asValueObject(undefined);
      } else {
        const allTemplates = (0, _templates.getTemplate)(templates, _constants.TEMPLATE_TYPE_VM);

        if (allTemplates.length > 0) {
          const userTemplate = allTemplates.find(template => template.metadata.name === value.value);
          updateTemplateData(userTemplate, newBasicSettings, dataVolumes);
        }
      }

      formValid = (0, _Form.validateForm)(formFields, newBasicSettings);
    }
  }

  onChange(newBasicSettings, formValid);
};

const updateTemplateData = (userTemplate, newBasicSettings, dataVolumes) => {
  if (userTemplate) {
    const vm = (0, _selectors2.selectVm)(userTemplate.objects); // update flavor

    const _getTemplateFlavors = (0, _selectors2.getTemplateFlavors)([userTemplate]),
          _getTemplateFlavors2 = _slicedToArray(_getTemplateFlavors, 1),
          flavor = _getTemplateFlavors2[0];

    newBasicSettings[_constants2.FLAVOR_KEY] = asValueObject(flavor);

    if (flavor === _constants.CUSTOM_FLAVOR) {
      newBasicSettings.cpu = asValueObject((0, _selectors.getCpu)(vm));
      const memory = (0, _selectors.getMemory)(vm);
      newBasicSettings.memory = memory ? asValueObject(parseInt(memory, 10)) : undefined;
    } // update os


    const _getTemplateOperating = (0, _selectors2.getTemplateOperatingSystems)([userTemplate]),
          _getTemplateOperating2 = _slicedToArray(_getTemplateOperating, 1),
          os = _getTemplateOperating2[0];

    newBasicSettings[_constants2.OPERATING_SYSTEM_KEY] = asValueObject(os); // update workload profile

    const _getTemplateWorkloadP = (0, _selectors2.getTemplateWorkloadProfiles)([userTemplate]),
          _getTemplateWorkloadP2 = _slicedToArray(_getTemplateWorkloadP, 1),
          workload = _getTemplateWorkloadP2[0];

    newBasicSettings[_constants2.WORKLOAD_PROFILE_KEY] = asValueObject(workload); // update cloud-init

    const cloudInitUserData = (0, _selectors.getCloudInitUserData)(vm);

    if (cloudInitUserData) {
      newBasicSettings[_constants2.CLOUD_INIT_KEY] = asValueObject(true);
      newBasicSettings[_constants2.USE_CLOUD_INIT_CUSTOM_SCRIPT_KEY] = asValueObject(true);
      newBasicSettings[_constants2.CLOUD_INIT_CUSTOM_SCRIPT_KEY] = asValueObject(cloudInitUserData || '');
    } else if ((0, _lodash.get)(newBasicSettings[_constants2.CLOUD_INIT_KEY], 'value')) {
      newBasicSettings[_constants2.CLOUD_INIT_KEY] = asValueObject(false);
      newBasicSettings[_constants2.USE_CLOUD_INIT_CUSTOM_SCRIPT_KEY] = asValueObject(false);
    } // update provision source


    const provisionSource = (0, _templates.getTemplateProvisionSource)(userTemplate, dataVolumes);

    if (provisionSource) {
      newBasicSettings[_constants2.PROVISION_SOURCE_TYPE_KEY] = asValueObject(provisionSource.type);

      switch (provisionSource.type) {
        case _constants.PROVISION_SOURCE_CONTAINER:
          newBasicSettings[_constants2.CONTAINER_IMAGE_KEY] = asValueObject(provisionSource.source);
          break;

        case _constants.PROVISION_SOURCE_URL:
          newBasicSettings[_constants2.IMAGE_URL_KEY] = asValueObject(provisionSource.source);
          break;

        case _constants.PROVISION_SOURCE_PXE:
          break;

        default:
          // eslint-disable-next-line
          console.warn(`Unknown provision source ${provisionSource.type}`);
          break;
      }
    } else {
      // eslint-disable-next-line
      console.warn(`Cannot detect provision source for template ${(0, _selectors.getName)(userTemplate)}`);
    }
  }

  return newBasicSettings;
}; // Do clean-up


const onCloseBasic = async (basicSettings, callerContext) => {
  const v2vvmwareName = (0, _selectors2.getV2VVmwareName)(basicSettings);

  if (v2vvmwareName) {
    // This is a friendly help to keep things clean.
    // If missed here (e.g. when the browser window is closed), the kubevirt-vmware controller's garbage
    // collector will do following automatically after a delay.
    const resource = {
      metadata: {
        name: v2vvmwareName,
        // TODO: potential issue if the user changed the namespace after creation of the v2vvmware object
        // to fix: either store the namespace along the v2vvmwareName or empty v2vvmwareName on namespace change
        namespace: (0, _selectors2.settingsValue)(basicSettings, _constants2.NAMESPACE_KEY)
      }
    };

    try {
      await callerContext.k8sKill(_models.V2VVMwareModel, resource);
    } catch (error) {
      // eslint-disable-next-line no-console
      console.log('Failed to remove temporary V2VVMWare object. It is not an issue, it will be garbage collected later if still present, resource: ', resource, ', error: ', error);
    }
  }
};

exports.onCloseBasic = onCloseBasic;

class BasicSettingsTab extends _react.default.Component {
  constructor(_props) {
    super(_props);

    _defineProperty(this, "updateSelectedNamespace", props => {
      const basicSettings = props.basicSettings,
            namespaces = props.namespaces,
            selectedNamespace = props.selectedNamespace,
            templates = props.templates,
            createTemplate = props.createTemplate,
            WithResources = props.WithResources,
            k8sCreate = props.k8sCreate,
            k8sGet = props.k8sGet,
            k8sPatch = props.k8sPatch;
      const formFields = getFormFields(basicSettings, namespaces, templates, selectedNamespace, createTemplate, WithResources, k8sCreate, k8sGet, k8sPatch);
      const valid = (0, _Form.validateForm)(formFields, basicSettings);
      publish(props, asValueObject((0, _selectors.getName)(selectedNamespace)), _constants2.NAMESPACE_KEY, valid, formFields);
    });

    if (_props.selectedNamespace) {
      this.updateSelectedNamespace(_props);
    }
  }

  componentDidUpdate(prevProps) {
    const newNamespace = this.props.selectedNamespace;
    const oldNamespace = prevProps.selectedNamespace;

    if ((0, _selectors.getName)(newNamespace) !== (0, _selectors.getName)(oldNamespace)) {
      this.updateSelectedNamespace(this.props);
    }
  }

  render() {
    const _this$props = this.props,
          basicSettings = _this$props.basicSettings,
          namespaces = _this$props.namespaces,
          templates = _this$props.templates,
          selectedNamespace = _this$props.selectedNamespace,
          createTemplate = _this$props.createTemplate,
          WithResources = _this$props.WithResources,
          k8sCreate = _this$props.k8sCreate,
          k8sGet = _this$props.k8sGet,
          k8sPatch = _this$props.k8sPatch;
    const formFields = getFormFields(basicSettings, namespaces, templates, selectedNamespace, createTemplate, WithResources, k8sCreate, k8sGet, k8sPatch);
    return _react.default.createElement(_Form.FormFactory, {
      fields: formFields,
      fieldsValues: basicSettings,
      onFormChange: (newValue, target, formValid) => publish(this.props, newValue, target, formValid, formFields)
    });
  }

}

exports.BasicSettingsTab = BasicSettingsTab;
BasicSettingsTab.defaultProps = {
  selectedNamespace: undefined,
  createTemplate: false
};
BasicSettingsTab.propTypes = {
  WithResources: _propTypes.default.func.isRequired,
  k8sCreate: _propTypes.default.func.isRequired,
  k8sGet: _propTypes.default.func.isRequired,
  k8sPatch: _propTypes.default.func.isRequired,
  templates: _propTypes.default.array.isRequired,
  namespaces: _propTypes.default.array.isRequired,
  selectedNamespace: _propTypes.default.object,
  // used only in initialization
  basicSettings: _propTypes.default.object.isRequired,
  // eslint-disable-next-line react/no-unused-prop-types
  onChange: _propTypes.default.func.isRequired,
  createTemplate: _propTypes.default.bool,
  // eslint-disable-next-line react/no-unused-prop-types
  dataVolumes: _propTypes.default.array.isRequired
};