"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.StorageTab = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _lodash = require("lodash");

var _constants = require("../../../constants");

var _TableFactory = require("../../Table/TableFactory");

var _Form = require("../../Form");

var _validations = require("../../../utils/validations");

var _constants2 = require("../../Table/constants");

var _constants3 = require("./constants");

var _selectors = require("../../../utils/selectors");

var _strings = require("./strings");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function _slicedToArray(arr, i) { return _arrayWithHoles(arr) || _iterableToArrayLimit(arr, i) || _nonIterableRest(); }

function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance"); }

function _iterableToArrayLimit(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"] != null) _i["return"](); } finally { if (_d) throw _e; } } return _arr; }

function _arrayWithHoles(arr) { if (Array.isArray(arr)) return arr; }

const validateDataVolumeStorage = storage => {
  const errors = Array(4).fill(null);

  if (!storage || storage.id == null) {
    errors[0] = _strings.ERROR_EMPTY_ENTITY; // row error on index 0
  }

  const nameValidation = (0, _validations.validateDNS1123SubdomainValue)(storage.name);

  if ((0, _lodash.get)(nameValidation, 'type') === _constants.VALIDATION_ERROR_TYPE) {
    errors[1] = `Name ${nameValidation.message}`;
  }

  if (!storage.size || storage.size <= 0) {
    errors[2] = _strings.ERROR_POSITIVE_SIZE;
  }

  return errors;
};

const validatePvcStorage = storage => {
  const errors = Array(4).fill(null);

  if (!storage || storage.id == null) {
    errors[0] = _strings.ERROR_EMPTY_ENTITY; // row error on index 0
  }

  const nameValidation = (0, _validations.validateDNS1123SubdomainValue)(storage.name);

  if ((0, _lodash.get)(nameValidation, 'type') === _constants.VALIDATION_ERROR_TYPE) {
    errors[1] = `Name ${nameValidation.message}`;
  }

  return errors;
};

const validateContainerStorage = storage => {
  const errors = Array(4).fill(null);

  if (!storage || storage.id == null) {
    errors[0] = _strings.ERROR_EMPTY_ENTITY; // row error on index 0
  }

  const nameValidation = (0, _validations.validateDNS1123SubdomainValue)(storage.name);

  if ((0, _lodash.get)(nameValidation, 'type') === _constants.VALIDATION_ERROR_TYPE) {
    errors[1] = `Name ${nameValidation.message}`;
  }

  return errors;
};

const validateDiskNamespace = (storages, pvcs, namespace) => {
  const availablePvcs = pvcs.filter(pvc => pvc.metadata.namespace === namespace);
  storages.filter(storage => storage.storageType === _constants3.STORAGE_TYPE_PVC).forEach(storage => {
    if (!storage.errors) {
      storage.errors = new Array(4).fill(null);
    }

    storage.errors[1] = availablePvcs.some(pvc => pvc.metadata.name === storage.name) ? null : _strings.ERROR_DISK_NOT_FOUND;
  });
};

const setBootableDisk = (disks, bootDisk) => {
  disks.forEach(disk => {
    disk.isBootable = false;
  });

  if (bootDisk) {
    bootDisk.isBootable = true;
  }
};

const findBootDisk = bootableDisks => {
  if (!bootableDisks || bootableDisks.length === 0) {
    return undefined;
  }

  let bootDisk; // lets check template storage boot order

  bootableDisks.filter(disk => disk.templateStorage).forEach(disk => {
    const bootOrder = (0, _lodash.get)(disk.templateStorage, 'disk.bootOrder');

    if (bootOrder && (!bootDisk || bootOrder < bootDisk.templateStorage.disk.bootOrder)) {
      bootDisk = disk;
    }
  }); // if we still did not find any boot disk, lets mark the first one

  if (!bootDisk) {
    var _bootableDisks = _slicedToArray(bootableDisks, 1);

    bootDisk = _bootableDisks[0];
  }

  return bootDisk;
};

const hasError = disk => disk.errors ? disk.errors.some(error => !!error) : false;

const resolveBootability = (rows, sourceType) => {
  const rootStorage = rows.find(row => row.rootStorage);

  if (rootStorage) {
    setBootableDisk(rows, rootStorage);
  } else if (!rows.some(row => row.isBootable && !hasError(row))) {
    let bootableDisks;

    switch (sourceType) {
      case _constants.PROVISION_SOURCE_CONTAINER:
        bootableDisks = rows.filter(row => row.storageType === _constants3.STORAGE_TYPE_CONTAINER);
        break;

      case _constants.PROVISION_SOURCE_URL:
        bootableDisks = rows.filter(row => row.storageType === _constants3.STORAGE_TYPE_DATAVOLUME);
        break;

      case _constants.PROVISION_SOURCE_PXE:
        bootableDisks = rows.filter(row => row.storageType === _constants3.STORAGE_TYPE_PVC || row.storageType === _constants3.STORAGE_TYPE_DATAVOLUME);
        break;

      default:
        break;
    }

    const bootDisk = findBootDisk(bootableDisks);
    setBootableDisk(rows, bootDisk);
  }

  return rows;
};

const resolvePvcStorage = (storage, persistentVolumeClaims, storageClasses, units) => {
  const pvcStorage = persistentVolumeClaims.find(pvc => (0, _selectors.getName)(pvc) === storage.name);
  const pvcStorageClassName = (0, _selectors.getPvcStorageClassName)(pvcStorage);
  return _objectSpread({}, storage, {
    name: (0, _selectors.getName)(pvcStorage),
    size: (0, _selectors.getGibStorageSize)(units, (0, _selectors.getPvcResources)(pvcStorage)),
    storageClass: (0, _selectors.getName)(storageClasses.find(clazz => (0, _selectors.getName)(clazz) === pvcStorageClassName)),
    storageType: _constants3.STORAGE_TYPE_PVC
  });
};

const resolveTemplateStorage = (storage, persistentVolumeClaims, storageClasses, units, sourceType) => {
  const templateStorage = _objectSpread({}, storage);

  if (!templateStorage.name) {
    templateStorage.name = storage.templateStorage.disk.name;
  }

  if (storage.templateStorage.volume.persistentVolumeClaim) {
    // TODO if PVC not found add error
    // TODO also check other possible incorrect things in template storage definition
    const pvc = persistentVolumeClaims.find(p => (0, _selectors.getName)(p) === storage.templateStorage.volume.persistentVolumeClaim.claimName);

    if (!templateStorage.size) {
      templateStorage.size = (0, _selectors.getGibStorageSize)(units, (0, _selectors.getPvcResources)(pvc));
    }

    if (!templateStorage.storageClass) {
      templateStorage.storageClass = (0, _selectors.getName)(storageClasses.find(clazz => (0, _selectors.getName)(clazz) === pvc.spec.storageClassName));
    }

    templateStorage.storageType = _constants3.STORAGE_TYPE_PVC;
  } else if (storage.templateStorage.volume.dataVolume) {
    const dataVolume = storage.templateStorage.dataVolume || storage.templateStorage.dataVolumeTemplate;

    if (!templateStorage.size) {
      templateStorage.size = (0, _selectors.getGibStorageSize)(units, (0, _selectors.getDataVolumeResources)(dataVolume));
    }

    if (!templateStorage.storageClass) {
      templateStorage.storageClass = (0, _selectors.getDataVolumeStorageClassName)(dataVolume);
    }

    templateStorage.storageType = _constants3.STORAGE_TYPE_DATAVOLUME;
  } else if (storage.templateStorage.volume.containerDisk) {
    templateStorage.storageType = _constants3.STORAGE_TYPE_CONTAINER;
  }

  return templateStorage;
};

const resolveInitialStorages = (initialStorages, persistentVolumeClaims, storageClasses, units, sourceType, namespace) => {
  let nextId = Math.max(...initialStorages.map(storage => storage.id || 0), 0) + 1;
  const storages = initialStorages.map(storage => {
    let result = {
      id: nextId++,
      editable: true
    };

    if (storage.templateStorage) {
      result = _objectSpread({}, result, resolveTemplateStorage(storage, persistentVolumeClaims, storageClasses, units, sourceType));
    } else {
      switch (storage.storageType) {
        case _constants3.STORAGE_TYPE_CONTAINER:
          result = _objectSpread({}, result, storage);
          break;

        case _constants3.STORAGE_TYPE_DATAVOLUME:
          result = _objectSpread({}, result, storage);
          break;

        case _constants3.STORAGE_TYPE_PVC:
        default:
          result = _objectSpread({}, result, resolvePvcStorage(storage, persistentVolumeClaims, storageClasses, units));
      }
    }

    return result;
  });
  validateDiskNamespace(storages, persistentVolumeClaims, namespace);
  resolveBootability(storages, sourceType);
  return storages;
};

const validateStorage = row => {
  switch (row.storageType) {
    case _constants3.STORAGE_TYPE_PVC:
      return validatePvcStorage(row);

    case _constants3.STORAGE_TYPE_DATAVOLUME:
      return validateDataVolumeStorage(row);

    case _constants3.STORAGE_TYPE_CONTAINER:
      return validateContainerStorage(row);

    default:
      return Array(4).fill(null);
  }
};

const publishResults = (rows, otherStorages, publish) => {
  // TODO bootable device is required for URL, Container
  let valid = true;
  const storages = rows.map(({
    templateStorage,
    rootStorage,
    storageType,
    id,
    name,
    size,
    storageClass,
    isBootable,
    errors
  }) => {
    const result = {
      rootStorage,
      storageType,
      id,
      name,
      size,
      storageClass,
      isBootable,
      errors
    };

    if (templateStorage) {
      result.templateStorage = templateStorage;
    }

    if (valid && errors) {
      for (const error of errors) {
        if (error) {
          valid = false;
          break;
        }
      }
    }

    return result;
  });
  storages.push(...otherStorages);
  publish(storages, valid);
};

class StorageTab extends _react.default.Component {
  constructor(props) {
    super(props);

    _defineProperty(this, "onRowActivate", rows => {
      this.setState({
        rows,
        editing: true
      });
    });

    _defineProperty(this, "onRowUpdate", (rows, updatedRowId, editing) => {
      const index = (0, _lodash.findIndex)(rows, row => row.id === updatedRowId);

      if (rows[index].storageType === _constants3.STORAGE_TYPE_PVC) {
        rows[index] = resolvePvcStorage(rows[index], this.props.persistentVolumeClaims, this.props.storageClasses, this.props.units);
      }

      const updatedRow = rows[index];
      updatedRow.errors = validateStorage(updatedRow);
      this.rowsChanged(rows, editing);
    });

    _defineProperty(this, "rowsChanged", (rows, editing) => {
      resolveBootability(rows, this.props.sourceType);
      publishResults(rows, this.state.otherStorages, this.props.onChange);
      this.setState({
        rows,
        editing
      });
    });

    _defineProperty(this, "create", storageType => {
      this.setState(state => ({
        nextId: state.nextId + 1,
        rows: [...state.rows, {
          id: state.nextId,
          isBootable: false,
          editable: true,
          edit: true,
          // trigger immediate edit
          storageType
        }]
      }));
    });

    _defineProperty(this, "getColumns", () => [{
      header: {
        label: _strings.HEADER_NAME,
        props: {
          style: {
            width: '50%'
          }
        }
      },
      property: 'name',
      renderConfig: storage => storage.storageType === _constants3.STORAGE_TYPE_PVC ? {
        id: 'name-attach-edit',
        type: _Form.DROPDOWN,
        choices: this.props.persistentVolumeClaims.filter(pvc => pvc.metadata.namespace === this.props.namespace).map(_selectors.getName).filter(pvc => !this.state.rows.some(row => row.name === pvc)),
        initialValue: '--- Select Storage ---'
      } : {
        id: 'name-edit',
        type: _Form.TEXT
      }
    }, {
      header: {
        label: _strings.HEADER_SIZE,
        props: {
          style: {
            width: '23%'
          }
        }
      },
      property: 'size',
      renderConfig: storage => storage.storageType === _constants3.STORAGE_TYPE_DATAVOLUME ? {
        id: 'size-edit',
        type: _Form.POSITIVE_NUMBER
      } : null
    }, {
      header: {
        label: _strings.HEADER_STORAGE_CLASS,
        props: {
          style: {
            width: '23%'
          }
        }
      },
      property: 'storageClass',
      renderConfig: storage => storage.storageType === _constants3.STORAGE_TYPE_DATAVOLUME ? {
        id: 'storage-edit',
        type: _Form.DROPDOWN,
        choices: this.props.storageClasses.map(_selectors.getName),
        initialValue: '--- Select ---'
      } : null
    }, {
      header: {
        props: {
          style: {
            width: '4%'
          }
        }
      },
      type: _constants2.ACTIONS_TYPE,
      renderConfig: storage => storage.rootStorage ? null : {
        id: 'actions',
        actions: [{
          actionType: _constants2.DELETE_ACTION,
          text: _strings.REMOVE_DISK_BUTTON
        }],
        visibleOnEdit: false
      }
    }]);

    _defineProperty(this, "getActionButtons", () => [{
      className: 'kubevirt-create-vm-wizard__button-create-disk',
      onClick: () => this.create(_constants3.STORAGE_TYPE_DATAVOLUME),
      id: 'create-storage-btn',
      text: _strings.CREATE_DISK_BUTTON,
      disabled: this.state.editing
    }, {
      className: 'kubevirt-create-vm-wizard__button-attach-disk',
      onClick: () => this.create(_constants3.STORAGE_TYPE_PVC),
      id: 'attach-disk-btn',
      text: _strings.ATTACH_DISK_BUTTON,
      disabled: this.state.editing
    }]);

    _defineProperty(this, "getFormFields", (disks, sourceType) => ({
      bootableDisk: {
        id: 'bootable-disk-dropdown',
        title: _strings.BOOTABLE_DISK,
        type: _Form.DROPDOWN,
        defaultValue: _strings.SELECT_BOOTABLE_DISK,
        choices: disks.filter(disk => !hasError(disk)).map(disk => ({
          name: disk.name,
          id: disk.id
        })),
        disabled: sourceType === _constants.PROVISION_SOURCE_CONTAINER || sourceType === _constants.PROVISION_SOURCE_URL,
        required: sourceType === _constants.PROVISION_SOURCE_CONTAINER || sourceType === _constants.PROVISION_SOURCE_URL
      }
    }));

    _defineProperty(this, "onFormChange", newValue => {
      this.setState(state => {
        state.rows.forEach(row => {
          row.isBootable = row.id === newValue.value.id;
        });
        publishResults(state.rows, state.otherStorages, this.props.onChange);
        return state.rows;
      });
    });

    const _rows = resolveInitialStorages(props.initialStorages, props.persistentVolumeClaims, props.storageClasses, props.units, props.sourceType, props.namespace);

    this.state = {
      // eslint-disable-next-line react/no-unused-state
      nextId: Math.max(..._rows.map(disk => disk.id || 0), 0) + 1,
      rows: _rows.filter(row => row.storageType),
      otherStorages: _rows.filter(row => !row.storageType),
      editing: false
    };
    publishResults(this.state.rows, this.state.otherStorages, this.props.onChange);
  }

  render() {
    const columns = this.getColumns();
    const actionButtons = this.getActionButtons();
    const bootableDisk = this.state.rows.find(row => row.isBootable);
    const values = {
      bootableDisk: {
        value: bootableDisk ? bootableDisk.name : undefined,
        validation: this.props.sourceType !== _constants.PROVISION_SOURCE_PXE && this.state.rows.length === 0 ? (0, _validations.getValidationObject)(_strings.ERROR_NO_BOOTABLE_DISK) : undefined
      }
    };

    const bootableForm = _react.default.createElement(_Form.FormFactory, {
      fields: this.getFormFields(this.state.rows, this.props.sourceType),
      fieldsValues: values,
      onFormChange: this.onFormChange,
      textPosition: "text-left",
      labelSize: 2,
      controlSize: 10,
      formClassName: "kubevirt-create-vm-wizard__pxe-form"
    });

    return _react.default.createElement(_react.default.Fragment, null, _react.default.createElement(_TableFactory.TableFactory, {
      actionButtons: actionButtons,
      columns: columns,
      rows: this.state.rows,
      onRowUpdate: this.onRowUpdate,
      onRowDeleteOrMove: this.rowsChanged,
      onRowActivate: this.onRowActivate
    }), bootableForm);
  }

}

exports.StorageTab = StorageTab;
StorageTab.defaultProps = {
  initialStorages: []
};
StorageTab.propTypes = {
  storageClasses: _propTypes.default.array.isRequired,
  persistentVolumeClaims: _propTypes.default.array.isRequired,
  initialStorages: _propTypes.default.array,
  // StorageTab keeps it's own state
  onChange: _propTypes.default.func.isRequired,
  units: _propTypes.default.object.isRequired,
  sourceType: _propTypes.default.string.isRequired,
  namespace: _propTypes.default.string.isRequired
};