"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addTemplateClone = exports.clone = void 0;

var _lodash = require("lodash");

var _models = require("../models");

var _selectors = require("../utils/selectors");

var _utils = require("../utils");

var _vmBuilder = require("./vmBuilder");

var _constants = require("../constants");

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const clone = async (k8sCreate, k8sPatch, vm, newName, newNamespace, newDescription, startVm, persistentVolumeClaims, dataVolumes) => {
  const stopVmPatch = (0, _utils.getStartStopPatch)(vm, false);
  await k8sPatch(_models.VirtualMachineModel, vm, stopVmPatch);
  const osName = (0, _selectors.getOperatingSystemName)(vm);
  cleanVm(vm);
  cleanNetworks(vm);
  addPvcClones(vm, newName, persistentVolumeClaims);
  addDataVolumeClones(vm, newName, dataVolumes);
  updateVm(vm, newName, newNamespace, newDescription, startVm, osName);
  return k8sCreate(_models.VirtualMachineModel, vm);
};

exports.clone = clone;

const cleanNetworks = vm => {
  (0, _selectors.getInterfaces)(vm).forEach(intface => delete intface.macAddress);
};

const cleanVm = vm => {
  delete vm.metadata.selfLink;
  delete vm.metadata.resourceVersion;
  delete vm.metadata.uid;
  delete vm.metadata.creationTimestamp;
  delete vm.metadata.generation; // delete all datavolumetemplates

  if ((0, _lodash.has)(vm.spec, 'dataVolumeTemplates')) {
    delete vm.spec.dataVolumeTemplates;
  }

  if ((0, _lodash.has)(vm.spec, 'template.spec.domain.firmware')) {
    delete vm.spec.template.spec.domain.firmware;
  }

  delete vm.status;
};

const updateVm = (vm, name, namespace, description, startVm, osName) => {
  vm.metadata.name = name;
  vm.metadata.namespace = namespace;
  vm.metadata.annotations = {
    description
  };

  if (osName) {
    vm.metadata.annotations[`${_constants.TEMPLATE_OS_NAME_ANNOTATION}/${(0, _selectors.getOperatingSystem)(vm)}`] = osName;
  }

  (0, _vmBuilder.addTemplateLabel)(vm, _constants.TEMPLATE_VM_NAME_LABEL, name); // for pairing service-vm (like for RDP)

  vm.spec = _objectSpread({}, vm.spec || {}, {
    running: startVm
  });
};

const addDataVolumeClones = (vm, newName, dataVolumes) => {
  const volumes = (0, _selectors.getVolumes)(vm);
  volumes.filter(volume => volume.dataVolume).forEach(volume => {
    const dvName = volume.dataVolume.name;
    const dataVolume = dataVolumes.find(dv => (0, _selectors.getName)(dv) === dvName && (0, _selectors.getNamespace)(dv) === (0, _selectors.getNamespace)(vm));

    if (dataVolume) {
      const template = addTemplateClone(vm, dvName, (0, _selectors.getNamespace)(dataVolume), (0, _selectors.getDataVolumeAccessModes)(dataVolume), (0, _selectors.getDataVolumeStorageSize)(dataVolume), (0, _selectors.getDataVolumeStorageClassName)(dataVolume), newName);
      volume.dataVolume = {
        name: (0, _selectors.getName)(template)
      };
    }
  });
};

const addPvcClones = (vm, newName, persistentVolumeClaims) => {
  const volumes = (0, _selectors.getVolumes)(vm);
  volumes.filter(volume => volume.persistentVolumeClaim).forEach(volume => {
    const pvcName = volume.persistentVolumeClaim.claimName;
    const pvc = persistentVolumeClaims.find(p => (0, _selectors.getName)(p) === pvcName && (0, _selectors.getNamespace)(p) === (0, _selectors.getNamespace)(vm));
    const template = addTemplateClone(vm, pvcName, (0, _selectors.getNamespace)(pvc), (0, _selectors.getPvcAccessModes)(pvc), (0, _selectors.getPvcStorageSize)(pvc), (0, _selectors.getPvcStorageClassName)(pvc), newName);
    delete volume.persistentVolumeClaim;
    volume.dataVolume = {
      name: (0, _selectors.getName)(template)
    };
  });
};

const addTemplateClone = (vm, pvcName, pvcNamespace, accessModes, size, storageClassName, vmName) => {
  const template = getPvcCloneTemplate(pvcName, pvcNamespace, accessModes, size, storageClassName, vmName);
  return (0, _vmBuilder.addDataVolumeTemplate)(vm, template);
};

exports.addTemplateClone = addTemplateClone;

const getPvcCloneTemplate = (pvcName, pvcNamespace, accessModes, size, storageClassName, vmName) => ({
  metadata: {
    name: (0, _utils.generateDiskName)(vmName, pvcName, true)
  },
  spec: {
    source: {
      pvc: {
        name: pvcName,
        namespace: pvcNamespace
      }
    },
    pvc: {
      accessModes,
      resources: {
        requests: {
          storage: size
        }
      },
      storageClassName
    }
  }
});