"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.retrieveVmTemplate = exports.getTemplateProvisionSource = exports.hasAutoAttachPodInterface = exports.getTemplateInterfaces = exports.getTemplateStorages = exports.getUserTemplate = exports.getTemplate = exports.getTemplatesLabelValues = exports.getTemplatesWithLabels = void 0;

var _lodash = require("lodash");

var _selectors = require("./selectors");

var _selectors2 = require("../k8s/selectors");

var _template = require("../k8s/objects/template");

var _constants = require("../constants");

var _models = require("../models");

var _constants2 = require("../components/Wizard/CreateVmWizard/constants");

const getTemplatesWithLabels = (templates, labels) => {
  const filteredTemplates = [...templates];
  labels.forEach(label => {
    if (label !== undefined) {
      (0, _lodash.pull)(filteredTemplates, (0, _lodash.remove)(filteredTemplates, template => Object.keys(template.metadata.labels).find(l => l === label) === undefined));
    }
  });
  return filteredTemplates;
};

exports.getTemplatesWithLabels = getTemplatesWithLabels;

const getTemplatesLabelValues = (templates, label) => {
  const labelValues = [];
  templates.forEach(t => {
    const labels = Object.keys(t.metadata.labels || []).filter(l => l.startsWith(label));
    labels.forEach(l => {
      const labelParts = l.split('/');

      if (labelParts.length > 1) {
        const labelName = labelParts[labelParts.length - 1];

        if (labelValues.indexOf(labelName) === -1) {
          labelValues.push(labelName);
        }
      }
    });
  });
  return labelValues;
};

exports.getTemplatesLabelValues = getTemplatesLabelValues;

const getTemplate = (templates, type) => {
  const filteredTemplates = templates.filter(template => {
    const labels = (0, _lodash.get)(template, 'metadata.labels', {});
    return labels[_constants.TEMPLATE_TYPE_LABEL] === type;
  });
  return type === _constants.TEMPLATE_TYPE_BASE && filteredTemplates.length === 0 ? _template.baseTemplates : filteredTemplates;
};

exports.getTemplate = getTemplate;

const getUserTemplate = (templates, userTemplateName) => {
  const userTemplates = getTemplate(templates, _constants.TEMPLATE_TYPE_VM);
  return userTemplates.find(template => template.metadata.name === userTemplateName);
};

exports.getUserTemplate = getUserTemplate;

const getTemplateStorages = (template, dataVolumes) => {
  const vm = (0, _selectors2.selectVm)(template.objects);
  const volumes = (0, _selectors.getVolumes)(vm);
  const dataVolumeTemplates = (0, _selectors.getDataVolumeTemplates)(vm);
  return (0, _selectors.getDisks)(vm).map(disk => {
    const volume = volumes.find(v => v.name === disk.name);
    const storage = {
      disk,
      volume
    };

    if ((0, _lodash.get)(volume, 'dataVolume')) {
      storage.dataVolumeTemplate = dataVolumeTemplates.find(d => (0, _selectors.getName)(d) === (0, _lodash.get)(volume.dataVolume, 'name'));
      storage.dataVolume = dataVolumes.find(d => (0, _selectors.getName)(d) === (0, _lodash.get)(volume.dataVolume, 'name') && (0, _selectors.getNamespace)(d) === (0, _selectors.getNamespace)(template));
    }

    return storage;
  });
};

exports.getTemplateStorages = getTemplateStorages;

const getTemplateInterfaces = ({
  objects
}) => {
  const vm = (0, _selectors2.selectVm)(objects);
  return (0, _selectors.getInterfaces)(vm).map(i => {
    const network = (0, _selectors.getNetworks)(vm).find(n => n.name === i.name);
    return {
      network,
      interface: i
    };
  });
};

exports.getTemplateInterfaces = getTemplateInterfaces;

const hasAutoAttachPodInterface = ({
  objects
}) => {
  const vm = (0, _selectors2.selectVm)(objects);
  return (0, _lodash.get)(vm, 'spec.template.spec.domain.devices.autoattachPodInterface', true);
};

exports.hasAutoAttachPodInterface = hasAutoAttachPodInterface;

const getTemplateProvisionSource = (template, dataVolumes) => {
  const vm = (0, _selectors2.selectVm)(template.objects);

  if ((0, _selectors.getInterfaces)(vm).some(i => i.bootOrder === 1)) {
    return {
      type: _constants.PROVISION_SOURCE_PXE
    };
  }

  const bootDisk = (0, _selectors.getDisks)(vm).find(disk => disk.bootOrder === 1);

  if (bootDisk) {
    const bootVolume = (0, _selectors.getVolumes)(vm).find(volume => volume.name === bootDisk.name);

    if (bootVolume && bootVolume.containerDisk) {
      return {
        type: _constants.PROVISION_SOURCE_CONTAINER,
        source: bootVolume.containerDisk.image
      };
    }

    if (bootVolume && bootVolume.dataVolume) {
      let dataVolume = (0, _selectors.getDataVolumeTemplates)(vm).find(dv => dv.metadata.name === bootVolume.dataVolume.name);

      if (!dataVolume) {
        dataVolume = dataVolumes.find(d => (0, _selectors.getName)(d) === bootVolume.dataVolume.name && (0, _selectors.getNamespace)(d) === (0, _selectors.getNamespace)(template));
      }

      if (dataVolume) {
        const source = (0, _selectors.getDataVolumeSourceType)(dataVolume);

        switch (source.type) {
          case _constants2.DATA_VOLUME_SOURCE_URL:
            return {
              type: _constants.PROVISION_SOURCE_URL,
              source: source.url
            };

          default:
            return null;
        }
      }
    }
  }

  return null;
};

exports.getTemplateProvisionSource = getTemplateProvisionSource;

const retrieveVmTemplate = (k8sGet, vm) => new Promise((resolve, reject) => {
  const template = (0, _selectors.getVmTemplate)(vm);

  if (!template) {
    resolve(null);
  }

  const getTemplatePromise = k8sGet(_models.TemplateModel, template.name, template.namespace);
  getTemplatePromise.then(result => resolve(result)).catch(error => {
    let mockedTemplate;

    if ((0, _lodash.get)(error, 'json.code') === 404) {
      // maybe common-templates are not installed, fallback on mocked templates
      mockedTemplate = _template.baseTemplates.find(bTemplate => (0, _selectors.getNamespace)(bTemplate) === template.namespace && (0, _selectors.getName)(bTemplate) === template.name);
    }

    if (mockedTemplate) {
      resolve(mockedTemplate);
    }

    reject(error);
  });
});

exports.retrieveVmTemplate = retrieveVmTemplate;