"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var fs = require("fs");
var jest_config_1 = require("jest-config");
var setFromArgv = require("jest-config/build/set_from_argv");
var path = require("path");
var tsc = require("typescript");
function parseConfig(argv) {
    if (argv.config && typeof argv.config === 'string') {
        if (argv.config[0] === '{' && argv.config[argv.config.length - 1] === '}') {
            return JSON.parse(argv.config);
        }
    }
    return argv.config;
}
function loadJestConfigFromFile(filePath, argv) {
    var config = JSON.parse(fs.readFileSync(filePath, 'utf-8'));
    config.rootDir = config.rootDir
        ? path.resolve(path.dirname(filePath), config.rootDir)
        : process.cwd();
    return jest_config_1.normalize(config, argv);
}
function loadJestConfigFromPackage(filePath, argv) {
    var R_OK = (fs.constants && fs.constants.R_OK) || fs['R_OK'];
    try {
        fs.accessSync(filePath, R_OK);
    }
    catch (e) {
        return null;
    }
    var packageData = require(filePath);
    var config = packageData.jest || {};
    var root = path.dirname(filePath);
    config.rootDir = config.rootDir ? path.resolve(root, config.rootDir) : root;
    return jest_config_1.normalize(config, argv);
}
function readRawConfig(argv, root) {
    var rawConfig = parseConfig(argv);
    if (typeof rawConfig === 'string') {
        return loadJestConfigFromFile(path.resolve(process.cwd(), rawConfig), argv);
    }
    if (typeof rawConfig === 'object') {
        var config = Object.assign({}, rawConfig);
        config.rootDir = config.rootDir || root;
        return jest_config_1.normalize(config, argv);
    }
    var packageConfig = loadJestConfigFromPackage(path.join(root, 'package.json'), argv);
    return packageConfig || jest_config_1.normalize({ rootDir: root }, argv);
}
function getJestConfig(root) {
    var yargs = require('yargs');
    var argv = yargs(process.argv.slice(2)).argv;
    var rawConfig = readRawConfig(argv, root);
    return Object.freeze(setFromArgv.default(rawConfig, argv));
}
exports.getJestConfig = getJestConfig;
function getTSJestConfig(globals) {
    return globals && globals['ts-jest'] ? globals['ts-jest'] : {};
}
exports.getTSJestConfig = getTSJestConfig;
function formatTscParserErrors(errors) {
    return errors.map(function (s) { return JSON.stringify(s, null, 4); }).join('\n');
}
function readCompilerOptions(configPath) {
    var loaded = tsc.readConfigFile(configPath, function (file) {
        var read = tsc.sys.readFile(file);
        if (!read) {
            throw new Error("ENOENT: no such file or directory, open '" + configPath + "'");
        }
        return read;
    });
    if (loaded.error) {
        throw new Error(JSON.stringify(loaded.error, null, 4));
    }
    var basePath = path.dirname(configPath);
    var parsedConfig = tsc.parseJsonConfigFileContent(loaded.config, tsc.sys, basePath);
    if (parsedConfig.errors.length > 0) {
        var formattedErrors = formatTscParserErrors(parsedConfig.errors);
        throw new Error("Some errors occurred while attempting to read from " + configPath + ": " + formattedErrors);
    }
    return parsedConfig.options;
}
function getStartDir() {
    var grandparent = path.resolve(__dirname, '..', '..');
    if (grandparent.endsWith('/node_modules')) {
        return process.cwd();
    }
    return '.';
}
function getPathToClosestTSConfig(startDir, previousDir) {
    if (!startDir) {
        return getPathToClosestTSConfig(getStartDir());
    }
    var tsConfigPath = path.join(startDir, 'tsconfig.json');
    var startDirPath = path.resolve(startDir);
    var previousDirPath = path.resolve(previousDir || '/');
    if (startDirPath === previousDirPath || fs.existsSync(tsConfigPath)) {
        return tsConfigPath;
    }
    return getPathToClosestTSConfig(path.join(startDir, '..'), startDir);
}
function getTSConfigOptionFromConfig(globals) {
    var defaultTSConfigFile = getPathToClosestTSConfig();
    if (!globals) {
        return defaultTSConfigFile;
    }
    var tsJestConfig = getTSJestConfig(globals);
    if (globals.__TS_CONFIG__) {
        console.warn("Using globals > __TS_CONFIG__ option for setting TS config is deprecated.\nPlease set config using this option:\nglobals > ts-jest > tsConfigFile (string).\nMore information at https://github.com/kulshekhar/ts-jest#tsconfig");
        return globals.__TS_CONFIG__;
    }
    else if (tsJestConfig.tsConfigFile) {
        return tsJestConfig.tsConfigFile;
    }
    return defaultTSConfigFile;
}
exports.getTSConfigOptionFromConfig = getTSConfigOptionFromConfig;
function mockGlobalTSConfigSchema(globals) {
    var config = getTSConfigOptionFromConfig(globals);
    return typeof config === 'string'
        ? { 'ts-jest': { tsConfigFile: config } }
        : { __TS_CONFIG__: config };
}
exports.mockGlobalTSConfigSchema = mockGlobalTSConfigSchema;
var tsConfigCache = {};
function getTSConfig(globals, collectCoverage) {
    if (collectCoverage === void 0) { collectCoverage = false; }
    var config = getTSConfigOptionFromConfig(globals);
    var skipBabel = getTSJestConfig(globals).skipBabel;
    var isReferencedExternalFile = typeof config === 'string';
    var tsConfigCacheKey = JSON.stringify([
        skipBabel,
        collectCoverage,
        isReferencedExternalFile ? config : undefined,
    ]);
    if (tsConfigCacheKey in tsConfigCache) {
        return tsConfigCache[tsConfigCacheKey];
    }
    if (isReferencedExternalFile) {
        var configFileName = config;
        var configPath = path.resolve(config);
        config = readCompilerOptions(configPath);
        if (configFileName === path.join(getStartDir(), 'tsconfig.json')) {
            config.module = tsc.ModuleKind.CommonJS;
        }
    }
    delete config.sourceMap;
    config.inlineSourceMap = true;
    config.inlineSources = true;
    delete config.outDir;
    var result;
    if (isReferencedExternalFile) {
        config.jsx = config.jsx || tsc.JsxEmit.React;
        config.module = config.module || tsc.ModuleKind.CommonJS;
        if (config.allowSyntheticDefaultImports && !skipBabel) {
            config.module = tsc.ModuleKind.ES2015;
        }
        result = config;
    }
    else {
        config.jsx = config.jsx || 'react';
        config.module = config.module || 'commonjs';
        if (config.allowSyntheticDefaultImports && !skipBabel) {
            config.module = 'es2015';
        }
        var converted = tsc.convertCompilerOptionsFromJson(config, undefined);
        if (converted.errors && converted.errors.length > 0) {
            var formattedErrors = formatTscParserErrors(converted.errors);
            throw new Error("Some errors occurred while attempting to convert " + JSON.stringify(config) + ": " + formattedErrors);
        }
        result = converted.options;
    }
    tsConfigCache[tsConfigCacheKey] = result;
    return result;
}
exports.getTSConfig = getTSConfig;
//# sourceMappingURL=data:application/json;base64,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