# Ansible Automation Platform Azure Billing Operator

An operator for Kubernetes built with [Operator SDK](https://github.com/operator-framework/operator-sdk) and [Ansible](https://github.com/ansible/ansible).

## Purpose

This operator is meant to provide a billing cronjob on Ansible Automation Platform deployment made on Azure.

## Usage

### Deploy a Kubernetes/OpenShift cluster

This can be achieved either with [minikube](https://github.com/kubernetes/minikube) or [CodeReady Containers](https://github.com/code-ready/crc) (crc).

### Install the operator-sdk

The `operator-sdk` command is needed for deploying the operator.

See the [documentation](https://sdk.operatorframework.io/docs/installation/) for installing it.

### Connect to your cluster

First login to your k8s cluster.

```
$ oc login -u <user> <cluster-api-url>
```

You might also want to create a new project.

```
$ oc new-project azure
```

### Get the operator sources

```
$ git clone https://github.com/ansible/ansible-automation-platform-azure-billing-operator
$ cd ansible-automation-platform-azure-billing-operator
```

### Build the operator/bundle container images

First login to the registry with either `docker` or `podman`.

```
$ podman login quay.io
Username: <user>
Password: ******
Login Succeeded!
```

**Note:** `quay.io` registry is used as an example but any other registries can be used.

Then create two repositories on your namespace for the operator and bundle container images.
Once created, you can start to build and push those images.

**Note:** When using `podman` you also need to set the BUILDER environment variable (default value is docker).

```
$ export BUILDER="podman"
$ export IMG="quay.io/<user>/ansible-automation-platform-azure-billing-operator:dev"
$ export BUNDLE_IMG="quay.io/<user>/ansible-automation-platform-azure-billing-bundle:dev"
$ make container-build container-push
podman build -t quay.io/<user>/ansible-automation-platform-azure-billing-operator:dev .
(...)
podman push quay.io/<user>/ansible-automation-platform-azure-billing-operator:dev
(...)
$ make bundle bundle-build bundle-push
operator-sdk generate kustomize manifests -q
cd config/manager && /home/<user>/ansible-automation-platform-azure-billing-operator/bin/kustomize edit set image controller=quay.io/<user>/ansible-automation-platform-azure-billing-operator:dev
/home/<user>/ansible-automation-platform-azure-billing-operator/bin/kustomize build config/manifests | operator-sdk generate bundle -q --overwrite --version 0.0.1
INFO[0000] Creating bundle.Dockerfile
INFO[0000] Creating bundle/metadata/annotations.yaml
INFO[0000] Bundle metadata generated suceessfully
operator-sdk bundle validate ./bundle
INFO[0000] All validation tests have completed successfully
podman build -f bundle.Dockerfile -t quay.io/<user>/ansible-automation-platform-azure-billing-bundle:dev .
(...)
podman push quay.io/<user>/ansible-automation-platform-azure-billing-bundle:dev
(...)
```

### Install the operator

If you want to deploy the operator via OLM:

```
$ operator-sdk run bundle ${BUNDLE_IMG}
INFO[0009] Successfully created registry pod: nsible-automation-platform-azure-billing-operator-bundle-v0-0-1
INFO[0009] Created CatalogSource: aap-azure-billing-operator-catalog
INFO[0009] OperatorGroup "operator-sdk-og" created
INFO[0009] Created Subscription: aap-azure-billing-operator-v0-0-1-sub
INFO[0013] Approved InstallPlan install-xtkxt for the Subscription: aap-azure-billing-operator-v0-0-1-sub
INFO[0013] Waiting for ClusterServiceVersion "azure/aap-azure-billing-operator.v0.0.1" to reach 'Succeeded' phase
INFO[0013]   Waiting for ClusterServiceVersion "azure/aap-azure-billing-operator.v0.0.1" to appear
INFO[0020]   Found ClusterServiceVersion "azure/aap-azure-billing-operator.v0.0.1" phase: Pending
INFO[0022]   Found ClusterServiceVersion "azure/aap-azure-billing-operator.v0.0.1" phase: InstallReady
INFO[0023]   Found ClusterServiceVersion "azure/aap-azure-billing-operator.v0.0.1" phase: Installing
INFO[0043]   Found ClusterServiceVersion "azure/aap-azure-billing-operator.v0.0.1" phase: Succeeded
INFO[0043] OLM has successfully installed "aap-azure-billing-operator.v0.0.1"
```

Otherwise you can do a direct deployment:

```
$ export NAMESPACE=azure
$ make deploy
cd config/manager && /home/<user>/ansible-automation-platform-azure-billing-operator/bin/kustomize edit set image controller=quay.io/<user>/ansible-automation-platform-azure-billing-operator:dev
/home/<user>/ansible-automation-platform-azure-billing-operator/bin/kustomize build config/default | kubectl apply -f -
namespace/aap-azure-billing-operator created
customresourcedefinition.apiextensions.k8s.io/ansibleautomationplatformazurebillings.platform.ansible.com created
serviceaccount/aap-azure-billing-operator-controller-manager created
role.rbac.authorization.k8s.io/aap-azure-billing-operator-leader-election-role created
clusterrole.rbac.authorization.k8s.io/aap-azure-billing-operator-manager-role created
clusterrole.rbac.authorization.k8s.io/aap-azure-billing-operator-metrics-reader configured
clusterrole.rbac.authorization.k8s.io/aap-azure-billing-operator-proxy-role created
rolebinding.rbac.authorization.k8s.io/aap-azure-billing-operator-leader-election-rolebinding created
clusterrolebinding.rbac.authorization.k8s.io/aap-azure-billing-operator-manager-rolebinding created
clusterrolebinding.rbac.authorization.k8s.io/aap-azure-billing-operator-proxy-rolebinding created
configmap/aap-azure-billing-operator-manager-config created
service/aap-azure-billing-operator-controller-manager-metrics-service created
deployment.apps/aap-azure-billing-operator-controller-manager created
```

### Create an AnsibleAutomationPlatformAzureBilling CR

**Note**: `automationcontroller_db_secret` and `azure_billing_db_secret` are mandatories parameters.

`automationcontroller_db_secret` and `azure_billing_db_secret` reference secret name and must be formatted as follows:

```yaml
---
apiVersion: v1
kind: Secret
metadata:
  name: foo
  namespace: azure
stringData:
  host: <external ip or url resolvable by the cluster>
  port: <external port, this usually defaults to 5432>
  database: <desired database name>
  username: <username to connect as>
  password: <password to connect with>
type: Opaque
```

```yaml
---
apiVersion: platform.ansible.com/v1alpha1
kind: AnsibleAutomationPlatformAzureBilling
metadata:
  name: billing-demo
  namespace: azure
spec:
  automationcontroller_db_secret: foo
  azure_billing_db_secret: bar
```

Finally, use `oc` to create the azure billing instance in your cluster:

```
$ oc apply -f billing-demo.yml
ansibleautomationplatformazurebilling.platform.ansible.com/billing-demo created
```

You should be able to see the cronjob resource by running:

```
$ oc get cronjobs
NAME                SCHEDULE      SUSPEND   ACTIVE   LAST SCHEDULE   AGE
aap-azure-billing   0 */4 * * *   False     0        <none>          58s
```

### Advanced Configuration

There are few variables that are customizable for the UI image application

| Name                      | Description                |
| --------------------------| -------------------------- |
| automationcontroller_db_secret | Automation Controller database secret name |
| azure_billing_db_secret | Azure Billing database secret name |
| image                     | Path of the image to pull  |
| image_version             | Image version to pull      |
| pod_identity_binding_label | Azure Active Directory pod-managed Identity |

```yaml
---
spec:
  ...
  image: quay.io/foo/bar
  image_version: v1.6.42
  pod_identity_binding_label: aap-foo-bar
```

### Remove the AnsibleAutomationPlatformAzureBilling instance

To uninstall the AnsibleAutomationPlatformAzureBilling instance, run the following command.

```
$ oc delete ansibleautomationplatformazurebilling billing-demo
ansibleautomationplatformazurebilling.platform.ansible.com "billing-demo" deleted
```

### Remove the operator

If the operator was installed via OLM:

```
$ operator-sdk cleanup aap-azure-billing-operator
INFO[0000] subscription "aap-azure-billing-operator-v0-0-1-sub" deleted
INFO[0000] customresourcedefinition "ansibleautomationplatformazurebillings.platform.ansible.com" deleted
INFO[0000] clusterserviceversion "aap-azure-billing-operator.v0.0.1" deleted
INFO[0000] catalogsource "aap-azure-billing-operator-catalog" deleted
INFO[0000] operatorgroup "operator-sdk-og" deleted
INFO[0000] Operator "aap-azure-billing-operator" uninstalled
```

Otherwise:

```
$ export NAMESPACE=azure
$ make undeploy
/home/<user>/ansible-automation-platform-azure-billing-operator/bin/kustomize build config/default | kubectl delete -f -
namespace "aap-azure-billing-operator" deleted
customresourcedefinition.apiextensions.k8s.io "ansibleautomationplatformazurebillings.platform.ansible.com" deleted
serviceaccount "aap-azure-billing-operator-controller-manager" deleted
role.rbac.authorization.k8s.io "aap-azure-billing-operator-leader-election-role" deleted
clusterrole.rbac.authorization.k8s.io "aap-azure-billing-operator-manager-role" deleted
clusterrole.rbac.authorization.k8s.io "aap-azure-billing-operator-metrics-reader" deleted
clusterrole.rbac.authorization.k8s.io "aap-azure-billing-operator-proxy-role" deleted
rolebinding.rbac.authorization.k8s.io "aap-azure-billing-operator-leader-election-rolebinding" deleted
clusterrolebinding.rbac.authorization.k8s.io "aap-azure-billing-operator-manager-rolebinding" deleted
clusterrolebinding.rbac.authorization.k8s.io "aap-azure-billing-operator-proxy-rolebinding" deleted
configmap "aap-azure-billing-operator-manager-config" deleted
service "aap-azure-billing-operator-controller-manager-metrics-service" deleted
deployment.apps "aap-azure-billing-operator-controller-manager" deleted
```
