#!/bin/bash

# Simple script to do a release

TARGET_REMOTE=upstream
TARGET_VERSION="$1"
set -eu
set -o pipefail

if test "${TARGET_VERSION}" = ""; then
	echo "USAGE: $0 <version>" >&2
	exit 1
fi

TARGET_TAG="v${TARGET_VERSION}"

if ! echo "${TARGET_VERSION}" | egrep -q '^[0-9]\.[0-9]\.[0-9]$'; then
	echo "Error: Target version '${TARGET_VERSION}' is not well-formed. Must be X.Y.Z" >&2
	exit 1
fi

echo "*** checking for current git branch"
RELEASE_BRANCH=$(git rev-parse --abbrev-ref HEAD || true)
if [[ $RELEASE_BRANCH = release-* ]]; then
	echo "***   branch is $RELEASE_BRANCH"
	IMAGE_TAG=${TARGET_TAG}
else
	echo "Error: Branch $RELEASE_BRANCH is not release branch" >&2
	exit 1
fi

if ! test -f VERSION; then
	echo "Error: You should be in repository root." >&2
	exit 1
fi

echo "${TARGET_VERSION}" > VERSION

echo
echo "*** checking for existence of git tag ${TARGET_TAG}"
if git tag -l "${TARGET_VERSION}" | grep -q "${TARGET_TAG}"; then
	echo "Error: Tag with version ${TARGET_TAG} already exists." >&2
	exit 1
fi

echo
echo "*** generating new manifests"
export IMAGE_TAG="${TARGET_TAG}"
make manifests

echo
echo "*** performing release commit"
git commit -s -m "Release ${TARGET_TAG}" VERSION manifests/ docs/ .github hack/ go.mod go.sum CHANGELOG.md examples/
git tag ${TARGET_TAG}

echo
echo "*** build docker image"
make image


# Include optional parameters in the command output below
set +u
CONTAINER_REGISTRY_OUTPUT=""
if test "${CONTAINER_REGISTRY}" != ""; then
	CONTAINER_REGISTRY_OUTPUT="CONTAINER_REGISTRY='$CONTAINER_REGISTRY'"
fi

IMAGE_NAMESPACE_OUTPUT=""
if test "${IMAGE_NAMESPACE}" != ""; then
	IMAGE_NAMESPACE_OUTPUT="IMAGE_NAMESPACE='$IMAGE_NAMESPACE'"
fi
set -u

echo "*** done"
echo
echo "If everything is fine, push changes to GitHub and the destination container registry:"
echo 
echo "   git push ${TARGET_REMOTE} $RELEASE_BRANCH ${TARGET_TAG}"
echo "   make ${CONTAINER_REGISTRY_OUTPUT} ${IMAGE_NAMESPACE_OUTPUT} IMAGE_TAG='${TARGET_TAG}'  image-push"
echo
echo "Then, create release tag"
