import { Observable } from 'rxjs';
import { find as higherOrder } from 'rxjs/operators';

/* tslint:disable:max-line-length */
export function find<T, S extends T>(this: Observable<T>,
                                     predicate: (value: T, index: number) => value is S,
                                     thisArg?: any): Observable<S | undefined>;
export function find<T>(this: Observable<T>,
                        predicate: (value: T, index: number) => boolean,
                        thisArg?: any): Observable<T | undefined>;
/* tslint:enable:max-line-length */

/**
 * Emits only the first value emitted by the source Observable that meets some
 * condition.
 *
 * <span class="informal">Finds the first value that passes some test and emits
 * that.</span>
 *
 * <img src="./img/find.png" width="100%">
 *
 * `find` searches for the first item in the source Observable that matches the
 * specified condition embodied by the `predicate`, and returns the first
 * occurrence in the source. Unlike {@link first}, the `predicate` is required
 * in `find`, and does not emit an error if a valid value is not found.
 *
 * @example <caption>Find and emit the first click that happens on a DIV element</caption>
 * var clicks = Rx.Observable.fromEvent(document, 'click');
 * var result = clicks.find(ev => ev.target.tagName === 'DIV');
 * result.subscribe(x => console.log(x));
 *
 * @see {@link filter}
 * @see {@link first}
 * @see {@link findIndex}
 * @see {@link take}
 *
 * @param {function(value: T, index: number, source: Observable<T>): boolean} predicate
 * A function called with each item to test for condition matching.
 * @param {any} [thisArg] An optional argument to determine the value of `this`
 * in the `predicate` function.
 * @return {Observable<T>} An Observable of the first item that matches the
 * condition.
 * @method find
 * @owner Observable
 */
export function find<T>(this: Observable<T>, predicate: (value: T, index: number, source: Observable<T>) => boolean,
                        thisArg?: any): Observable<T | undefined> {
  return higherOrder(predicate, thisArg)(this);
}
