'use strict'

const { format, safe, safeand, safenot, safenotor } = require('./safe-format')
const genfun = require('./generate-function')
const { resolveReference, joinPath } = require('./pointer')
const formats = require('./formats')
const { toPointer, ...functions } = require('./scope-functions')
const { scopeMethods } = require('./scope-utils')
const { buildName, types, jsHelpers } = require('./javascript')
const { knownKeywords, schemaVersions, knownVocabularies } = require('./known-keywords')
const { initTracing, andDelta, orDelta, applyDelta, isDynamic, inProperties } = require('./tracing')

const noopRegExps = new Set(['^[\\s\\S]*$', '^[\\S\\s]*$', '^[^]*$', '', '.*', '^', '$'])

// for checking schema parts in consume()
const schemaTypes = new Map(
  Object.entries({
    boolean: (arg) => typeof arg === 'boolean',
    array: (arg) => Array.isArray(arg) && Object.getPrototypeOf(arg) === Array.prototype,
    object: (arg) => arg && Object.getPrototypeOf(arg) === Object.prototype,
    finite: (arg) => Number.isFinite(arg),
    natural: (arg) => Number.isInteger(arg) && arg >= 0,
    string: (arg) => typeof arg === 'string',
    jsonval: (arg) => functions.deepEqual(arg, JSON.parse(JSON.stringify(arg))),
  })
)
const isPlainObject = schemaTypes.get('object')

const schemaIsOlderThan = ($schema, ver) =>
  schemaVersions.indexOf($schema) > schemaVersions.indexOf(`https://json-schema.org/${ver}/schema`)

// Helper methods for semi-structured paths
const propvar = (parent, keyname, inKeys = false, number = false) =>
  Object.freeze({ parent, keyname, inKeys, number }) // property by variable
const propimm = (parent, keyval, checked = false) => Object.freeze({ parent, keyval, checked }) // property by immediate value

const evaluatedStatic = Symbol('evaluatedStatic')
const optDynamic = Symbol('optDynamic')

const constantValue = (schema) => {
  if (typeof schema === 'boolean') return schema
  if (isPlainObject(schema) && Object.keys(schema).length === 0) return true
  return undefined
}

const rootMeta = new WeakMap()
const compileSchema = (schema, root, opts, scope, basePathRoot = '') => {
  const {
    mode = 'default',
    useDefaults = false,
    removeAdditional = false, // supports additionalProperties: false and additionalItems: false
    includeErrors = false,
    allErrors = false,
    dryRun, // unused, just for rest siblings
    allowUnusedKeywords = opts.mode === 'lax',
    allowUnreachable = opts.mode === 'lax',
    requireSchema = opts.mode === 'strong',
    requireValidation = opts.mode === 'strong',
    requireStringValidation = opts.mode === 'strong',
    forbidNoopValues = opts.mode === 'strong', // e.g. $recursiveAnchor: false (it's false by default)
    complexityChecks = opts.mode === 'strong',
    unmodifiedPrototypes = false, // assumes no mangled Object/Array prototypes
    isJSON = false, // assume input to be JSON, which e.g. makes undefined impossible
    $schemaDefault = null,
    formats: optFormats = {},
    weakFormats = opts.mode !== 'strong',
    extraFormats = false,
    schemas, // always a Map, produced at wrapper
    ...unknown
  } = opts
  const fmts = {
    ...formats.core,
    ...(weakFormats ? formats.weak : {}),
    ...(extraFormats ? formats.extra : {}),
    ...optFormats,
  }
  if (Object.keys(unknown).length !== 0)
    throw new Error(`Unknown options: ${Object.keys(unknown).join(', ')}`)

  if (!['strong', 'lax', 'default'].includes(mode)) throw new Error(`Invalid mode: ${mode}`)
  if (!includeErrors && allErrors) throw new Error('allErrors requires includeErrors to be enabled')
  if (requireSchema && $schemaDefault) throw new Error('requireSchema forbids $schemaDefault')
  if (mode === 'strong') {
    const strong = { requireValidation, requireStringValidation, complexityChecks, requireSchema }
    const weak = { weakFormats, allowUnusedKeywords }
    for (const [k, v] of Object.entries(strong)) if (!v) throw new Error(`Strong mode demands ${k}`)
    for (const [k, v] of Object.entries(weak)) if (v) throw new Error(`Strong mode forbids ${k}`)
  }

  const { gensym, getref, genref, genformat } = scopeMethods(scope)

  const buildPath = (prop) => {
    const path = []
    let curr = prop
    while (curr) {
      if (!curr.name) path.unshift(curr)
      curr = curr.parent || curr.errorParent
    }

    // fast case when there are no variables inside path
    if (path.every((part) => part.keyval !== undefined))
      return format('%j', toPointer(path.map((part) => part.keyval)))

    // Be very careful while refactoring, this code significantly affects includeErrors performance
    // It attempts to construct fast code presentation for paths, e.g. "#/abc/"+pointerPart(key0)+"/items/"+i0
    const stringParts = ['#']
    const stringJoined = () => {
      const value = stringParts.map(functions.pointerPart).join('/')
      stringParts.length = 0
      return value
    }
    let res = null
    for (const { keyname, keyval, number } of path) {
      if (keyname) {
        if (!number) scope.pointerPart = functions.pointerPart
        const value = number ? keyname : format('pointerPart(%s)', keyname)
        const str = `${stringJoined()}/`
        res = res ? format('%s+%j+%s', res, str, value) : format('%j+%s', str, value)
      } else if (keyval) stringParts.push(keyval)
    }
    return stringParts.length > 0 ? format('%s+%j', res, `/${stringJoined()}`) : res
  }

  const funname = genref(schema)
  let validate = null // resolve cyclic dependencies
  const wrap = (...args) => {
    const res = validate(...args)
    wrap.errors = validate.errors
    return res
  }
  scope[funname] = wrap

  const fun = genfun()
  fun.write('function validate(data, recursive) {')
  if (includeErrors) fun.write('validate.errors = null')
  if (allErrors) fun.write('let errorCount = 0')
  if (opts[optDynamic]) fun.write('validate.evaluatedDynamic = null')

  const helpers = jsHelpers(fun, scope, propvar, { unmodifiedPrototypes, isJSON }, noopRegExps)
  const { present, forObjectKeys, forArray, patternTest, compare } = helpers

  const recursiveAnchor = schema && schema.$recursiveAnchor === true
  const getMeta = () => rootMeta.get(root) || {}
  const basePathStack = basePathRoot ? [basePathRoot] : []
  const visit = (errors, history, current, node, schemaPath, trace = {}, { constProp } = {}) => {
    // e.g. top-level data and property names, OR already checked by present() in history, OR in keys and not undefined
    const isSub = history.length > 0 && history[history.length - 1].prop === current
    const queryCurrent = () => history.filter((h) => h.prop === current)
    const definitelyPresent =
      !current.parent || current.checked || (current.inKeys && isJSON) || queryCurrent().length > 0

    const name = buildName(current)
    const currPropImm = (...args) => propimm(current, ...args)

    const error = ({ path = [], prop = current, source, suberr }) => {
      const schemaP = toPointer([...schemaPath, ...path])
      const dataP = includeErrors ? buildPath(prop) : null
      if (includeErrors === true && errors && source) {
        // we can include absoluteKeywordLocation later, perhaps
        scope.errorMerge = functions.errorMerge
        const args = [source, schemaP, dataP]
        if (allErrors) {
          fun.write('if (validate.errors === null) validate.errors = []')
          fun.write('validate.errors.push(...%s.map(e => errorMerge(e, %j, %s)))', ...args)
        } else fun.write('validate.errors = [errorMerge(%s[0], %j, %s)]', ...args)
      } else if (includeErrors === true && errors) {
        const errorJS = format('{ keywordLocation: %j, instanceLocation: %s }', schemaP, dataP)
        if (allErrors) {
          fun.write('if (%s === null) %s = []', errors, errors)
          fun.write('%s.push(%s)', errors, errorJS)
        } else fun.write('%s = [%s]', errors, errorJS) // Array assignment is significantly faster, do not refactor the two branches
      }
      if (suberr) mergeerror(suberr) // can only happen in allErrors
      if (allErrors) fun.write('errorCount++')
      else fun.write('return false')
    }
    const errorIf = (condition, errorArgs) => fun.if(condition, () => error(errorArgs))

    const fail = (msg, value) => {
      const comment = value !== undefined ? ` ${JSON.stringify(value)}` : ''
      throw new Error(`${msg}${comment} at ${joinPath(basePathRoot, toPointer(schemaPath))}`)
    }
    const enforce = (ok, ...args) => ok || fail(...args)
    const laxMode = (ok, ...args) => enforce(mode === 'lax' || ok, ...args)
    const enforceMinMax = (a, b) => laxMode(!(node[b] < node[a]), `Invalid ${a} / ${b} combination`)
    const enforceValidation = (msg, suffix = 'should be specified') =>
      enforce(!requireValidation, `[requireValidation] ${msg} ${suffix}`)
    const subPath = (...args) => [...schemaPath, ...args]
    const uncertain = (msg) =>
      enforce(!removeAdditional && !useDefaults, `[removeAdditional/useDefaults] uncertain: ${msg}`)
    const complex = (msg, arg) => enforce(!complexityChecks, `[complexityChecks] ${msg}`, arg)

    // evaluated tracing
    const stat = initTracing()
    const evaluateDelta = (delta) => applyDelta(stat, delta)

    if (typeof node === 'boolean') {
      if (node === true) {
        enforceValidation('schema = true', 'is not allowed') // any is valid here
        return { stat } // nothing is evaluated for true
      }
      errorIf(definitelyPresent || current.inKeys ? true : present(current), {}) // node === false
      evaluateDelta({ type: [] }) // everything is evaluated for false
      return { stat }
    }

    enforce(isPlainObject(node), 'Schema is not an object')
    for (const key of Object.keys(node))
      enforce(knownKeywords.includes(key) || allowUnusedKeywords, 'Keyword not supported:', key)

    if (Object.keys(node).length === 0) {
      enforceValidation('empty rules node', 'is not allowed')
      return { stat } // nothing to validate here, basically the same as node === true
    }

    const unused = new Set(Object.keys(node))
    const consume = (prop, ...ruleTypes) => {
      enforce(unused.has(prop), 'Unexpected double consumption:', prop)
      enforce(functions.hasOwn(node, prop), 'Is not an own property:', prop)
      enforce(ruleTypes.every((t) => schemaTypes.has(t)), 'Invalid type used in consume')
      enforce(ruleTypes.some((t) => schemaTypes.get(t)(node[prop])), 'Unexpected type for', prop)
      unused.delete(prop)
    }
    const get = (prop, ...ruleTypes) => {
      if (node[prop] !== undefined) consume(prop, ...ruleTypes)
      return node[prop]
    }
    const handle = (prop, ruleTypes, handler, errorArgs = {}) => {
      if (node[prop] === undefined) return false
      // opt-out on null is explicit in both places here, don't set default
      consume(prop, ...ruleTypes)
      if (handler !== null) {
        const condition = handler(node[prop])
        if (condition !== null) errorIf(condition, { path: [prop], ...errorArgs })
      }
      return true
    }

    if (node === root) {
      const $schema = get('$schema', 'string') || $schemaDefault
      if ($schema) {
        const version = $schema.replace(/^http:\/\//, 'https://').replace(/#$/, '')
        enforce(schemaVersions.includes(version), 'Unexpected schema version:', version)
        rootMeta.set(root, { exclusiveRefs: schemaIsOlderThan(version, 'draft/2019-09') })
      } else enforce(!requireSchema, '[requireSchema] $schema is required')
      handle('$vocabulary', ['object'], ($vocabulary) => {
        for (const [vocab, flag] of Object.entries($vocabulary)) {
          if (flag === false) continue
          enforce(flag === true && knownVocabularies.includes(vocab), 'Unknown vocabulary:', vocab)
        }
        return null
      })
    }

    handle('examples', ['array'], null) // unused, meta-only
    for (const ignore of ['title', 'description', '$comment']) handle(ignore, ['string'], null) // unused, meta-only strings
    for (const ignore of ['deprecated', 'readOnly', 'writeOnly']) handle(ignore, ['boolean'], null) // unused, meta-only flags

    handle('$defs', ['object'], null) || handle('definitions', ['object'], null) // defs are allowed, those are validated on usage

    const basePath = () => (basePathStack.length > 0 ? basePathStack[basePathStack.length - 1] : '')
    const setId = ($id) => {
      basePathStack.push(joinPath(basePath(), $id))
      return null
    }
    handle('$id', ['string'], setId) || handle('id', ['string'], setId)
    handle('$anchor', ['string'], null) // $anchor is used only for ref resolution, on usage

    if (node === schema && (recursiveAnchor || !forbidNoopValues))
      handle('$recursiveAnchor', ['boolean'], null) // already applied

    // evaluated: declare dynamic
    const needUnevaluated = (rule) =>
      opts[optDynamic] && (node[rule] || node[rule] === false || node === schema)
    const local = Object.freeze({
      items: needUnevaluated('unevaluatedItems') ? gensym('evaluatedItems') : null,
      props: needUnevaluated('unevaluatedProperties') ? gensym('evaluatedProps') : null,
    })
    const dyn = { items: local.items || trace.items, props: local.props || trace.props }
    const canSkipDynamic = () =>
      (!dyn.items || stat.items === Infinity) && (!dyn.props || stat.properties.includes(true))
    const evaluateDeltaDynamic = (delta) => {
      // Skips applying those that have already been proved statically
      if (dyn.items && delta.items > stat.items) fun.write('%s.push(%d)', dyn.items, delta.items)
      if (dyn.props && delta.properties.includes(true) && !stat.properties.includes(true)) {
        fun.write('%s[0].push(true)', dyn.props)
      } else if (dyn.props) {
        const inStat = (properties, patterns) => inProperties(stat, { properties, patterns })
        const properties = delta.properties.filter((x) => !inStat([x], []))
        const patterns = delta.patterns.filter((x) => !inStat([], [x]))
        if (properties.length > 0) fun.write('%s[0].push(...%j)', dyn.props, properties)
        if (patterns.length > 0) fun.write('%s[1].push(...%j)', dyn.props, patterns)
      }
    }
    const applyDynamicToDynamic = (target, items, props) => {
      if (isDynamic(stat).items && target.items && items)
        fun.write('%s.push(...%s)', target.items, items)
      if (isDynamic(stat).properties && target.props && props) {
        fun.write('%s[0].push(...%s[0])', target.props, props)
        fun.write('%s[1].push(...%s[1])', target.props, props)
      }
    }

    const applyRef = (n, errorArgs) => {
      // evaluated: propagate static from ref to current, skips cyclic.
      // Can do this before the call as the call is just a write
      const delta = (scope[n] && scope[n][evaluatedStatic]) || { unknown: true } // assume unknown if ref is cyclic
      evaluateDelta(delta)
      // Allow recursion to here only if $recursiveAnchor is true, else skip from deep recursion
      const recursive = recursiveAnchor ? format('recursive || validate') : format('recursive')
      if (!includeErrors && canSkipDynamic()) return format('!%s(%s, %s)', n, name, recursive) // simple case
      const res = gensym('res')
      const err = gensym('err') // Save and restore errors in case of recursion (if needed)
      const suberr = gensym('suberr')
      if (includeErrors) fun.write('const %s = validate.errors', err)
      fun.write('const %s = %s(%s, %s)', res, n, name, recursive)
      if (includeErrors) fun.write('const %s = %s.errors', suberr, n)
      if (includeErrors) fun.write('validate.errors = %s', err)
      errorIf(safenot(res), { ...errorArgs, source: suberr })
      // evaluated: propagate dynamic from ref to current
      fun.if(res, () => {
        const items = isDynamic(delta).items ? format('%s.evaluatedDynamic[0]', n) : null
        const props = isDynamic(delta).properties ? format('%s.evaluatedDynamic[1]', n) : null
        applyDynamicToDynamic(dyn, items, props)
      })
      return null
    }

    /* Preparation and methods, post-$ref validation will begin at the end of the function */

    // This is used for typechecks, null means * here
    const allIn = (arr, valid) => arr && arr.every((s) => valid.includes(s)) // all arr entries are in valid
    const someIn = (arr, possible) => possible.some((x) => arr === null || arr.includes(x)) // all possible are in arrs

    const parentCheckedType = (...valid) => queryCurrent().some((h) => allIn(h.stat.type, valid))
    const definitelyType = (...valid) => allIn(stat.type, valid) || parentCheckedType(...valid)
    const typeApplicable = (...possible) =>
      someIn(stat.type, possible) && queryCurrent().every((h) => someIn(h.stat.type, possible))

    const enforceRegex = (source, target = node) => {
      enforce(typeof source === 'string', 'Invalid pattern:', source)
      if (requireValidation || requireStringValidation)
        enforce(/^\^.*\$$/.test(source), 'Should start with ^ and end with $:', source)
      if ((/[{+*].*[{+*]/.test(source) || /\)[{+*]/.test(source)) && target.maxLength === undefined)
        complex('maxLength should be specified for pattern:', source)
    }

    // Those checks will need to be skipped if another error is set in this block before those ones
    const haveComplex = node.uniqueItems || node.pattern || node.patternProperties || node.format
    const prev = allErrors && haveComplex ? gensym('prev') : null
    const prevWrap = (shouldWrap, writeBody) =>
      fun.if(shouldWrap && prev !== null ? format('errorCount === %s', prev) : true, writeBody)

    const nexthistory = () => [...history, { stat, prop: current }]
    // Can not be used before undefined check! The one performed by present()
    const rule = (...args) => visit(errors, nexthistory(), ...args).stat
    const subrule = (suberr, ...args) => {
      if (args[0] === current) {
        const constval = constantValue(args[1])
        if (constval === true) return { sub: format('true'), delta: {} }
        if (constval === false) return { sub: format('false'), delta: { type: [] } }
      }
      const sub = gensym('sub')
      fun.write('const %s = (() => {', sub)
      if (allErrors) fun.write('let errorCount = 0') // scoped error counter
      const { stat: delta } = visit(suberr, nexthistory(), ...args)
      if (allErrors) {
        fun.write('return errorCount === 0')
      } else fun.write('return true')
      fun.write('})()')
      return { sub, delta }
    }

    const suberror = () => {
      const suberr = includeErrors && allErrors ? gensym('suberr') : null
      if (suberr) fun.write('let %s = null', suberr)
      return suberr
    }
    const mergeerror = (suberr) => {
      // suberror can be null e.g. on failed empty contains
      if (suberr !== null) fun.if(suberr, () => fun.write('%s.push(...%s)', errors, suberr))
    }

    // Extracted single additional(Items/Properties) rules, for reuse with unevaluated(Items/Properties)
    const additionalItems = (rulePath, limit) => {
      const handled = handle(rulePath, ['object', 'boolean'], (ruleValue) => {
        if (ruleValue === false) {
          if (!removeAdditional) return format('%s.length > %s', name, limit)
          fun.write('if (%s.length > %s) %s.length = %s', name, limit, name, limit)
          return null
        }
        forArray(current, limit, (prop) => rule(prop, ruleValue, subPath(rulePath)))
        return null
      })
      if (handled) evaluateDelta({ items: Infinity })
    }
    const additionalProperties = (rulePath, condition) => {
      const handled = handle(rulePath, ['object', 'boolean'], (ruleValue) => {
        forObjectKeys(current, (sub, key) => {
          fun.if(condition(key), () => {
            if (ruleValue === false && removeAdditional) fun.write('delete %s[%s]', name, key)
            else rule(sub, ruleValue, subPath(rulePath))
          })
        })
        return null
      })
      if (handled) evaluateDelta({ properties: [true] })
    }
    const additionalCondition = (key, properties, patternProperties) =>
      safeand(
        ...properties.map((p) => format('%s !== %j', key, p)),
        ...patternProperties.map((p) => safenot(patternTest(p, key)))
      )

    /* Checks inside blocks are independent, they are happening on the same code depth */

    const checkNumbers = () => {
      const minMax = (value, operator) => format('!(%d %c %s)', value, operator, name) // don't remove negation, accounts for NaN

      if (Number.isFinite(node.exclusiveMinimum)) {
        handle('exclusiveMinimum', ['finite'], (min) => minMax(min, '<'))
      } else {
        handle('minimum', ['finite'], (min) => minMax(min, node.exclusiveMinimum ? '<' : '<='))
        handle('exclusiveMinimum', ['boolean'], null) // handled above
      }

      if (Number.isFinite(node.exclusiveMaximum)) {
        handle('exclusiveMaximum', ['finite'], (max) => minMax(max, '>'))
        enforceMinMax('minimum', 'exclusiveMaximum')
        enforceMinMax('exclusiveMinimum', 'exclusiveMaximum')
      } else if (node.maximum !== undefined) {
        handle('maximum', ['finite'], (max) => minMax(max, node.exclusiveMaximum ? '>' : '>='))
        handle('exclusiveMaximum', ['boolean'], null) // handled above
        enforceMinMax('minimum', 'maximum')
        enforceMinMax('exclusiveMinimum', 'maximum')
      }

      const multipleOf = node.multipleOf === undefined ? 'divisibleBy' : 'multipleOf' // draft3 support
      handle(multipleOf, ['finite'], (value) => {
        enforce(value > 0, `Invalid ${multipleOf}:`, value)
        const [frac, exp] = `${value}.`.split('.')[1].split('e-')
        const e = frac.length + (exp ? Number(exp) : 0)
        if (Number.isInteger(value * 2 ** e)) return format('%s %% %d !== 0', name, value) // exact
        scope.isMultipleOf = functions.isMultipleOf
        const args = [name, value, e, Math.round(value * Math.pow(10, e))] // precompute for performance
        return format('!isMultipleOf(%s, %d, 1e%d, %d)', ...args)
      })
    }

    const checkStrings = () => {
      handle('maxLength', ['natural'], (max) => {
        scope.stringLength = functions.stringLength
        return format('%s.length > %d && stringLength(%s) > %d', name, max, name, max)
      })
      handle('minLength', ['natural'], (min) => {
        scope.stringLength = functions.stringLength
        return format('%s.length < %d || stringLength(%s) < %d', name, min, name, min)
      })
      enforceMinMax('minLength', 'maxLength')

      prevWrap(true, () => {
        const checkFormat = (fmtname, target, formatsObj = fmts) => {
          const known = typeof fmtname === 'string' && functions.hasOwn(formatsObj, fmtname)
          enforce(known, 'Unrecognized format used:', fmtname)
          const formatImpl = formatsObj[fmtname]
          const valid = formatImpl instanceof RegExp || typeof formatImpl === 'function'
          enforce(valid, 'Invalid format used:', fmtname)
          if (formatImpl instanceof RegExp) {
            // built-in formats are fine, check only ones from options
            if (functions.hasOwn(optFormats, fmtname)) enforceRegex(formatImpl.source)
            return format('!%s.test(%s)', genformat(formatImpl), target)
          }
          return format('!%s(%s)', genformat(formatImpl), target)
        }

        handle('format', ['string'], (value) => {
          evaluateDelta({ fullstring: true })
          return checkFormat(value, name)
        })

        handle('pattern', ['string'], (pattern) => {
          enforceRegex(pattern)
          evaluateDelta({ fullstring: true })
          return noopRegExps.has(pattern) ? null : safenot(patternTest(pattern, name))
        })

        enforce(node.contentSchema !== false, 'contentSchema cannot be set to false')
        if (node.contentEncoding || node.contentMediaType || node.contentSchema) {
          const dec = gensym('dec')
          if (node.contentMediaType) fun.write('let %s = %s', dec, name)

          if (node.contentEncoding === 'base64') {
            errorIf(checkFormat('base64', name, formats.extra), { path: ['contentEncoding'] })
            if (node.contentMediaType) {
              scope.deBase64 = functions.deBase64
              fun.write('try {')
              fun.write('%s = deBase64(%s)', dec, dec)
            }
            consume('contentEncoding', 'string')
          } else enforce(!node.contentEncoding, 'Unknown contentEncoding:', node.contentEncoding)

          let json = false
          if (node.contentMediaType === 'application/json') {
            fun.write('try {')
            fun.write('%s = JSON.parse(%s)', dec, dec)
            json = true
            consume('contentMediaType', 'string')
          } else enforce(!node.contentMediaType, 'Unknown contentMediaType:', node.contentMediaType)

          if (node.contentSchema) {
            enforce(json, 'contentSchema requires contentMediaType application/json')
            const decprop = Object.freeze({ name: dec, errorParent: current })
            rule(decprop, node.contentSchema, subPath('contentSchema')) // TODO: isJSON true for speed?
            consume('contentSchema', 'object', 'array')
            evaluateDelta({ fullstring: true })
          }
          if (node.contentMediaType) {
            fun.write('} catch (e) {')
            error({ path: ['contentMediaType'] })
            fun.write('}')
            if (node.contentEncoding) {
              fun.write('} catch (e) {')
              error({ path: ['contentEncoding'] })
              fun.write('}')
            }
          }
        }
      })
    }

    const checkArrays = () => {
      handle('maxItems', ['natural'], (max) => {
        if (Array.isArray(node.items) && node.items.length > max)
          fail(`Invalid maxItems: ${max} is less than items array length`)
        return format('%s.length > %d', name, max)
      })
      handle('minItems', ['natural'], (min) => format('%s.length < %d', name, min)) // can be higher that .items length with additionalItems
      enforceMinMax('minItems', 'maxItems')

      handle('items', ['object', 'array', 'boolean'], (items) => {
        if (Array.isArray(items)) {
          for (let p = 0; p < items.length; p++) rule(currPropImm(p), items[p], subPath(`${p}`))
          evaluateDelta({ items: items.length })
        } else {
          forArray(current, format('0'), (prop) => rule(prop, items, subPath('items')))
          evaluateDelta({ items: Infinity })
        }
        return null
      })

      if (Array.isArray(node.items))
        additionalItems('additionalItems', format('%d', node.items.length))
      // Else additionalItems is allowed, but ignored per some spec tests!
      // We do nothing and let it throw except for in allowUnusedKeywords mode
      // As a result, omitting .items is not allowed by default, only in allowUnusedKeywords mode

      handle('contains', ['object', 'boolean'], () => {
        uncertain('contains')
        const passes = gensym('passes')
        fun.write('let %s = 0', passes)

        const suberr = suberror()
        forArray(current, format('0'), (prop) => {
          const { sub } = subrule(suberr, prop, node.contains, subPath('contains'))
          fun.if(sub, () => fun.write('%s++', passes))
          // evaluateDelta({ unknown: true }) // draft2020: contains counts towards evaluatedItems
        })

        if (!handle('minContains', ['natural'], (mn) => format('%s < %d', passes, mn), { suberr }))
          errorIf(format('%s < 1', passes), { path: ['contains'], suberr })

        handle('maxContains', ['natural'], (max) => format('%s > %d', passes, max))
        enforceMinMax('minContains', 'maxContains')

        return null
      })

      const uniqueSimple = () => {
        if (node.maxItems !== undefined) return true
        if (Array.isArray(node.items) && node.additionalItems === false) return true
        const itemsSimple = (ischema) => {
          if (!isPlainObject(ischema)) return false
          if (ischema.enum || functions.hasOwn(ischema, 'const')) return true
          if (ischema.type) {
            const itemTypes = Array.isArray(ischema.type) ? ischema.type : [ischema.type]
            const primitiveTypes = ['null', 'boolean', 'number', 'integer', 'string']
            if (itemTypes.every((itemType) => primitiveTypes.includes(itemType))) return true
          }
          if (ischema.$ref) {
            const [sub] = resolveReference(root, schemas, ischema.$ref, basePath())[0] || []
            if (itemsSimple(sub)) return true
          }
          return false
        }
        if (itemsSimple(node.items)) return true
        return false
      }
      prevWrap(true, () => {
        handle('uniqueItems', ['boolean'], (uniqueItems) => {
          if (uniqueItems === false) return null
          if (!uniqueSimple()) complex('maxItems should be specified for non-primitive uniqueItems')
          Object.assign(scope, { unique: functions.unique, deepEqual: functions.deepEqual })
          return format('!unique(%s)', name)
        })
      })
    }

    // if allErrors is false, we can skip present check for required properties validated before
    const checked = (p) =>
      !allErrors &&
      (stat.required.includes(p) || queryCurrent().some((h) => h.stat.required.includes(p)))

    const checkObjects = () => {
      const propertiesCount = format('Object.keys(%s).length', name)
      handle('maxProperties', ['natural'], (max) => format('%s > %d', propertiesCount, max))
      handle('minProperties', ['natural'], (min) => format('%s < %d', propertiesCount, min))
      enforceMinMax('minProperties', 'maxProperties')

      handle('propertyNames', ['object', 'boolean'], (names) => {
        forObjectKeys(current, (sub, key) => {
          const nameSchema = typeof names === 'object' ? { type: 'string', ...names } : names
          const nameprop = Object.freeze({ name: key, errorParent: sub, type: 'string' })
          rule(nameprop, nameSchema, subPath('propertyNames'))
        })
        return null
      })

      handle('required', ['array'], (required) => {
        for (const req of required) {
          if (checked(req)) continue
          const prop = currPropImm(req)
          errorIf(safenot(present(prop)), { path: ['required'], prop })
        }
        evaluateDelta({ required })
        return null
      })

      for (const dependencies of ['dependencies', 'dependentRequired', 'dependentSchemas']) {
        handle(dependencies, ['object'], (value) => {
          for (const key of Object.keys(value)) {
            const deps = typeof value[key] === 'string' ? [value[key]] : value[key]
            const item = currPropImm(key, checked(key))
            if (Array.isArray(deps) && dependencies !== 'dependentSchemas') {
              const clauses = deps.filter((k) => !checked(k)).map((k) => present(currPropImm(k)))
              const condition = safenot(safeand(...clauses))
              const errorArgs = { path: [dependencies, key] }
              if (clauses.length === 0) {
                // nothing to do
              } else if (item.checked) {
                errorIf(condition, errorArgs)
                evaluateDelta({ required: deps })
              } else {
                errorIf(safeand(present(item), condition), errorArgs)
              }
            } else if (
              ((typeof deps === 'object' && !Array.isArray(deps)) || typeof deps === 'boolean') &&
              dependencies !== 'dependentRequired'
            ) {
              uncertain(dependencies)
              fun.if(item.checked ? true : present(item), () => {
                const delta = rule(current, deps, subPath(dependencies, key), dyn)
                evaluateDelta(orDelta({}, delta))
                evaluateDeltaDynamic(delta)
              })
            } else fail(`Unexpected ${dependencies} entry`)
          }
          return null
        })
      }

      handle('properties', ['object'], (properties) => {
        for (const p of Object.keys(properties)) {
          if (constProp === p) continue // checked in discriminator, avoid double-check
          rule(currPropImm(p, checked(p)), properties[p], subPath('properties', p))
        }
        evaluateDelta({ properties: Object.keys(properties) })
        return null
      })

      prevWrap(node.patternProperties, () => {
        handle('patternProperties', ['object'], (patternProperties) => {
          forObjectKeys(current, (sub, key) => {
            for (const p of Object.keys(patternProperties)) {
              enforceRegex(p, node.propertyNames || {})
              fun.if(patternTest(p, key), () => {
                rule(sub, patternProperties[p], subPath('patternProperties', p))
              })
            }
          })
          evaluateDelta({ patterns: Object.keys(patternProperties) })
          return null
        })
        if (node.additionalProperties || node.additionalProperties === false) {
          const properties = Object.keys(node.properties || {})
          const patternProperties = Object.keys(node.patternProperties || {})
          const condition = (key) => additionalCondition(key, properties, patternProperties)
          additionalProperties('additionalProperties', condition)
        }
      })
    }

    const checkConst = () => {
      if (handle('const', ['jsonval'], (val) => safenot(compare(name, val)))) return true
      return handle('enum', ['array'], (vals) => {
        const objects = vals.filter((value) => value && typeof value === 'object')
        const primitive = vals.filter((value) => !(value && typeof value === 'object'))
        return safenotor(...[...primitive, ...objects].map((value) => compare(name, value)))
      })
    }

    const checkGeneric = () => {
      handle('not', ['object', 'boolean'], (not) => subrule(null, current, not, subPath('not')).sub)
      if (node.not) uncertain('not')

      const thenOrElse = node.then || node.then === false || node.else || node.else === false
      if (thenOrElse)
        handle('if', ['object', 'boolean'], (ifS) => {
          uncertain('if/then/else')
          const { sub, delta: deltaIf } = subrule(null, current, ifS, subPath('if'), dyn)
          let handleElse, handleThen, deltaElse, deltaThen
          handle('else', ['object', 'boolean'], (elseS) => {
            handleElse = () => {
              deltaElse = rule(current, elseS, subPath('else'), dyn)
              evaluateDeltaDynamic(deltaElse)
            }
            return null
          })
          handle('then', ['object', 'boolean'], (thenS) => {
            handleThen = () => {
              deltaThen = rule(current, thenS, subPath('then'), dyn)
              evaluateDeltaDynamic(andDelta(deltaIf, deltaThen))
            }
            return null
          })
          fun.if(sub, handleThen, handleElse)
          evaluateDelta(orDelta(deltaElse || {}, andDelta(deltaIf, deltaThen || {})))
          return null
        })

      const performAllOf = (allOf, rulePath = 'allOf') => {
        enforce(allOf.length > 0, `${rulePath} cannot be empty`)
        for (const [key, sch] of Object.entries(allOf))
          evaluateDelta(rule(current, sch, subPath(rulePath, key), dyn))
        return null
      }
      handle('allOf', ['array'], (allOf) => performAllOf(allOf))

      let handleDiscriminator = null
      handle('discriminator', ['object'], (discriminator) => {
        const seen = new Set()
        const fix = (check, message, arg) => enforce(check, `[discriminator]: ${message}`, arg)
        const { propertyName: pname, mapping: map, ...e0 } = discriminator
        const prop = currPropImm(pname)
        fix(pname && !node.oneOf !== !node.anyOf, 'need propertyName, oneOf OR anyOf')
        fix(Object.keys(e0).length === 0, 'only "propertyName" and "mapping" are supported')
        const keylen = (obj) => (isPlainObject(obj) ? Object.keys(obj).length : null)
        handleDiscriminator = (branches, ruleName) => {
          const runDiscriminator = () => {
            fun.write('switch (%s) {', buildName(prop)) // we could also have used ifs for complex types
            let delta
            for (const [i, branch] of Object.entries(branches)) {
              const { const: myval, enum: myenum, ...e1 } = (branch.properties || {})[pname] || {}
              let vals = myval !== undefined ? [myval] : myenum
              if (!vals && branch.$ref) {
                const [sub] = resolveReference(root, schemas, branch.$ref, basePath())[0] || []
                enforce(isPlainObject(sub), 'failed to resolve $ref:', branch.$ref)
                const rprop = (sub.properties || {})[pname] || {}
                vals = rprop.const !== undefined ? [rprop.const] : rprop.enum
              }
              const ok1 = Array.isArray(vals) && vals.length > 0
              fix(ok1, 'branches should have unique string const or enum values for [propertyName]')
              const ok2 = Object.keys(e1).length === 0 && (!myval || !myenum)
              fix(ok2, 'only const OR enum rules are allowed on [propertyName] in branches')
              for (const val of vals) {
                const okMapping = !map || (functions.hasOwn(map, val) && map[val] === branch.$ref)
                fix(okMapping, 'mismatching mapping for', val)
                const valok = typeof val === 'string' && !seen.has(val)
                fix(valok, 'const/enum values for [propertyName] should be unique strings')
                seen.add(val)
                fun.write('case %j:', val)
              }
              const subd = rule(current, branch, subPath(ruleName, i), dyn, { constProp: pname })
              evaluateDeltaDynamic(subd)
              delta = delta ? orDelta(delta, subd) : subd
              fun.write('break')
            }
            fix(map === undefined || keylen(map) === seen.size, 'mismatching mapping size')
            evaluateDelta(delta)
            fun.write('default:')
            error({ path: [ruleName] })
            fun.write('}')
          }
          const propCheck = () => {
            if (!checked(pname)) {
              const errorPath = ['discriminator', 'propertyName']
              fun.if(present(prop), runDiscriminator, () => error({ path: errorPath, prop }))
            } else runDiscriminator()
          }
          if (allErrors || !functions.deepEqual(stat.type, ['object'])) {
            fun.if(types.get('object')(name), propCheck, () => error({ path: ['discriminator'] }))
          } else propCheck()
          // can't evaluateDelta on type and required to not break the checks below, but discriminator
          // is usually used with refs anyway so those won't be of much use
          fix(functions.deepEqual(stat.type, ['object']), 'has to be checked for type:', 'object')
          fix(stat.required.includes(pname), 'propertyName should be placed in required:', pname)
          return null
        }
        return null
      })

      handle('anyOf', ['array'], (anyOf) => {
        enforce(anyOf.length > 0, 'anyOf cannot be empty')
        if (anyOf.length === 1) return performAllOf(anyOf)
        if (handleDiscriminator) return handleDiscriminator(anyOf, 'anyOf')
        uncertain('anyOf, use discriminator to make it certain')
        const suberr = suberror()
        if (!canSkipDynamic()) {
          // In this case, all have to be checked to gather evaluated properties
          const entries = Object.entries(anyOf).map(([key, sch]) =>
            subrule(suberr, current, sch, subPath('anyOf', key), dyn)
          )
          evaluateDelta(entries.reduce((acc, cur) => orDelta(acc, cur.delta), {}))
          const condition = safenotor(...entries.map(({ sub }) => sub))
          errorIf(condition, { path: ['anyOf'], suberr })
          for (const { delta, sub } of entries) fun.if(sub, () => evaluateDeltaDynamic(delta))
          return null
        }
        let delta
        let body = () => error({ path: ['anyOf'], suberr })
        for (const [key, sch] of Object.entries(anyOf).reverse()) {
          const oldBody = body
          body = () => {
            const { sub, delta: deltaVar } = subrule(suberr, current, sch, subPath('anyOf', key))
            fun.if(safenot(sub), oldBody)
            delta = delta ? orDelta(delta, deltaVar) : deltaVar
          }
        }
        body()
        evaluateDelta(delta)
        return null
      })

      handle('oneOf', ['array'], (oneOf) => {
        enforce(oneOf.length > 0, 'oneOf cannot be empty')
        if (oneOf.length === 1) return performAllOf(oneOf)
        if (handleDiscriminator) return handleDiscriminator(oneOf, 'oneOf')
        uncertain('oneOf, use discriminator to make it certain')
        const passes = gensym('passes')
        fun.write('let %s = 0', passes)
        const suberr = suberror()
        let delta
        let i = 0
        const entries = Object.entries(oneOf).map(([key, sch]) => {
          if (!includeErrors && i++ > 1) errorIf(format('%s > 1', passes), { path: ['oneOf'] })
          const entry = subrule(suberr, current, sch, subPath('oneOf', key), dyn)
          fun.if(entry.sub, () => fun.write('%s++', passes))
          delta = delta ? orDelta(delta, entry.delta) : entry.delta
          return entry
        })
        evaluateDelta(delta)
        errorIf(format('%s !== 1', passes), { path: ['oneOf'] })
        fun.if(format('%s === 0', passes), () => mergeerror(suberr)) // if none matched, dump all errors
        for (const entry of entries) fun.if(entry.sub, () => evaluateDeltaDynamic(entry.delta))
        return null
      })
    }

    const typeWrap = (checkBlock, validTypes, queryType) => {
      const [funSize, unusedSize] = [fun.size(), unused.size]
      fun.if(definitelyType(...validTypes) ? true : queryType, checkBlock)
      // enforce check that non-applicable blocks are empty and no rules were applied
      if (funSize !== fun.size() || unusedSize !== unused.size)
        enforce(typeApplicable(...validTypes), `Unexpected rules in type`, node.type)
    }

    // Unevaluated validation
    const checkArraysFinal = () => {
      if (stat.items === Infinity) {
        // Everything is statically evaluated, so this check is unreachable. Allow only 'false' rule here.
        if (node.unevaluatedItems === false) consume('unevaluatedItems', 'boolean')
      } else if (node.unevaluatedItems || node.unevaluatedItems === false) {
        if (isDynamic(stat).items) {
          if (!opts[optDynamic]) throw new Error('Dynamic unevaluated tracing is not enabled')
          additionalItems('unevaluatedItems', format('Math.max(%d, ...%s)', stat.items, dyn.items))
        } else {
          additionalItems('unevaluatedItems', format('%d', stat.items))
        }
      }
    }
    const checkObjectsFinal = () => {
      prevWrap(stat.patterns.length > 0 || stat.dyn.patterns.length > 0 || stat.unknown, () => {
        if (stat.properties.includes(true)) {
          // Everything is statically evaluated, so this check is unreachable. Allow only 'false' rule here.
          if (node.unevaluatedProperties === false) consume('unevaluatedProperties', 'boolean')
        } else if (node.unevaluatedProperties || node.unevaluatedProperties === false) {
          const notStatic = (key) => additionalCondition(key, stat.properties, stat.patterns)
          if (isDynamic(stat).properties) {
            if (!opts[optDynamic]) throw new Error('Dynamic unevaluated tracing is not enabled')
            scope.propertyIn = functions.propertyIn
            const notDynamic = (key) => format('!propertyIn(%s, %s)', key, dyn.props)
            const condition = (key) => safeand(notStatic(key), notDynamic(key))
            additionalProperties('unevaluatedProperties', condition)
          } else {
            additionalProperties('unevaluatedProperties', notStatic)
          }
        }
      })
    }

    const performValidation = () => {
      if (prev !== null) fun.write('const %s = errorCount', prev)
      if (checkConst()) {
        // const/enum shouldn't have any other validation rules except for already checked type/$ref
        enforce(unused.size === 0, 'Unexpected keywords mixed with const or enum:', [...unused])
        const typeKeys = [...types.keys()] // we don't extract type from const/enum, it's enough that we know that it's present
        evaluateDelta({ properties: [true], items: Infinity, type: typeKeys, fullstring: true }) // everything is evaluated for const
        return
      }

      typeWrap(checkNumbers, ['number', 'integer'], types.get('number')(name))
      typeWrap(checkStrings, ['string'], types.get('string')(name))
      typeWrap(checkArrays, ['array'], types.get('array')(name))
      typeWrap(checkObjects, ['object'], types.get('object')(name))
      checkGeneric()

      // evaluated: apply static + dynamic
      typeWrap(checkArraysFinal, ['array'], types.get('array')(name))
      typeWrap(checkObjectsFinal, ['object'], types.get('object')(name))

      // evaluated: propagate dynamic to parent dynamic (aka trace)
      // static to parent is merged via return value
      applyDynamicToDynamic(trace, local.items, local.props)
    }

    // main post-presence check validation function
    const writeMain = () => {
      if (local.items) fun.write('const %s = [0]', local.items)
      if (local.props) fun.write('const %s = [[], []]', local.props)

      // refs
      handle('$ref', ['string'], ($ref) => {
        const resolved = resolveReference(root, schemas, $ref, basePath())
        const [sub, subRoot, path] = resolved[0] || []
        if (!sub && sub !== false) fail('failed to resolve $ref:', $ref)
        const n = getref(sub) || compileSchema(sub, subRoot, opts, scope, path)
        return applyRef(n, { path: ['$ref'] })
      })
      if (node.$ref && getMeta().exclusiveRefs) {
        enforce(!opts[optDynamic], 'unevaluated* is supported only on draft2019-09 and above')
        return // ref overrides any sibling keywords for older schemas
      }
      handle('$recursiveRef', ['string'], ($recursiveRef) => {
        enforce($recursiveRef === '#', 'Behavior of $recursiveRef is defined only for "#"')
        // Resolve to recheck that recursive ref is enabled
        const resolved = resolveReference(root, schemas, '#', basePath())
        const [sub, subRoot, path] = resolved[0] || []
        laxMode(sub.$recursiveAnchor, '$recursiveRef without $recursiveAnchor')
        if (!sub.$recursiveAnchor || !recursiveAnchor) {
          // regular ref
          const n = getref(sub) || compileSchema(sub, subRoot, opts, scope, path)
          return applyRef(n, { path: ['$recursiveRef'] })
        }
        // Apply deep recursion from here only if $recursiveAnchor is true, else just run self
        const n = recursiveAnchor ? format('(recursive || validate)') : format('validate')
        return applyRef(n, { path: ['$recursiveRef'] })
      })

      // typecheck
      let typeCheck = null
      handle('type', ['string', 'array'], (type) => {
        const typearr = Array.isArray(type) ? type : [type]
        for (const t of typearr) enforce(typeof t === 'string' && types.has(t), 'Unknown type:', t)
        if (current.type) {
          enforce(functions.deepEqual(typearr, [current.type]), 'One type allowed:', current.type)
          evaluateDelta({ type: [current.type] })
          return null
        }
        if (parentCheckedType(...typearr)) return null
        const filteredTypes = typearr.filter((t) => typeApplicable(t))
        if (filteredTypes.length === 0) fail('No valid types possible')
        evaluateDelta({ type: typearr }) // can be safely done here, filteredTypes already prepared
        typeCheck = safenotor(...filteredTypes.map((t) => types.get(t)(name)))
        return null
      })

      // main validation block
      // if type validation was needed and did not return early, wrap this inside an else clause.
      if (typeCheck && allErrors) {
        fun.if(typeCheck, () => error({ path: ['type'] }), performValidation)
      } else {
        if (typeCheck) errorIf(typeCheck, { path: ['type'] })
        performValidation()
      }

      // account for maxItems to recheck if they limit items. TODO: perhaps we could keep track of this in stat?
      if (stat.items < Infinity && node.maxItems <= stat.items) evaluateDelta({ items: Infinity })
    }

    // presence check and call main validation block
    if (node.default !== undefined && useDefaults) {
      if (definitelyPresent) fail('Can not apply default value here (e.g. at root)')
      const defvalue = get('default', 'jsonval')
      fun.if(present(current), writeMain, () => fun.write('%s = %j', name, defvalue))
    } else {
      handle('default', ['jsonval'], null) // unused
      fun.if(definitelyPresent ? true : present(current), writeMain)
    }

    // Checks related to static schema analysis
    if (!allowUnreachable) enforce(!fun.optimizedOut, 'some checks are never reachable')
    if (isSub) {
      const logicalOp = ['not', 'if', 'then', 'else'].includes(schemaPath[schemaPath.length - 1])
      const branchOp = ['oneOf', 'anyOf', 'allOf'].includes(schemaPath[schemaPath.length - 2])
      const depOp = ['dependencies', 'dependentSchemas'].includes(schemaPath[schemaPath.length - 2])
      // Coherence check, unreachable, double-check that we came from expected path
      enforce(logicalOp || branchOp || depOp, 'Unexpected')
    } else if (!schemaPath.includes('not')) {
      // 'not' does not mark anything as evaluated (unlike even if/then/else), so it's safe to exclude from these
      // checks, as we are sure that everything will be checked without it. It can be viewed as a pure add-on.
      if (!stat.type) enforceValidation('type')
      if (typeApplicable('array') && stat.items !== Infinity)
        enforceValidation(node.items ? 'additionalItems or unevaluatedItems' : 'items rule')
      if (typeApplicable('object') && !stat.properties.includes(true))
        enforceValidation('additionalProperties or unevaluatedProperties')
      if (typeof node.propertyNames !== 'object')
        for (const sub of ['additionalProperties', 'unevaluatedProperties'])
          if (node[sub]) enforceValidation(`wild-card ${sub}`, 'requires propertyNames')
      if (!stat.fullstring && requireStringValidation) {
        const stringWarning = 'pattern, format or contentSchema should be specified for strings'
        fail(`[requireStringValidation] ${stringWarning}, use pattern: ^[\\s\\S]*$ to opt-out`)
      }
    }
    if (node.properties && !node.required) enforceValidation('if properties is used, required')
    enforce(unused.size === 0 || allowUnusedKeywords, 'Unprocessed keywords:', [...unused])

    return { stat, local } // return statically evaluated
  }

  const { stat, local } = visit(format('validate.errors'), [], { name: safe('data') }, schema, [])

  // evaluated: return dynamic for refs
  if (opts[optDynamic] && (isDynamic(stat).items || isDynamic(stat).properties)) {
    if (!local) throw new Error('Failed to trace dynamic properties') // Unreachable
    fun.write('validate.evaluatedDynamic = [%s, %s]', local.items, local.props)
  }

  if (allErrors) fun.write('return errorCount === 0')
  else fun.write('return true')

  fun.write('}')

  validate = fun.makeFunction(scope)
  validate[evaluatedStatic] = stat
  delete scope[funname] // more logical key order
  scope[funname] = validate
  return funname
}

const compile = (schema, opts) => {
  try {
    const scope = Object.create(null)
    return { scope, ref: compileSchema(schema, schema, opts, scope) }
  } catch (e) {
    // For performance, we try to build the schema without dynamic tracing first, then re-run with
    // it enabled if needed. Enabling it without need can give up to about 40% performance drop.
    if (!opts[optDynamic] && e.message === 'Dynamic unevaluated tracing is not enabled')
      return compile(schema, { ...opts, [optDynamic]: true })
    throw e
  }
}

module.exports = { compile }
