"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Oas3Types = void 0;
const _1 = require(".");
const ref_utils_1 = require("../ref-utils");
const responseCodeRegexp = /^[0-9][0-9Xx]{2}$/;
const DefinitionRoot = {
    properties: {
        openapi: null,
        info: 'Info',
        servers: _1.listOf('Server'),
        security: _1.listOf('SecurityRequirement'),
        tags: _1.listOf('Tag'),
        externalDocs: 'ExternalDocs',
        paths: 'PathMap',
        components: 'Components',
        'x-webhooks': 'WebhooksMap',
    },
    required: ['openapi', 'paths', 'info'],
};
const Tag = {
    properties: {
        name: { type: 'string' },
        description: { type: 'string' },
        externalDocs: 'ExternalDocs',
    },
    required: ['name'],
};
const ExternalDocs = {
    properties: {
        description: { type: 'string' },
        url: { type: 'string' },
    },
    required: ['url'],
};
const Server = {
    properties: {
        url: { type: 'string' },
        description: { type: 'string' },
        variables: _1.mapOf('ServerVariable'),
    },
    required: ['url'],
};
const ServerVariable = {
    properties: {
        enum: {
            type: 'array',
            items: { type: 'string' },
        },
        default: { type: 'string' },
        description: null,
    },
    required: ['default'],
};
const SecurityRequirement = {
    properties: {},
    additionalProperties: { type: 'array', items: { type: 'string' } },
};
const Info = {
    properties: {
        title: { type: 'string' },
        version: { type: 'string' },
        description: { type: 'string' },
        termsOfService: { type: 'string' },
        contact: 'Contact',
        license: 'License',
    },
    required: ['title', 'version'],
};
const Contact = {
    properties: {
        name: { type: 'string' },
        url: { type: 'string' },
        email: { type: 'string' },
    },
};
const License = {
    properties: {
        name: { type: 'string' },
        url: { type: 'string' },
    },
    required: ['name'],
};
const PathMap = {
    properties: {},
    additionalProperties: (_value, key) => key.startsWith('/') ? 'PathItem' : undefined,
};
const WebhooksMap = {
    properties: {},
    additionalProperties: () => 'PathItem',
};
const PathItem = {
    properties: {
        $ref: { type: 'string' },
        servers: _1.listOf('Server'),
        parameters: _1.listOf('Parameter'),
        summary: { type: 'string' },
        description: { type: 'string' },
        get: 'Operation',
        put: 'Operation',
        post: 'Operation',
        delete: 'Operation',
        options: 'Operation',
        head: 'Operation',
        patch: 'Operation',
        trace: 'Operation',
    },
};
const Parameter = {
    properties: {
        name: { type: 'string' },
        in: { enum: ['query', 'header', 'path', 'cookie'] },
        description: { type: 'string' },
        required: { type: 'boolean' },
        deprecated: { type: 'boolean' },
        allowEmptyValue: { type: 'boolean' },
        style: {
            enum: ['form', 'simple', 'label', 'matrix', 'spaceDelimited', 'pipeDelimited', 'deepObject'],
        },
        explode: { type: 'boolean' },
        allowReserved: { type: 'boolean' },
        schema: 'Schema',
        example: { isExample: true },
        examples: _1.mapOf('Example'),
        content: 'MediaTypeMap',
    },
    required: ['name', 'in'],
};
const Callback = {
    properties: {},
    additionalProperties: 'PathItem',
};
const Operation = {
    properties: {
        tags: {
            type: 'array',
            items: { type: 'string' },
        },
        summary: { type: 'string' },
        description: { type: 'string' },
        externalDocs: 'ExternalDocs',
        operationId: { type: 'string' },
        parameters: _1.listOf('Parameter'),
        security: _1.listOf('SecurityRequirement'),
        servers: _1.listOf('Server'),
        requestBody: 'RequestBody',
        responses: 'ResponsesMap',
        deprecated: { type: 'boolean' },
        callbacks: _1.mapOf('Callback'),
        'x-codeSamples': _1.listOf('XCodeSample'),
        'x-code-samples': _1.listOf('XCodeSample'), // deprecated
    },
    required: ['responses'],
};
const XCodeSample = {
    properties: {
        lang: { type: 'string' },
        label: { type: 'string' },
        source: { type: 'string' },
    },
};
const RequestBody = {
    properties: {
        description: { type: 'string' },
        required: { type: 'boolean' },
        content: 'MediaTypeMap',
    },
    required: ['content'],
};
const MediaTypeMap = {
    properties: {},
    additionalProperties: 'MediaType',
};
const MediaType = {
    properties: {
        schema: 'Schema',
        example: { isExample: true },
        examples: _1.mapOf('Example'),
        encoding: _1.mapOf('Encoding'),
    },
};
const Example = {
    properties: {
        value: { isExample: true },
        summary: { type: 'string' },
        description: { type: 'string' },
        externalValue: { type: 'string' },
    },
};
const Encoding = {
    properties: {
        contentType: { type: 'string' },
        headers: _1.mapOf('Header'),
        style: {
            enum: ['form', 'simple', 'label', 'matrix', 'spaceDelimited', 'pipeDelimited', 'deepObject'],
        },
        explode: { type: 'boolean' },
        allowReserved: { type: 'boolean' },
    },
};
const Header = {
    properties: {
        description: { type: 'string' },
        required: { type: 'boolean' },
        deprecated: { type: 'boolean' },
        allowEmptyValue: { type: 'boolean' },
        style: {
            enum: ['form', 'simple', 'label', 'matrix', 'spaceDelimited', 'pipeDelimited', 'deepObject'],
        },
        explode: { type: 'boolean' },
        allowReserved: { type: 'boolean' },
        schema: 'Schema',
        example: { isExample: true },
        examples: _1.mapOf('Example'),
        content: 'MediaTypeMap',
    },
};
const ResponsesMap = {
    properties: { default: 'Response' },
    additionalProperties: (_v, key) => responseCodeRegexp.test(key) ? 'Response' : undefined,
};
const Response = {
    properties: {
        description: { type: 'string' },
        headers: _1.mapOf('Header'),
        content: 'MediaTypeMap',
        links: _1.mapOf('Link'),
    },
    required: ['description'],
};
const Link = {
    properties: {
        operationRef: { type: 'string' },
        operationId: { type: 'string' },
        parameters: null,
        requestBody: null,
        description: { type: 'string' },
        server: 'Server',
    },
};
const Schema = {
    properties: {
        externalDocs: 'ExternalDocs',
        discriminator: 'Discriminator',
        title: { type: 'string' },
        multipleOf: { type: 'number', minimum: 0 },
        maximum: { type: 'number' },
        minimum: { type: 'number' },
        exclusiveMaximum: { type: 'boolean' },
        exclusiveMinimum: { type: 'boolean' },
        maxLength: { type: 'number', minimum: 0 },
        minLength: { type: 'number', minimum: 0 },
        pattern: { type: 'string' },
        maxItems: { type: 'number', minimum: 0 },
        minItems: { type: 'number', minimum: 0 },
        uniqueItems: { type: 'boolean' },
        maxProperties: { type: 'number', minimum: 0 },
        minProperties: { type: 'number', minimum: 0 },
        required: { type: 'array', items: { type: 'string' } },
        enum: { type: 'array' },
        type: {
            enum: ['object', 'array', 'string', 'number', 'integer', 'boolean', 'null'],
        },
        allOf: _1.listOf('Schema'),
        anyOf: _1.listOf('Schema'),
        oneOf: _1.listOf('Schema'),
        not: 'Schema',
        properties: 'SchemaProperties',
        items: (value) => {
            if (Array.isArray(value)) {
                return _1.listOf('Schema');
            }
            else {
                return 'Schema';
            }
        },
        additionalProperties: (value) => {
            if (typeof value === 'boolean') {
                return { type: 'boolean' };
            }
            else {
                return 'Schema';
            }
        },
        description: { type: 'string' },
        format: { type: 'string' },
        default: null,
        nullable: { type: 'boolean' },
        readOnly: { type: 'boolean' },
        writeOnly: { type: 'boolean' },
        xml: 'Xml',
        example: { isExample: true },
        deprecated: { type: 'boolean' },
    },
};
const Xml = {
    properties: {
        name: { type: 'string' },
        namespace: { type: 'string' },
        prefix: { type: 'string' },
        attribute: { type: 'boolean' },
        wrapped: { type: 'boolean' },
    },
};
const SchemaProperties = {
    properties: {},
    additionalProperties: 'Schema',
};
const DiscriminatorMapping = {
    properties: {},
    additionalProperties: (value) => {
        if (ref_utils_1.isMappingRef(value)) {
            return { type: 'string', directResolveAs: 'Schema' };
        }
        else {
            return { type: 'string' };
        }
    },
};
const Discriminator = {
    properties: {
        propertyName: { type: 'string' },
        mapping: 'DiscriminatorMapping',
    },
    required: ['propertyName'],
};
const Components = {
    properties: {
        parameters: 'NamedParameters',
        schemas: 'NamedSchemas',
        responses: 'NamedResponses',
        examples: 'NamedExamples',
        requestBodies: 'NamedRequestBodies',
        headers: 'NamedHeaders',
        securitySchemes: 'NamedSecuritySchemes',
        links: 'NamedLinks',
        callbacks: 'NamedCallbacks',
    },
};
const ImplicitFlow = {
    properties: {
        refreshUrl: { type: 'string' },
        scopes: { type: 'object', additionalProperties: { type: 'string' } },
        authorizationUrl: { type: 'string' },
    },
    required: ['authorizationUrl', 'scopes']
};
const PasswordFlow = {
    properties: {
        refreshUrl: { type: 'string' },
        scopes: { type: 'object', additionalProperties: { type: 'string' } },
        tokenUrl: { type: 'string' },
    },
    required: ['tokenUrl', 'scopes'],
};
const ClientCredentials = {
    properties: {
        refreshUrl: { type: 'string' },
        scopes: { type: 'object', additionalProperties: { type: 'string' } },
        tokenUrl: { type: 'string' },
    },
    required: ['tokenUrl', 'scopes'],
};
const AuthorizationCode = {
    properties: {
        refreshUrl: { type: 'string' },
        authorizationUrl: { type: 'string' },
        scopes: { type: 'object', additionalProperties: { type: 'string' } },
        tokenUrl: { type: 'string' },
    },
    required: ['authorizationUrl', 'tokenUrl', 'scopes'],
};
const SecuritySchemeFlows = {
    properties: {
        implicit: 'ImplicitFlow',
        password: 'PasswordFlow',
        clientCredentials: 'ClientCredentials',
        authorizationCode: 'AuthorizationCode',
    },
};
const SecurityScheme = {
    properties: {
        type: { enum: ['apiKey', 'http', 'oauth2', 'openIdConnect'] },
        description: { type: 'string' },
        name: { type: 'string' },
        in: { type: 'string', enum: ['query', 'header', 'cookie'] },
        scheme: { type: 'string' },
        bearerFormat: { type: 'string' },
        flows: 'SecuritySchemeFlows',
        openIdConnectUrl: { type: 'string' },
    },
    required(value) {
        switch (value === null || value === void 0 ? void 0 : value.type) {
            case 'apiKey':
                return ['type', 'name', 'in'];
            case 'http':
                return ['type', 'scheme'];
            case 'oauth2':
                return ['type', 'flows'];
            case 'openIdConnect':
                return ['type', 'openIdConnectUrl'];
            default:
                return ['type'];
        }
    },
    allowed(value) {
        switch (value === null || value === void 0 ? void 0 : value.type) {
            case 'apiKey':
                return ['type', 'name', 'in', 'description'];
            case 'http':
                return ['type', 'scheme', 'bearerFormat', 'description'];
            case 'oauth2':
                return ['type', 'flows', 'description'];
            case 'openIdConnect':
                return ['type', 'openIdConnectUrl', 'description'];
            default:
                return ['type', 'description'];
        }
    },
    extensionsPrefix: 'x-',
};
exports.Oas3Types = {
    DefinitionRoot,
    Tag,
    ExternalDocs,
    Server,
    ServerVariable,
    SecurityRequirement,
    Info,
    Contact,
    License,
    PathMap,
    PathItem,
    Parameter,
    Operation,
    Callback,
    RequestBody,
    MediaTypeMap,
    MediaType,
    Example,
    Encoding,
    Header,
    ResponsesMap,
    Response,
    Link,
    Schema,
    Xml,
    SchemaProperties,
    DiscriminatorMapping,
    Discriminator,
    Components,
    NamedSchemas: _1.mapOf('Schema'),
    NamedResponses: _1.mapOf('Response'),
    NamedParameters: _1.mapOf('Parameter'),
    NamedExamples: _1.mapOf('Example'),
    NamedRequestBodies: _1.mapOf('RequestBody'),
    NamedHeaders: _1.mapOf('Header'),
    NamedSecuritySchemes: _1.mapOf('SecurityScheme'),
    NamedLinks: _1.mapOf('Link'),
    NamedCallbacks: _1.mapOf('PathItem'),
    ImplicitFlow,
    PasswordFlow,
    ClientCredentials,
    AuthorizationCode,
    SecuritySchemeFlows,
    SecurityScheme,
    XCodeSample,
    WebhooksMap,
};
