/*
 Copyright 2019 The Tekton Authors

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
*/

package main

import (
	"bytes"
	"encoding/json"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"reflect"
	"testing"

	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"
	"github.com/google/go-github/v29/github"
)

func TestInvalidGitHubToken(t *testing.T) {
	body := marshalEvent(t, makeCommentEvent("created", "/mario unknown"))
	r := createRequest("POST", "/", "issue_comment", body)
	// github.ValidatePayload only checks the header if the secret is not empty.
	h := makeMarioHandler("secret", defaultRegistry)
	w := httptest.NewRecorder()

	h(w, r)

	assertBadRequestResponse(t, w, "missing signature")
}

func TestInvalidHookEventType(t *testing.T) {
	body := marshalEvent(t, nil)
	r := createRequest("POST", "/", "not_a_real_event_type", body)
	h := makeMarioHandler("", defaultRegistry)
	w := httptest.NewRecorder()

	h(w, r)

	assertBadRequestResponse(t, w, "unknown X-Github-Event in message: not_a_real_event_type")
}

func TestCommentActions(t *testing.T) {
	for _, action := range []string{"edited", "deleted"} {
		body := marshalEvent(t, makeCommentEvent(action, "/mario unknown"))
		r := createRequest("POST", "/", "issue_comment", body)
		h := makeMarioHandler("", defaultRegistry)
		w := httptest.NewRecorder()

		h(w, r)

		assertBadRequestResponse(t, w, "only new comments are supported")
	}
}

func TestUnknownCommand(t *testing.T) {
	body := marshalEvent(t, makeCommentEvent("created", "/mario unknown"))
	r := createRequest("POST", "/", "issue_comment", body)
	h := makeMarioHandler("", defaultRegistry)
	w := httptest.NewRecorder()

	h(w, r)

	assertBadRequestResponse(t, w, "unknown Mario command")
}

func TestBuildCommand(t *testing.T) {
	body := marshalEvent(t, makeCommentEvent("created", "/mario build ctx myimage"))
	r := createRequest("POST", "/", "issue_comment", body)
	h := makeMarioHandler("", defaultRegistry)
	w := httptest.NewRecorder()

	h(w, r)

	want := triggerPayload{
		BuildUUID:     "3c949657-72b4-442c-ba2a-e730e89e3ed5",
		GitRepository: "github.com/tektoncd/plumbing",
		GitRevision:   "pull/20/head",
		ContextPath:   "ctx",
		TargetImage:   defaultRegistry + "/myimage",
		PullRequestID: "20",
	}

	resp := w.Result()
	assertResponsePayload(t, resp, &want, cmpopts.IgnoreFields(want, "BuildUUID"))
}

// creates a GitHub hook type request - no secret is provided in testing.
func createRequest(method, url, event string, body []byte, opts ...requestOption) *http.Request {
	req := httptest.NewRequest(method, url, bytes.NewBuffer(body))
	req.Header.Set("Content-Type", "application/json")
	req.Header.Set("X-Github-Event", event)
	req.Header.Set("X-Github-Delivery", "testing-123")
	for _, o := range opts {
		o(req)
	}
	return req
}

type requestOption func(*http.Request)

func marshalEvent(t *testing.T, evt interface{}) []byte {
	t.Helper()
	b, err := json.Marshal(evt)
	if err != nil {
		t.Fatal(err)
	}
	return b
}

func makeCommentEvent(action, body string) *github.IssueCommentEvent {
	return &github.IssueCommentEvent{
		Action: github.String(action),
		Comment: &github.IssueComment{
			Body: github.String(body),
		},
		Issue: &github.Issue{
			Number: github.Int(20),
		},
		Repo: &github.Repository{
			FullName: github.String("tektoncd/plumbing"),
		},
	}
}

func assertResponseStatus(t *testing.T, resp *http.Response, want int) {
	t.Helper()
	if resp.StatusCode != want {
		t.Fatalf("incorrect response: got %v, want %v", resp.StatusCode, want)
	}
}

func assertResponsePayload(t *testing.T, resp *http.Response, v interface{}, opts ...cmp.Option) {
	t.Helper()
	body, err := ioutil.ReadAll(resp.Body)
	defer resp.Body.Close()
	if err != nil {
		t.Fatal(err)
	}
	// This assumes that v is a pointer to a type, and unmarshals to a new value
	// of that type for the purposes of comparison.
	objType := reflect.TypeOf(v).Elem()
	obj := reflect.New(objType).Interface()
	err = json.Unmarshal(body, &obj)
	if err != nil {
		t.Fatal(err)
	}
	if diff := cmp.Diff(obj, v, opts...); diff != "" {
		t.Fatalf("compare failed: %s\n", diff)
	}
}

func assertBadRequestResponse(t *testing.T, rr *httptest.ResponseRecorder, s string) {
	resp := rr.Result()
	assertResponseStatus(t, resp, http.StatusBadRequest)
	assertResponsePayload(t, resp, makeErrorResponse(s))
}

func makeErrorResponse(s string) *triggerErrorPayload {
	return &triggerErrorPayload{Error: s}
}
