---
title: Docker
---

Since [v0.31.0](https://github.com/goreleaser/goreleaser/releases/tag/v0.31.0),
GoReleaser supports building and pushing Docker images.

## How it works

You can declare multiple Docker images. They will be matched against
the binaries generated by your `builds` section.

If you have only one `build` setup,
the configuration is as easy as adding the
name of your image to your `.goreleaser.yml` file:

The docker image declaration supports templating. Learn more about the [name template engine](/customization/templates).

```yaml
dockers:
  - image_templates:
    - user/repo
```

You also need to create a `Dockerfile` in your project's root folder:

```dockerfile
FROM scratch
COPY mybin /
ENTRYPOINT ["/mybin"]
```

This configuration will build and push a Docker image named `user/repo:tagname`.

!!! info
    Note that were are not building any go files in the docker
    build phase, we are merely copying the binary to a `scratch` image and
    setting up the entrypoint.

## Customization

Of course, you can customize a lot of things:

```yaml
# .goreleaser.yml
dockers:
  # You can have multiple Docker images.
  -
    # GOOS of the built binary that should be used.
    goos: linux

    # GOARCH of the built binary that should be used.
    goarch: amd64

    # GOARM of the built binary that should be used.
    goarm: ''

    # Name templates of the built binaries that should be used.
    binaries:
    - mybinary

    # Build IDs to gather the binaries from.
    builds:
    - mybuild

    # Templates of the Docker image names.
    image_templates:
    - "myuser/myimage:latest"
    - "myuser/myimage:{{ .Tag }}"
    - "myuser/myimage:{{ .Tag }}-{{ .Env.GO_VERSION }}"
    - "myuser/myimage:v{{ .Major }}"
    - "gcr.io/myuser/myimage:latest"

    # Skips the docker push. Could be useful if you also do draft releases.
    # If set to auto, the release will not be pushed to the docker repository
    # in case there is an indicator for prerelease in the tag e.g. v1.0.0-rc1
    # Defaults to false.
    skip_push: false

    # Path to the Dockerfile (from the project root).
    dockerfile: Dockerfile

    # Template of the docker build flags.
    build_flag_templates:
    - "--pull"
    - "--label=org.opencontainers.image.created={{.Date}}"
    - "--label=org.opencontainers.image.name={{.ProjectName}}"
    - "--label=org.opencontainers.image.revision={{.FullCommit}}"
    - "--label=org.opencontainers.image.version={{.Version}}"
    - "--build-arg=FOO={{.Env.Bar}}"

    # If your Dockerfile copies files other than the binary itself,
    # you should list them here as well.
    # Note that goreleaser will create the same structure inside the temporary
    # folder, so if you add `foo/bar.json` here, on your Dockerfile you can
    # `COPY foo/bar.json /whatever.json`.
    # Also note that the paths here are relative to the folder in which
    # goreleaser is being run.
    # This field does not support wildcards, you can add an entire folder here
    # and use wildcards when you `COPY`/`ADD` in your Dockerfile.
    extra_files:
    - config.yml
```

!!! tip
    Learn more about the [name template engine](/customization/templates).

These settings should allow you to generate multiple Docker images,
for example, using multiple `FROM` statements,
as well as generate one image for each binary in your project.

## Generic Image Names

Some users might want to keep their image name as generic as possible.
That can be accomplished simply by adding template language in the definition:

```yaml
# .goreleaser.yml
project: foo
dockers:
  -
    binaries:
    - mybinary
    image_templates:
    - "myuser/{{.ProjectName}}"
```

This will build and public the following images:

- `myuser/foo`

!!! tip
    Learn more about the [name template engine](/customization/templates).

## Keeping docker images updated for current major

Some users might want to when version to push docker tags `:v1`, `:v1.6`,
`:v1.6.4` and `:latest` when `v1.6.4` (for example) is built. That can be
accomplished by using multiple `image_templates`:

```yaml
# .goreleaser.yml
dockers:
  -
    binaries:
    - mybinary
    image_templates:
    - "myuser/myimage:{{ .Tag }}"
    - "myuser/myimage:v{{ .Major }}"
    - "myuser/myimage:v{{ .Major }}.{{ .Minor }}"
    - "myuser/myimage:latest"
```

This will build and publish the following images:

- `myuser/myimage:v1.6.4`
- `myuser/myimage:v1`
- `myuser/myimage:v1.6`
- `myuser/myimage:latest`

With these settings you can hopefully push several different docker images
with multiple tags.

!!! tip
    Learn more about the [name template engine](/customization/templates).

## Publishing to multiple docker registries

Some users might want to push images to multiple docker registries. That can be
accomplished by using multiple `image_templates`:

```yaml
# .goreleaser.yml
dockers:
  -
    binaries:
    - mybinary
    image_templates:
    - "docker.io/myuser/myimage:{{ .Tag }}"
    - "docker.io/myuser/myimage:latest"
    - "gcr.io/myuser/myimage:{{ .Tag }}"
    - "gcr.io/myuser/myimage:latest"
```

This will build and publish the following images to `docker.io` and `gcr.io`:

- `myuser/myimage:v1.6.4`
- `myuser/myimage:latest`

## Applying docker build flags

Build flags can be applied using `build_flag_templates`. The flags must be
valid docker build flags.

```yaml
# .goreleaser.yml
dockers:
  -
    binaries:
    - mybinary
    image_templates:
    - "myuser/myimage"
    build_flag_templates:
    - "--pull"
    - "--label=org.opencontainers.image.created={{.Date}}"
    - "--label=org.opencontainers.image.name={{.ProjectName}}"
    - "--label=org.opencontainers.image.revision={{.FullCommit}}"
    - "--label=org.opencontainers.image.version={{.Version}}"
```

This will execute the following command:

```bash
docker build -t myuser/myimage . \
  --pull \
  --label=org.opencontainers.image.created=2020-01-19T15:58:07Z" \
  --label=org.opencontainers.image.name=mybinary" \
  --label=org.opencontainers.image.revision=da39a3ee5e6b4b0d3255bfef95601890afd80709" \
  --label=org.opencontainers.image.version=1.6.4
```

!!! tip
    Learn more about the [name template engine](/customization/templates).
