package controllers

import (
	ctx "context"
	"flag"
	"github.com/go-logr/logr"
	"github.com/redhat-developer/service-binding-operator/apis"
	"github.com/redhat-developer/service-binding-operator/pkg/reconcile/pipeline"
	"github.com/redhat-developer/service-binding-operator/pkg/reconcile/pipeline/context"
	"k8s.io/apimachinery/pkg/api/errors"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/client-go/dynamic"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller"
	"sigs.k8s.io/controller-runtime/pkg/predicate"
)

var (
	MaxConcurrentReconciles int
)

func RegisterFlags(flags *flag.FlagSet) {
	flags.IntVar(&MaxConcurrentReconciles, "max-concurrent-reconciles", 1, "max-concurrent-reconciles is the maximum number of concurrent Reconciles which can be run. Defaults to 1.")
}

type BindingReconciler struct {
	client.Client
	Log       logr.Logger
	Scheme    *runtime.Scheme
	dynClient dynamic.Interface // kubernetes dynamic api client

	pipeline pipeline.Pipeline

	PipelineProvider func(dynamic.Interface, context.K8STypeLookup) pipeline.Pipeline

	ReconcilingObject func() apis.Object
}

// SetupWithManager sets up the controller with the Manager.
func (r *BindingReconciler) SetupWithManager(mgr ctrl.Manager) error {
	client, err := dynamic.NewForConfig(mgr.GetConfig())
	if err != nil {
		return err
	}

	r.dynClient = client

	r.pipeline = r.PipelineProvider(r.dynClient, context.ResourceLookup(mgr.GetRESTMapper()))

	return ctrl.NewControllerManagedBy(mgr).
		For(r.ReconcilingObject()).
		WithEventFilter(predicate.GenerationChangedPredicate{}).
		WithOptions(controller.Options{MaxConcurrentReconciles: MaxConcurrentReconciles}).
		Complete(r)
}

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the ServiceBinding object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.7.0/pkg/reconcile
func (r *BindingReconciler) Reconcile(req ctrl.Request) (ctrl.Result, error) {
	log := r.Log.WithValues("serviceBinding", req.NamespacedName)
	var ctx = ctx.Background()
	serviceBinding := r.ReconcilingObject()

	err := r.Get(ctx, req.NamespacedName, serviceBinding)
	if err != nil {
		if errors.IsNotFound(err) {
			// Request object not found, could have been deleted after reconcile request.
			// Owned objects are automatically garbage collected. For additional cleanup logic use finalizers.
			// Return and don't requeue
			log.Info("ServiceBinding resource not found. Ignoring since object must be deleted", "name", req.NamespacedName, "err", err)
			return ctrl.Result{}, nil
		}
		// Error reading the object - requeue the request.
		log.Error(err, "Failed to get ServiceBinding", "name", req.NamespacedName, "err", err)
		return ctrl.Result{}, err
	}
	if !serviceBinding.HasDeletionTimestamp() && apis.MaybeAddFinalizer(serviceBinding) {
		if err = r.Update(ctx, serviceBinding); err != nil {
			return ctrl.Result{}, err
		}
	}
	if !serviceBinding.HasDeletionTimestamp() {
		log.Info("Reconciling")
	} else {
		log.Info("Deleted, unbind the application")
	}
	retry, err := r.pipeline.Process(serviceBinding)
	if !retry && err == nil {
		if serviceBinding.HasDeletionTimestamp() {
			if apis.MaybeRemoveFinalizer(serviceBinding) {
				if err = r.Update(ctx, serviceBinding); err != nil {
					return ctrl.Result{}, err
				}
			}
		}
	}
	result := ctrl.Result{Requeue: retry}
	log.Info("Done", "retry", retry, "error", err)
	if retry {
		return result, err
	}
	return result, nil
}
