"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getManifest = void 0;
const registry_call_1 = require("./registry-call");
const contentTypes = require("./content-types");
const needle_1 = require("./needle");
/**
 * A request to get image manifest by tag or digest
 * @param registryBase the hostname (and optionally base path) from which to get the manifest.
 * @param repo the requested image repository.
 * @param tag the manifest tag or digest value
 * @param username optional, username to be used for basic auth.
 * @param password optional, password to be used for basic auth.
 * @param options optional, request options.
 * @param platform optional, the manifest platform (os and architecture), applicable only for multi-architecture manifests.
 *                 Defaults to linux/amd64. In case the requested manifest platform is not found an error is thrown.
 * @returns
 */
async function getManifest(registryBase, repo, tag, username, password, options = {}, platform) {
    let accept = `${contentTypes.MANIFEST_V2}`;
    if (shouldAcceptManifestList(tag, platform)) {
        accept += `, ${contentTypes.MANIFEST_LIST_V2}`;
    }
    const endpoint = `/${repo}/manifests/${tag}`;
    const manifestResponse = await registry_call_1.registryV2Call(registryBase, endpoint, accept, username, password, options);
    const contentType = manifestResponse.headers["content-type"];
    if (contentType === contentTypes.MANIFEST_V2) {
        return needle_1.parseResponseBody(manifestResponse);
    }
    if (contentType === contentTypes.MANIFEST_LIST_V2) {
        const manifestDigest = getManifestDigestByPlatform(manifestResponse, platform);
        // calling getManifest recursively is safe here, since we got a list of actual manifests (and not a list of manifest-lists)
        return getManifest(registryBase, repo, manifestDigest, username, password, platform, options);
    }
    // content types that are different from contentTypes.MANIFEST_V2 or contentTypes.MANIFEST_LIST_V2 are
    // not supported (e.g. application/vnd.docker.distribution.manifest.v1+json)
    throw new Error(`unsupported content-type: ${manifestResponse.headers["content-type"]}`);
}
exports.getManifest = getManifest;
function getManifestDigestByPlatform(manifestResponse, platform) {
    const defaultPlatform = {
        os: "linux",
        architecture: "amd64",
    };
    const platformFilter = platform ? platform : defaultPlatform;
    const manifestList = needle_1.parseResponseBody(manifestResponse);
    const manifestPlatform = getManifestByOsAndArch(manifestList.manifests, platformFilter.os, platformFilter.architecture);
    if (manifestPlatform) {
        return manifestPlatform.digest;
    }
    throw new Error(`no supported manifest found for platform: ${JSON.stringify(platformFilter)}`);
}
function getManifestByOsAndArch(platformManifest, os, architecture) {
    return platformManifest.find(manifest => manifest.platform.os === os &&
        manifest.platform.architecture === architecture);
}
// Accept manifest list content type in case that:
// - Manifest digest was requested - if it's a manifest list digest, a list is returned and some CRs are strict
//   about the returned content type being different from the requested one (gcr)
// - Specific platform was requested, therefore we need to get the list in order to find the specific platform
//   (by default the CR will return linux/amd64)
function shouldAcceptManifestList(tag, platform) {
    return tag.includes(":") || !!platform;
}
//# sourceMappingURL=get-manifest.js.map