"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.remoteBundleFactory = exports.uploadRemoteBundle = void 0;
/* eslint-disable no-await-in-loop */
const lodash_chunk_1 = __importDefault(require("lodash.chunk"));
const files_1 = require("./files");
const http_1 = require("./http");
const constants_1 = require("./constants");
const emitter_1 = __importDefault(require("./emitter"));
const utils_1 = require("./lib/utils");
async function* prepareRemoteBundle(baseURL, sessionToken, files, removedFiles = [], existingBundleId = null, maxPayload = constants_1.MAX_PAYLOAD, source) {
    let response;
    let bundleId = existingBundleId;
    const fileChunks = lodash_chunk_1.default(files, maxPayload / 300);
    emitter_1.default.createBundleProgress(0, fileChunks.length);
    for (const [i, chunkedFiles] of fileChunks.entries()) {
        const paramFiles = utils_1.fromEntries(chunkedFiles.map(d => [d.bundlePath, d.hash]));
        if (bundleId === null) {
            // eslint-disable-next-line no-await-in-loop
            response = await http_1.createBundle({
                baseURL,
                sessionToken,
                files: paramFiles,
                source,
            });
        }
        else {
            // eslint-disable-next-line no-await-in-loop
            response = await http_1.extendBundle({
                baseURL,
                sessionToken,
                bundleId,
                files: paramFiles,
                removedFiles,
            });
        }
        emitter_1.default.createBundleProgress(i + 1, fileChunks.length);
        if (response.type === 'error') {
            // TODO: process Error
            yield response;
            break;
        }
        bundleId = response.value.bundleId;
        yield response;
    }
}
/**
 * Splits files in buckets and upload in parallel
 * @param baseURL
 * @param sessionToken
 * @param remoteBundle
 */
async function uploadRemoteBundle(baseURL, sessionToken, bundleId, files, maxPayload = constants_1.MAX_PAYLOAD) {
    let uploadedFiles = 0;
    emitter_1.default.uploadBundleProgress(0, files.length);
    const uploadFileChunks = async (bucketFiles) => {
        const resp = await http_1.uploadFiles({
            baseURL,
            sessionToken,
            bundleId,
            content: bucketFiles.map(f => {
                return { fileHash: f.hash, fileContent: f.content || '' };
            }),
        });
        if (resp.type !== 'error') {
            uploadedFiles += bucketFiles.length;
            emitter_1.default.uploadBundleProgress(uploadedFiles, files.length);
            return true;
        }
        return false;
    };
    const tasks = [];
    for (const bucketFiles of files_1.composeFilePayloads(files, maxPayload)) {
        tasks.push(uploadFileChunks(bucketFiles));
    }
    if (tasks.length) {
        return (await Promise.all(tasks)).some(r => !!r);
    }
    return true;
}
exports.uploadRemoteBundle = uploadRemoteBundle;
async function fullfillRemoteBundle(baseURL, sessionToken, baseDir, remoteBundle, maxPayload = constants_1.MAX_PAYLOAD, maxAttempts = constants_1.MAX_UPLOAD_ATTEMPTS) {
    // Fulfill remote bundle by uploading only missing files (splitted in chunks)
    // Check remove bundle to make sure no missing files left
    let attempts = 0;
    while (remoteBundle.missingFiles.length && attempts < maxAttempts) {
        const missingFiles = await files_1.resolveBundleFiles(baseDir, remoteBundle.missingFiles);
        const isUploaded = await uploadRemoteBundle(baseURL, sessionToken, remoteBundle.bundleId, missingFiles, maxPayload);
        if (!isUploaded) {
            throw new Error('Failed to upload some files');
        }
        const bundleResponse = await http_1.checkBundle({
            baseURL,
            sessionToken,
            bundleId: remoteBundle.bundleId,
        });
        if (bundleResponse.type === 'error') {
            throw new Error('Failed to get remote bundle');
        }
        // eslint-disable-next-line no-param-reassign
        remoteBundle = bundleResponse.value;
        attempts += 1;
    }
    return remoteBundle;
}
async function remoteBundleFactory(baseURL, sessionToken, files, removedFiles = [], baseDir, existingBundleId = null, maxPayload = constants_1.MAX_PAYLOAD, source) {
    const bundleFactory = prepareRemoteBundle(baseURL, sessionToken, files, removedFiles, existingBundleId, maxPayload, source);
    let remoteBundle = null;
    for await (const response of bundleFactory) {
        if (response.type === 'error') {
            throw response.error;
        }
        remoteBundle = await fullfillRemoteBundle(baseURL, sessionToken, baseDir, response.value, maxPayload);
        if (remoteBundle.missingFiles.length) {
            throw new Error(`Failed to upload # files: ${remoteBundle.missingFiles.length}`);
        }
    }
    return remoteBundle;
}
exports.remoteBundleFactory = remoteBundleFactory;
//# sourceMappingURL=bundles.js.map