"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getUserInstructionLayersFromConfig = exports.getDetectedLayersInfoFromConfig = exports.getPlatformFromConfig = exports.getRootFsLayersFromConfig = exports.getImageIdFromManifest = exports.getManifestLayers = exports.extractArchive = void 0;
const path_1 = require("path");
const instruction_parser_1 = require("../../dockerfile/instruction-parser");
const types_1 = require("../../types");
var layer_1 = require("./layer");
Object.defineProperty(exports, "extractArchive", { enumerable: true, get: function () { return layer_1.extractArchive; } });
function getManifestLayers(manifest) {
    return manifest.Layers.map((layer) => path_1.normalize(layer));
}
exports.getManifestLayers = getManifestLayers;
function getImageIdFromManifest(manifest) {
    try {
        const imageId = manifest.Config.split(".")[0];
        if (imageId.includes(":")) {
            // imageId includes the algorithm prefix
            return imageId;
        }
        return `${types_1.HashAlgorithm.Sha256}:${imageId}`;
    }
    catch (err) {
        throw new Error("Failed to extract image ID from archive manifest");
    }
}
exports.getImageIdFromManifest = getImageIdFromManifest;
function getRootFsLayersFromConfig(imageConfig) {
    try {
        return imageConfig.rootfs.diff_ids;
    }
    catch (err) {
        throw new Error("Failed to extract rootfs array from image config");
    }
}
exports.getRootFsLayersFromConfig = getRootFsLayersFromConfig;
function getPlatformFromConfig(imageConfig) {
    return imageConfig.os && imageConfig.architecture
        ? `${imageConfig.os}/${imageConfig.architecture}`
        : undefined;
}
exports.getPlatformFromConfig = getPlatformFromConfig;
function getDetectedLayersInfoFromConfig(imageConfig) {
    const runInstructions = getUserInstructionLayersFromConfig(imageConfig)
        .filter((instruction) => !instruction.empty_layer && instruction.created_by)
        .map((instruction) => instruction.created_by.replace("# buildkit", ""));
    const dockerfilePackages = instruction_parser_1.getPackagesFromRunInstructions(runInstructions);
    const dockerfileLayers = instruction_parser_1.getLayersFromPackages(dockerfilePackages);
    return { dockerfilePackages, dockerfileLayers };
}
exports.getDetectedLayersInfoFromConfig = getDetectedLayersInfoFromConfig;
function getUserInstructionLayersFromConfig(imageConfig) {
    const diffInHours = (d1, d2) => Math.abs(d1 - d2) / 1000 / (60 * 60);
    const maxDiffInHours = 5;
    const history = imageConfig.history;
    if (!history) {
        return [];
    }
    const lastInstructionTime = new Date(history.slice(-1)[0].created);
    const userInstructionLayers = history.filter((layer) => {
        return (diffInHours(new Date(layer.created), lastInstructionTime) <=
            maxDiffInHours);
    });
    // should only happen if there are no layers created by user instructions
    if (userInstructionLayers.length === history.length) {
        return [];
    }
    return userInstructionLayers;
}
exports.getUserInstructionLayersFromConfig = getUserInstructionLayersFromConfig;
//# sourceMappingURL=index.js.map