"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.removeFileContent = exports.test = void 0;
const detect_1 = require("../../../../lib/detect");
const types_1 = require("./types");
const analytics_1 = require("./analytics");
const measurable_methods_1 = require("./measurable-methods");
const feature_flags_1 = require("../../../../lib/feature-flags");
const assert_iac_options_flag_1 = require("./assert-iac-options-flag");
const config = require("../../../../lib/config");
// this method executes the local processing engine and then formats the results to adapt with the CLI output.
// this flow is the default GA flow for IAC scanning.
async function test(pathToScan, options) {
    var _a;
    try {
        const org = (_a = options.org) !== null && _a !== void 0 ? _a : config.org;
        const iacOrgSettings = await measurable_methods_1.getIacOrgSettings(org);
        const customRulesPath = await customRulesPathForOrg(options.rules, org);
        await measurable_methods_1.initLocalCache({ customRulesPath });
        const filesToParse = await measurable_methods_1.loadFiles(pathToScan, options);
        const { parsedFiles, failedFiles } = await measurable_methods_1.parseFiles(filesToParse, options);
        // Duplicate all the files and run them through the custom engine.
        if (customRulesPath) {
            parsedFiles.push(...parsedFiles.map((file) => (Object.assign(Object.assign({}, file), { engineType: types_1.EngineType.Custom }))));
        }
        const scannedFiles = await measurable_methods_1.scanFiles(parsedFiles);
        const resultsWithCustomSeverities = await measurable_methods_1.applyCustomSeverities(scannedFiles, iacOrgSettings.customPolicies);
        const formattedResults = measurable_methods_1.formatScanResults(resultsWithCustomSeverities, options, iacOrgSettings.meta);
        analytics_1.addIacAnalytics(formattedResults);
        // TODO: add support for proper typing of old TestResult interface.
        return {
            results: formattedResults,
            // NOTE: No file or parsed file data should leave this function.
            failures: detect_1.isLocalFolder(pathToScan)
                ? failedFiles.map(removeFileContent)
                : undefined,
        };
    }
    finally {
        measurable_methods_1.cleanLocalCache();
    }
}
exports.test = test;
async function customRulesPathForOrg(customRulesPath, publicOrgId) {
    if (!customRulesPath)
        return;
    const isCustomRulesSupported = (await feature_flags_1.isFeatureFlagSupportedForOrg('iacCustomRules', publicOrgId)).ok ===
        true;
    if (isCustomRulesSupported) {
        return customRulesPath;
    }
    throw new assert_iac_options_flag_1.FlagError('rules');
}
function removeFileContent({ filePath, fileType, failureReason, projectType, }) {
    return {
        filePath,
        fileType,
        failureReason,
        projectType,
    };
}
exports.removeFileContent = removeFileContent;
//# sourceMappingURL=index.js.map