import { StringType } from './types';
import { ColorProperty } from 'csstype';
/**
 * Creates a color from a hex color code or named color.
 * e.g. color('red') or color('#FF0000') or color('#F00'))
 */
export declare function color(value: ColorProperty): ColorHelper;
/**
 * Creates a color from hue, saturation, and lightness.  Alpha is automatically set to 100%
 */
export declare function hsl(hue: number, saturation: string | number, lightness: string | number): ColorHelper;
/**
 * Creates a color from hue, saturation, lightness, and alpha
 */
export declare function hsla(hue: number, saturation: string | number, lightness: string | number, opacity: string | number): ColorHelper;
/**
 * Creates a color form the red, blue, and green color space.  Alpha is automatically set to 100%
 */
export declare function rgb(red: number, blue: number, green: number): ColorHelper;
/**
 * Creates a color form the red, blue, green, and alpha in the color space
 */
export declare function rgba(red: number, blue: number, green: number, alpha: string | number): ColorHelper;
/**
 * A CSS Color.  Includes utilities for converting between color types
 */
export declare class ColorHelper implements StringType<ColorProperty> {
    /**
     * Format of the color
     * @private
     */
    private f;
    /**
     * True if the color should output opacity in the formatted result
     * @private
     */
    private o;
    /**
     * Channel 0
     * @private
     */
    private r;
    /**
     * Channel 1
     * @private
     */
    private g;
    /**
     * Channel 2
     * @private
     */
    private b;
    /**
     * Channel Alpha
     * @private
     */
    private a;
    constructor(format: 'rgb' | 'hsl', r: number, g: number, b: number, a: number, hasAlpha: boolean);
    /**
     * Converts the stored color into string form (which is used by Free Style)
     */
    toString(): ColorProperty;
    /**
     * Converts to hex rgb(255, 255, 255) to #FFFFFF
     */
    toHexString(): string;
    /**
     * Converts to the Hue, Saturation, Lightness color space
     */
    toHSL(): ColorHelper;
    /**
     * Converts to the Hue, Saturation, Lightness color space and adds an alpha channel
     */
    toHSLA(): ColorHelper;
    /**
     * Converts to the Red, Green, Blue color space
     */
    toRGB(): ColorHelper;
    /**
     * Converts to the Red, Green, Blue color space and adds an alpha channel
     */
    toRGBA(): ColorHelper;
    red(): number;
    green(): number;
    blue(): number;
    hue(): number;
    saturation(): number;
    lightness(): number;
    alpha(): number;
    opacity(): number;
    invert(): ColorHelper;
    lighten(percent: string | number, relative?: boolean): ColorHelper;
    darken(percent: string | number, relative?: boolean): ColorHelper;
    saturate(percent: string | number, relative?: boolean): ColorHelper;
    desaturate(percent: string | number, relative?: boolean): ColorHelper;
    grayscale(): ColorHelper;
    fade(percent: string | number): ColorHelper;
    fadeOut(percent: string | number, relative?: boolean): ColorHelper;
    fadeIn(percent: string | number, relative?: boolean): ColorHelper;
    mix(mixin: string | ColorHelper, weight?: number): ColorHelper;
    tint(weight: number): ColorHelper;
    shade(weight: number): ColorHelper;
    spin(degrees: number): ColorHelper;
}
