"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.analyze = void 0;
const Debug = require("debug");
const archiveExtractor = require("../extractor");
const go_parser_1 = require("../go-parser");
const inputs_1 = require("../inputs");
const static_1 = require("../inputs/apk/static");
const static_2 = require("../inputs/apt/static");
const static_3 = require("../inputs/binaries/static");
const static_4 = require("../inputs/distroless/static");
const filePatternStatic = require("../inputs/file-pattern/static");
const static_5 = require("../inputs/java/static");
const static_6 = require("../inputs/node/static");
const static_7 = require("../inputs/os-release/static");
const static_8 = require("../inputs/rpm/static");
const applications_1 = require("./applications");
const java_1 = require("./applications/java");
const osReleaseDetector = require("./os-release");
const apk_1 = require("./package-managers/apk");
const apt_1 = require("./package-managers/apt");
const rpm_1 = require("./package-managers/rpm");
const debug = Debug("snyk");
async function analyze(targetImage, dockerfileAnalysis, imageType, imagePath, globsToFind, appScan) {
    const staticAnalysisActions = [
        static_1.getApkDbFileContentAction,
        static_2.getDpkgFileContentAction,
        static_2.getExtFileContentAction,
        static_8.getRpmDbFileContentAction,
        ...static_7.getOsReleaseActions,
        static_3.getNodeBinariesFileContentAction,
        static_3.getOpenJDKBinariesFileContentAction,
        static_4.getDpkgPackageFileContentAction,
    ];
    const checkForGlobs = shouldCheckForGlobs(globsToFind);
    if (checkForGlobs) {
        staticAnalysisActions.push(filePatternStatic.generateExtractAction(globsToFind.include, globsToFind.exclude));
    }
    if (appScan) {
        staticAnalysisActions.push(...[
            static_6.getNodeAppFileContentAction,
            static_5.getJarFileContentAction,
            go_parser_1.getGoModulesContentAction,
        ]);
    }
    const { imageId, manifestLayers, extractedLayers, rootFsLayers, autoDetectedUserInstructions, platform, imageLabels, } = await archiveExtractor.extractImageContent(imageType, imagePath, staticAnalysisActions);
    const [apkDbFileContent, aptDbFileContent, rpmDbFileContent,] = await Promise.all([
        static_1.getApkDbFileContent(extractedLayers),
        static_2.getAptDbFileContent(extractedLayers),
        static_8.getRpmDbFileContent(extractedLayers),
    ]);
    const distrolessAptFiles = static_4.getAptFiles(extractedLayers);
    const manifestFiles = [];
    if (checkForGlobs) {
        const matchingFiles = filePatternStatic.getMatchingFiles(extractedLayers);
        manifestFiles.push(...matchingFiles);
    }
    let osRelease;
    try {
        osRelease = await osReleaseDetector.detectStatically(extractedLayers, dockerfileAnalysis);
    }
    catch (err) {
        debug(`Could not detect OS release: ${JSON.stringify(err)}`);
        throw new Error("Failed to detect OS release");
    }
    let results;
    try {
        results = await Promise.all([
            apk_1.analyze(targetImage, apkDbFileContent),
            apt_1.analyze(targetImage, aptDbFileContent),
            rpm_1.analyze(targetImage, rpmDbFileContent),
            apt_1.analyzeDistroless(targetImage, distrolessAptFiles),
        ]);
    }
    catch (err) {
        debug(`Could not detect installed OS packages: ${JSON.stringify(err)}`);
        throw new Error("Failed to detect installed OS packages");
    }
    const binaries = static_3.getBinariesHashes(extractedLayers);
    const applicationDependenciesScanResults = [];
    if (appScan) {
        const nodeDependenciesScanResults = await applications_1.nodeFilesToScannedProjects(inputs_1.getFileContent(extractedLayers, static_6.getNodeAppFileContentAction.actionName));
        const jarFingerprintScanResults = await java_1.jarFilesToScannedProjects(inputs_1.getFileContent(extractedLayers, static_5.getJarFileContentAction.actionName), targetImage);
        const goModulesScanResult = await go_parser_1.goModulesToScannedProjects(inputs_1.getElfFileContent(extractedLayers, go_parser_1.getGoModulesContentAction.actionName));
        applicationDependenciesScanResults.push(...nodeDependenciesScanResults, ...jarFingerprintScanResults, ...goModulesScanResult);
    }
    return {
        imageId,
        osRelease,
        platform,
        results,
        binaries,
        imageLayers: manifestLayers,
        rootFsLayers,
        applicationDependenciesScanResults,
        manifestFiles,
        autoDetectedUserInstructions,
        imageLabels,
    };
}
exports.analyze = analyze;
function shouldCheckForGlobs(globsToFind) {
    return globsToFind.include.length > 0;
}
//# sourceMappingURL=static-analyzer.js.map