"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.scan = void 0;
const fs = require("fs");
const hosted_git_info_1 = require("hosted-git-info");
const path = require("path");
const find_1 = require("./find");
const git_1 = require("./git");
const hash_1 = require("./hash");
const debug_1 = require("./debug");
async function scan(options) {
    try {
        debug_1.debug('options %o', options);
        if (!options.path) {
            throw 'invalid options no path provided.';
        }
        if (!fs.existsSync(options.path)) {
            throw `'${options.path}' does not exist.`;
        }
        const filePaths = await find_1.find(options.path);
        debug_1.debug('%d files found', filePaths.length);
        const fingerprints = [];
        for (const filePath of filePaths) {
            const md5 = await hash_1.hash(filePath);
            fingerprints.push({
                filePath,
                hash: md5,
            });
        }
        const facts = [{ type: 'cpp-fingerprints', data: fingerprints }];
        const target = await git_1.getTarget();
        debug_1.debug('target %o', target);
        const gitInfo = hosted_git_info_1.fromUrl(target.remoteUrl);
        const name = options.projectName || (gitInfo === null || gitInfo === void 0 ? void 0 : gitInfo.project) || path.basename(options.path);
        debug_1.debug('name %o', name);
        const scanResults = [
            {
                facts,
                identity: {
                    type: 'cpp',
                },
                name,
                target,
            },
        ];
        return {
            scanResults,
        };
    }
    catch (error) {
        throw new Error(`Could not scan C/C++ project, ${error}`);
    }
}
exports.scan = scan;
//# sourceMappingURL=scan.js.map