"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getLevel = exports.getResults = exports.getRules = exports.createSarifOutputForOpenSource = void 0;
const upperFirst = require('lodash.upperfirst');
const groupBy = require('lodash.groupby');
const map = require('lodash.map');
const legacy_1 = require("../../../lib/snyk-test/legacy");
const LOCK_FILES_TO_MANIFEST_MAP = {
    'Gemfile.lock': 'Gemfile',
    'package-lock.json': 'package.json',
    'yarn.lock': 'package.json',
    'Gopkg.lock': 'Gopkg.toml',
    'go.sum': 'go.mod',
    'composer.lock': 'composer.json',
    'Podfile.lock': 'Podfile',
    'poetry.lock': 'pyproject.toml',
};
function createSarifOutputForOpenSource(testResults) {
    return {
        $schema: 'https://raw.githubusercontent.com/oasis-tcs/sarif-spec/master/Schemata/sarif-schema-2.1.0.json',
        version: '2.1.0',
        runs: testResults.map(replaceLockfileWithManifest).map((testResult) => ({
            tool: {
                driver: {
                    name: 'Snyk Open Source',
                    rules: getRules(testResult),
                },
            },
            results: getResults(testResult),
        })),
    };
}
exports.createSarifOutputForOpenSource = createSarifOutputForOpenSource;
function replaceLockfileWithManifest(testResult) {
    let targetFile = testResult.displayTargetFile || '';
    for (const [key, replacer] of Object.entries(LOCK_FILES_TO_MANIFEST_MAP)) {
        targetFile = targetFile.replace(new RegExp(key, 'g'), replacer);
    }
    return Object.assign(Object.assign({}, testResult), { vulnerabilities: testResult.vulnerabilities || [], displayTargetFile: targetFile });
}
function getRules(testResult) {
    const groupedVulnerabilities = groupBy(testResult.vulnerabilities, 'id');
    return map(groupedVulnerabilities, ([vuln, ...moreVulns]) => {
        var _a, _b, _c;
        const cves = (_b = (_a = vuln.identifiers) === null || _a === void 0 ? void 0 : _a.CVE) === null || _b === void 0 ? void 0 : _b.join();
        return {
            id: vuln.id,
            shortDescription: {
                text: `${upperFirst(vuln.severity)} severity - ${vuln.title} vulnerability in ${vuln.packageName}`,
            },
            fullDescription: {
                text: cves
                    ? `(${cves}) ${vuln.name}@${vuln.version}`
                    : `${vuln.name}@${vuln.version}`,
            },
            help: {
                text: '',
                markdown: `* Package Manager: ${testResult.packageManager}
* ${vuln.type === 'license' ? 'Module' : 'Vulnerable module'}: ${vuln.name}
* Introduced through: ${getIntroducedThrough(vuln)}
#### Detailed paths
${[vuln, ...moreVulns]
                    .map((item) => `* _Introduced through_: ${item.from.join(' › ')}`)
                    .join('\n')}
${vuln.description}`.replace(/##\s/g, '# '),
            },
            properties: {
                tags: [
                    'security',
                    ...(((_c = vuln.identifiers) === null || _c === void 0 ? void 0 : _c.CWE) || []),
                    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                    testResult.packageManager,
                ],
            },
        };
    });
}
exports.getRules = getRules;
function getResults(testResult) {
    const groupedVulnerabilities = groupBy(testResult.vulnerabilities, 'id');
    return map(groupedVulnerabilities, ([vuln]) => ({
        ruleId: vuln.id,
        level: getLevel(vuln),
        message: {
            text: `This file introduces a vulnerable ${vuln.packageName} package with a ${vuln.severity} severity vulnerability.`,
        },
        locations: [
            {
                physicalLocation: {
                    artifactLocation: {
                        uri: testResult.displayTargetFile,
                    },
                    region: {
                        startLine: vuln.lineNumber || 1,
                    },
                },
            },
        ],
    }));
}
exports.getResults = getResults;
function getLevel(vuln) {
    switch (vuln.severity) {
        case legacy_1.SEVERITY.CRITICAL:
        case legacy_1.SEVERITY.HIGH:
            return 'error';
        case legacy_1.SEVERITY.MEDIUM:
            return 'warning';
        case legacy_1.SEVERITY.LOW:
        default:
            return 'note';
    }
}
exports.getLevel = getLevel;
function getIntroducedThrough(vuln) {
    const [firstFrom, secondFrom] = vuln.from || [];
    return vuln.from.length > 2
        ? `${firstFrom}, ${secondFrom} and others`
        : vuln.from.length === 2
            ? `${firstFrom} and ${secondFrom}`
            : firstFrom;
}
//# sourceMappingURL=open-source-sarif-output.js.map