#!/usr/bin/env node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("source-map-support/register");
const Debug = require("debug");
const pathLib = require("path");
// import args as a first internal module
const args_1 = require("./args");
// parse args as a first thing; argsLib modifies global namespace
// therefore it is better to do it as a first thing to prevent bugs
// when modules use this global setting during their require phase
// TODO(code): remove once https://app.stepsize.com/issue/c2f6253e-7240-436f-943c-23a897558156/2-http-libraries-in-cli is solved
const globalArgs = args_1.args(process.argv);
// assert supported node runtime version
const runtime = require("./runtime");
// require analytics as soon as possible to start measuring execution time
const analytics = require("../lib/analytics");
const alerts = require("../lib/alerts");
const sln = require("../lib/sln");
const copy_1 = require("./copy");
const spinner = require("../lib/spinner");
const errors = require("../lib/errors/legacy-errors");
const ansiEscapes = require("ansi-escapes");
const detect_1 = require("../lib/detect");
const updater_1 = require("../lib/updater");
const errors_1 = require("../lib/errors");
const strip_ansi_1 = require("strip-ansi");
const exclude_flag_invalid_input_1 = require("../lib/errors/exclude-flag-invalid-input");
const modes_1 = require("./modes");
const json_file_output_bad_input_error_1 = require("../lib/errors/json-file-output-bad-input-error");
const json_file_output_1 = require("../lib/json-file-output");
const empty_sarif_output_error_1 = require("../lib/errors/empty-sarif-output-error");
const invalid_detection_depth_value_1 = require("../lib/errors/invalid-detection-depth-value");
const debug = Debug('snyk');
const EXIT_CODES = {
    VULNS_FOUND: 1,
    ERROR: 2,
    NO_SUPPORTED_MANIFESTS_FOUND: 3,
};
async function runCommand(args) {
    const commandResult = await args.method(...args.options._);
    const res = analytics.addDataAndSend({
        args: args.options._,
        command: args.command,
        org: args.options.org,
    });
    if (!commandResult) {
        return;
    }
    const result = commandResult.toString();
    if (result && !args.options.quiet) {
        if (args.options.copy) {
            copy_1.copy(result);
            console.log('Result copied to clipboard');
        }
        else {
            console.log(result);
        }
    }
    // also save the json (in error.json) to file if option is set
    if (args.command === 'test') {
        const jsonResults = commandResult.getJsonResult();
        await saveResultsToFile(args.options, 'json', jsonResults);
        const sarifResults = commandResult.getSarifResult();
        await saveResultsToFile(args.options, 'sarif', sarifResults);
    }
    return res;
}
async function handleError(args, error) {
    var _a;
    spinner.clearAll();
    let command = 'bad-command';
    let exitCode = EXIT_CODES.ERROR;
    const noSupportedManifestsFound = (_a = error.message) === null || _a === void 0 ? void 0 : _a.includes('Could not detect supported target files in');
    if (noSupportedManifestsFound) {
        exitCode = EXIT_CODES.NO_SUPPORTED_MANIFESTS_FOUND;
    }
    const vulnsFound = error.code === 'VULNS';
    if (vulnsFound) {
        // this isn't a bad command, so we won't record it as such
        command = args.command;
        exitCode = EXIT_CODES.VULNS_FOUND;
    }
    if (args.options.debug && !args.options.json) {
        const output = vulnsFound ? error.message : error.stack;
        console.log(output);
    }
    else if (args.options.json &&
        !(error instanceof errors_1.UnsupportedOptionCombinationError)) {
        const output = vulnsFound
            ? error.message
            : strip_ansi_1.default(error.json || error.stack);
        console.log(output);
    }
    else {
        if (!args.options.quiet) {
            const result = errors.message(error);
            if (args.options.copy) {
                copy_1.copy(result);
                console.log('Result copied to clipboard');
            }
            else {
                if (`${error.code}`.indexOf('AUTH_') === 0) {
                    // remove the last few lines
                    const erase = ansiEscapes.eraseLines(4);
                    process.stdout.write(erase);
                }
                console.log(result);
            }
        }
    }
    await saveResultsToFile(args.options, 'json', error.jsonStringifiedResults);
    await saveResultsToFile(args.options, 'sarif', error.sarifStringifiedResults);
    const analyticsError = vulnsFound
        ? {
            stack: error.jsonNoVulns,
            code: error.code,
            message: 'Vulnerabilities found',
        }
        : {
            stack: error.stack,
            code: error.code,
            message: error.message,
        };
    if (!vulnsFound && !error.stack) {
        // log errors that are not error objects
        analytics.add('error', true);
        analytics.add('command', args.command);
    }
    else {
        analytics.add('error-message', analyticsError.message);
        // Note that error.stack would also contain the error message
        // (see https://nodejs.org/api/errors.html#errors_error_stack)
        analytics.add('error', analyticsError.stack);
        analytics.add('error-code', error.code);
        analytics.add('error-str-code', error.strCode);
        analytics.add('command', args.command);
    }
    const res = analytics.addDataAndSend({
        args: args.options._,
        command,
        org: args.options.org,
    });
    return { res, exitCode };
}
function getFullPath(filepathFragment) {
    if (pathLib.isAbsolute(filepathFragment)) {
        return filepathFragment;
    }
    else {
        const fullPath = pathLib.join(process.cwd(), filepathFragment);
        return fullPath;
    }
}
async function saveJsonResultsToFile(stringifiedJson, jsonOutputFile) {
    if (!jsonOutputFile) {
        console.error('empty jsonOutputFile');
        return;
    }
    if (jsonOutputFile.constructor.name !== String.name) {
        console.error('--json-output-file should be a filename path');
        return;
    }
    await json_file_output_1.saveJsonToFileCreatingDirectoryIfRequired(jsonOutputFile, stringifiedJson);
}
function checkRuntime() {
    if (!runtime.isSupported(process.versions.node)) {
        console.error(`Node.js version ${process.versions.node} is an unsupported Node.js ` +
            `runtime! Supported runtime range is '${runtime.supportedRange}'`);
        console.error('Please upgrade your Node.js runtime. The last version of Snyk CLI that supports Node.js v8 is v1.454.0.');
        process.exit(EXIT_CODES.ERROR);
    }
}
// Throw error if user specifies package file name as part of path,
// and if user specifies multiple paths and used project-name option.
function checkPaths(args) {
    let count = 0;
    for (const path of args.options._) {
        if (typeof path === 'string' && detect_1.isPathToPackageFile(path)) {
            throw errors_1.MissingTargetFileError(path);
        }
        else if (typeof path === 'string') {
            if (++count > 1 && args.options['project-name']) {
                throw new errors_1.UnsupportedOptionCombinationError([
                    'multiple paths',
                    'project-name',
                ]);
            }
        }
    }
}
async function main() {
    updater_1.updateCheck();
    checkRuntime();
    let res;
    let failed = false;
    let exitCode = EXIT_CODES.ERROR;
    try {
        modes_1.modeValidation(globalArgs);
        // TODO: fix this, we do transformation to options and teh type doesn't reflect it
        validateUnsupportedOptionCombinations(globalArgs.options);
        if (globalArgs.options['app-vulns'] && globalArgs.options['json']) {
            throw new errors_1.UnsupportedOptionCombinationError([
                'Application vulnerabilities is currently not supported with JSON output. ' +
                    'Please try using —app-vulns only to get application vulnerabilities, or ' +
                    '—json only to get your image vulnerabilties, excluding the application ones.',
            ]);
        }
        if (globalArgs.options['group-issues'] && globalArgs.options['iac']) {
            throw new errors_1.UnsupportedOptionCombinationError([
                '--group-issues is currently not supported for Snyk IaC.',
            ]);
        }
        if (globalArgs.options['group-issues'] &&
            !globalArgs.options['json'] &&
            !globalArgs.options['json-file-output']) {
            throw new errors_1.UnsupportedOptionCombinationError([
                'JSON output is required to use --group-issues, try adding --json.',
            ]);
        }
        if (globalArgs.options.file &&
            typeof globalArgs.options.file === 'string' &&
            globalArgs.options.file.match(/\.sln$/)) {
            if (globalArgs.options['project-name']) {
                throw new errors_1.UnsupportedOptionCombinationError([
                    'file=*.sln',
                    'project-name',
                ]);
            }
            sln.updateArgs(globalArgs);
        }
        else if (typeof globalArgs.options.file === 'boolean') {
            throw new errors_1.FileFlagBadInputError();
        }
        if (typeof globalArgs.options.detectionDepth !== 'undefined' &&
            (globalArgs.options.detectionDepth <= 0 ||
                Number.isNaN(globalArgs.options.detectionDepth))) {
            throw new invalid_detection_depth_value_1.InvalidDetectionDepthValue();
        }
        validateUnsupportedSarifCombinations(globalArgs);
        validateOutputFile(globalArgs.options, 'json', new json_file_output_bad_input_error_1.JsonFileOutputBadInputError());
        validateOutputFile(globalArgs.options, 'sarif', new empty_sarif_output_error_1.SarifFileOutputEmptyError());
        checkPaths(globalArgs);
        res = await runCommand(globalArgs);
    }
    catch (error) {
        failed = true;
        const response = await handleError(globalArgs, error);
        res = response.res;
        exitCode = response.exitCode;
    }
    if (!globalArgs.options.json) {
        console.log(alerts.displayAlerts());
    }
    if (!process.env.TAP && failed) {
        debug('Exit code: ' + exitCode);
        process.exitCode = exitCode;
    }
    return res;
}
const cli = main().catch((e) => {
    console.error('Something unexpected went wrong: ', e.stack);
    console.error('Exit code: ' + EXIT_CODES.ERROR);
    process.exit(EXIT_CODES.ERROR);
});
if (module.parent) {
    // eslint-disable-next-line id-blacklist
    module.exports = cli;
}
function validateUnsupportedOptionCombinations(options) {
    const unsupportedAllProjectsCombinations = {
        'project-name': 'project-name',
        file: 'file',
        yarnWorkspaces: 'yarn-workspaces',
        packageManager: 'package-manager',
        docker: 'docker',
        allSubProjects: 'all-sub-projects',
    };
    const unsupportedYarnWorkspacesCombinations = {
        'project-name': 'project-name',
        file: 'file',
        packageManager: 'package-manager',
        docker: 'docker',
        allSubProjects: 'all-sub-projects',
    };
    if (options.scanAllUnmanaged && options.file) {
        throw new errors_1.UnsupportedOptionCombinationError(['file', 'scan-all-unmanaged']);
    }
    if (options.allProjects) {
        for (const option in unsupportedAllProjectsCombinations) {
            if (options[option]) {
                throw new errors_1.UnsupportedOptionCombinationError([
                    unsupportedAllProjectsCombinations[option],
                    'all-projects',
                ]);
            }
        }
    }
    if (options.yarnWorkspaces) {
        for (const option in unsupportedYarnWorkspacesCombinations) {
            if (options[option]) {
                throw new errors_1.UnsupportedOptionCombinationError([
                    unsupportedAllProjectsCombinations[option],
                    'yarn-workspaces',
                ]);
            }
        }
    }
    if (options.exclude) {
        if (!(options.allProjects || options.yarnWorkspaces)) {
            throw new errors_1.OptionMissingErrorError('--exclude', [
                '--yarn-workspaces',
                '--all-projects',
            ]);
        }
        if (typeof options.exclude !== 'string') {
            throw new errors_1.ExcludeFlagBadInputError();
        }
        if (options.exclude.indexOf(pathLib.sep) > -1) {
            throw new exclude_flag_invalid_input_1.ExcludeFlagInvalidInputError();
        }
    }
}
function validateUnsupportedSarifCombinations(args) {
    if (args.options['json-file-output'] && args.command !== 'test') {
        throw new errors_1.UnsupportedOptionCombinationError([
            args.command,
            'json-file-output',
        ]);
    }
    if (args.options['sarif'] && args.command !== 'test') {
        throw new errors_1.UnsupportedOptionCombinationError([args.command, 'sarif']);
    }
    if (args.options['sarif'] && args.options['json']) {
        throw new errors_1.UnsupportedOptionCombinationError([
            args.command,
            'sarif',
            'json',
        ]);
    }
    if (args.options['sarif-file-output'] && args.command !== 'test') {
        throw new errors_1.UnsupportedOptionCombinationError([
            args.command,
            'sarif-file-output',
        ]);
    }
    if (args.options['sarif'] &&
        args.options['docker'] &&
        !args.options['file']) {
        throw new errors_1.OptionMissingErrorError('sarif', ['--file']);
    }
    if (args.options['sarif-file-output'] &&
        args.options['docker'] &&
        !args.options['file']) {
        throw new errors_1.OptionMissingErrorError('sarif-file-output', ['--file']);
    }
}
async function saveResultsToFile(options, outputType, jsonResults) {
    const flag = `${outputType}-file-output`;
    const outputFile = options[flag];
    if (outputFile && jsonResults) {
        const outputFileStr = outputFile;
        const fullOutputFilePath = getFullPath(outputFileStr);
        await saveJsonResultsToFile(strip_ansi_1.default(jsonResults), fullOutputFilePath);
    }
}
function validateOutputFile(options, outputType, error) {
    const fileOutputValue = options[`${outputType}-file-output`];
    if (fileOutputValue === undefined) {
        return;
    }
    if (!fileOutputValue || typeof fileOutputValue !== 'string') {
        throw error;
    }
    // On Windows, seems like quotes get passed in
    if (fileOutputValue === "''" || fileOutputValue === '""') {
        throw error;
    }
}
//# sourceMappingURL=index.js.map