"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFormattedMonitorOutput = exports.monitorEcosystem = void 0;
const chalk_1 = require("chalk");
const config = require("../config");
const is_ci_1 = require("../is-ci");
const promise_1 = require("../request/promise");
const spinner = require("../../lib/spinner");
const plugins_1 = require("./plugins");
const format_monitor_response_1 = require("../../cli/commands/monitor/formatters/format-monitor-response");
const get_extra_project_count_1 = require("../plugins/get-extra-project-count");
const errors_1 = require("../errors");
const policy_1 = require("./policy");
const api_token_1 = require("../api-token");
const SEPARATOR = '\n-------------------------------------------------------\n';
async function monitorEcosystem(ecosystem, paths, options) {
    const plugin = plugins_1.getPlugin(ecosystem);
    const scanResultsByPath = {};
    for (const path of paths) {
        try {
            await spinner(`Analyzing dependencies in ${path}`);
            options.path = path;
            const pluginResponse = await plugin.scan(options);
            scanResultsByPath[path] = pluginResponse.scanResults;
        }
        catch (error) {
            if (ecosystem === 'docker' &&
                error.statusCode === 401 &&
                error.message === 'authentication required') {
                throw new errors_1.DockerImageNotFoundError(path);
            }
            throw error;
        }
        finally {
            spinner.clearAll();
        }
    }
    const [monitorResults, errors] = await monitorDependencies(scanResultsByPath, options);
    return [monitorResults, errors];
}
exports.monitorEcosystem = monitorEcosystem;
async function generateMonitorDependenciesRequest(scanResult, options) {
    // WARNING! This mutates the payload. The project name logic should be handled in the plugin.
    scanResult.name =
        options['project-name'] || config.PROJECT_NAME || scanResult.name;
    // WARNING! This mutates the payload. Policy logic should be in the plugin.
    const policy = await policy_1.findAndLoadPolicyForScanResult(scanResult, options);
    if (policy !== undefined) {
        scanResult.policy = policy.toString();
    }
    return {
        scanResult,
        method: 'cli',
        projectName: options['project-name'] || config.PROJECT_NAME || undefined,
    };
}
async function monitorDependencies(scans, options) {
    const results = [];
    const errors = [];
    for (const [path, scanResults] of Object.entries(scans)) {
        await spinner(`Monitoring dependencies in ${path}`);
        for (const scanResult of scanResults) {
            const monitorDependenciesRequest = await generateMonitorDependenciesRequest(scanResult, options);
            const configOrg = config.org ? decodeURIComponent(config.org) : undefined;
            const payload = {
                method: 'PUT',
                url: `${config.API}/monitor-dependencies`,
                json: true,
                headers: {
                    'x-is-ci': is_ci_1.isCI(),
                    authorization: api_token_1.getAuthHeader(),
                },
                body: monitorDependenciesRequest,
                qs: {
                    org: options.org || configOrg,
                },
            };
            try {
                const response = await promise_1.makeRequest(payload);
                results.push(Object.assign(Object.assign({}, response), { path,
                    scanResult }));
            }
            catch (error) {
                if (error.code === 401) {
                    throw errors_1.AuthFailedError();
                }
                if (error.code >= 400 && error.code < 500) {
                    throw new errors_1.MonitorError(error.code, error.message);
                }
                errors.push({
                    error: 'Could not monitor dependencies in ' + path,
                    path,
                    scanResult,
                });
            }
        }
        spinner.clearAll();
    }
    return [results, errors];
}
async function getFormattedMonitorOutput(results, monitorResults, errors, options) {
    for (const monitorResult of monitorResults) {
        let monOutput = '';
        if (monitorResult.ok) {
            monOutput = format_monitor_response_1.formatMonitorOutput(monitorResult.scanResult.identity.type, monitorResult, options, monitorResult.projectName, await get_extra_project_count_1.getExtraProjectCount(monitorResult.path, options, 
            // TODO: Fix to pass the old "inspectResult.plugin.meta.allSubProjectNames", which ecosystem uses this?
            // "allSubProjectNames" can become a Fact returned by a plugin.
            {}));
        }
        else {
            monOutput = format_monitor_response_1.formatErrorMonitorOutput(monitorResult.scanResult.identity.type, monitorResult, options);
        }
        results.push({
            ok: true,
            data: monOutput,
            path: monitorResult.path,
            projectName: monitorResult.id,
        });
    }
    for (const monitorError of errors) {
        results.push({
            ok: false,
            data: new errors_1.MonitorError(500, monitorError),
            path: monitorError.path,
        });
    }
    const outputString = results
        .map((res) => {
        if (res.ok) {
            return res.data;
        }
        const errorMessage = res.data && res.data.userMessage
            ? chalk_1.default.bold.red(res.data.userMessage)
            : res.data
                ? res.data.message
                : 'Unknown error occurred.';
        return (chalk_1.default.bold.white('\nMonitoring ' + res.path + '...\n\n') + errorMessage);
    })
        .join('\n' + SEPARATOR);
    if (results.every((res) => res.ok)) {
        return outputString;
    }
    throw new Error(outputString);
}
exports.getFormattedMonitorOutput = getFormattedMonitorOutput;
//# sourceMappingURL=monitor.js.map