"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MetricsCollector = exports.SyntheticMetric = exports.TimerMetric = exports.Metric = exports.SyntheticMetricInstance = exports.TimerMetricInstance = exports.MetricInstance = exports.METRIC_TYPE_SYNTHETIC = exports.METRIC_TYPE_TIMER = void 0;
const debug = require('debug')('snyk-metrics');
exports.METRIC_TYPE_TIMER = 'timer';
exports.METRIC_TYPE_SYNTHETIC = 'synthetic';
class MetricInstance {
}
exports.MetricInstance = MetricInstance;
class TimerMetricInstance extends MetricInstance {
    /**
     * Creates a new TimerMetricInstance
     * @param metricTag used for logging to identify the metric
     */
    constructor(metricTag) {
        super();
        this.startTimeMs = 0;
        this.endTimeMs = 0;
        this.metricTag = metricTag;
    }
    getValue() {
        if (this.startTimeMs !== 0 && this.endTimeMs !== 0) {
            return this.endTimeMs - this.startTimeMs;
        }
        else {
            return undefined;
        }
    }
    start() {
        if (this.startTimeMs === 0) {
            this.startTimeMs = Date.now();
            debug(`Timer ${this.metricTag} started at ${this.startTimeMs}.`);
        }
        else {
            debug('Invalid Timer use: start() called when timer already stopped');
        }
    }
    stop() {
        if (this.endTimeMs === 0) {
            this.endTimeMs = Date.now();
            debug(`Timer ${this.metricTag} stopped at ${this.endTimeMs}. Elapsed time is ${this.getValue()}`);
        }
        else {
            debug('Invalid Timer use: stop() called when timer already stopped');
        }
    }
}
exports.TimerMetricInstance = TimerMetricInstance;
class SyntheticMetricInstance extends MetricInstance {
    constructor() {
        super(...arguments);
        this.value = 0;
    }
    setValue(value) {
        this.value = value;
    }
    getValue() {
        return this.value;
    }
}
exports.SyntheticMetricInstance = SyntheticMetricInstance;
class Metric {
    constructor(name, metricType, context) {
        this.instances = [];
        this.name = name;
        this.metricType = metricType;
        this.context = context;
    }
    clear() {
        this.instances = [];
    }
    getValues() {
        return this.instances.map((mi) => mi.getValue() || 0);
    }
    getTotal() {
        const sumMetricValues = (accum, current) => {
            const currentTimerMs = current.getValue() || 0;
            return (accum = accum + currentTimerMs);
        };
        const total = this.instances.reduce(sumMetricValues, 0);
        return total;
    }
}
exports.Metric = Metric;
class TimerMetric extends Metric {
    createInstance() {
        const t = new TimerMetricInstance(`${this.metricType}/${this.name}`);
        this.instances.push(t);
        return t;
    }
}
exports.TimerMetric = TimerMetric;
class SyntheticMetric extends Metric {
    createInstance() {
        const sm = new SyntheticMetricInstance();
        this.instances.push(sm);
        return sm;
    }
}
exports.SyntheticMetric = SyntheticMetric;
class MetricsCollector {
    static getAllMetrics() {
        const metrics = [
            MetricsCollector.NETWORK_TIME,
            MetricsCollector.CPU_TIME,
        ];
        const res = {};
        for (const m of metrics) {
            res[m.name] = {
                type: m.metricType,
                values: m.getValues(),
                total: m.getTotal(),
            };
        }
        return res;
    }
}
exports.MetricsCollector = MetricsCollector;
MetricsCollector.NETWORK_TIME = new TimerMetric('network_time', 'timer', 'Total time spent making and waiting on network requests');
MetricsCollector.CPU_TIME = new SyntheticMetric('cpu_time', 'synthetic', 'Time spent on things other than network requests');
//# sourceMappingURL=metrics.js.map