"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCodeAnalysisAndParseResults = void 0;
const code_client_1 = require("@snyk/code-client");
const legacy_1 = require("../../snyk-test/legacy");
const api_token_1 = require("../../api-token");
const config = require("../../config");
const spinner = require("../../spinner");
const utils_1 = require("./utils");
const unsupported_feature_snyk_code_error_1 = require("./errors/unsupported-feature-snyk-code-error");
const proxy_from_env_1 = require("proxy-from-env");
const global_agent_1 = require("global-agent");
async function getCodeAnalysisAndParseResults(root, options) {
    const currentLabel = '';
    await spinner.clearAll();
    utils_1.analysisProgressUpdate(currentLabel);
    const codeAnalysis = await getCodeAnalysis(root, options);
    spinner.clearAll();
    return parseSecurityResults(codeAnalysis);
}
exports.getCodeAnalysisAndParseResults = getCodeAnalysisAndParseResults;
async function getCodeAnalysis(root, options) {
    const baseURL = config.CODE_CLIENT_PROXY_URL;
    // TODO(james) This mirrors the implementation in request.ts and we need to use this for deeproxy calls
    // This ensures we support lowercase http(s)_proxy values as well
    // The weird IF around it ensures we don't create an envvar with
    // a value of undefined, which throws error when trying to use it as a proxy
    if (process.env.HTTP_PROXY || process.env.http_proxy) {
        process.env.HTTP_PROXY = process.env.HTTP_PROXY || process.env.http_proxy;
    }
    if (process.env.HTTPS_PROXY || process.env.https_proxy) {
        process.env.HTTPS_PROXY =
            process.env.HTTPS_PROXY || process.env.https_proxy;
    }
    const proxyUrl = proxy_from_env_1.getProxyForUrl(baseURL);
    if (proxyUrl) {
        global_agent_1.bootstrap({
            environmentVariableNamespace: '',
        });
    }
    const sessionToken = api_token_1.api() || '';
    const severity = options.severityThreshold
        ? severityToAnalysisSeverity(options.severityThreshold)
        : code_client_1.AnalysisSeverity.info;
    const paths = [root];
    const sarif = true;
    const result = await code_client_1.analyzeFolders({
        baseURL,
        sessionToken,
        severity,
        paths,
        sarif,
    });
    return result.sarifResults;
}
function severityToAnalysisSeverity(severity) {
    if (severity === legacy_1.SEVERITY.CRITICAL) {
        throw new unsupported_feature_snyk_code_error_1.FeatureNotSupportedBySnykCodeError(legacy_1.SEVERITY.CRITICAL);
    }
    const severityLevel = {
        low: 1,
        medium: 2,
        high: 3,
    };
    return severityLevel[severity];
}
function parseSecurityResults(codeAnalysis) {
    let securityRulesMap;
    const rules = codeAnalysis.runs[0].tool.driver.rules;
    const results = codeAnalysis.runs[0].results;
    if (rules) {
        securityRulesMap = getSecurityRulesMap(rules);
        codeAnalysis.runs[0].tool.driver.rules = Object.values(securityRulesMap);
    }
    if (results && securityRulesMap) {
        codeAnalysis.runs[0].results = getSecurityResultsOnly(results, Object.keys(securityRulesMap));
    }
    return codeAnalysis;
}
function getSecurityRulesMap(rules) {
    const securityRulesMap = rules.reduce((acc, rule) => {
        var _a;
        const { id: ruleId, properties } = rule;
        const isSecurityRule = (_a = properties === null || properties === void 0 ? void 0 : properties.categories) === null || _a === void 0 ? void 0 : _a.some((category) => category.toLowerCase() === 'security');
        if (isSecurityRule) {
            acc[ruleId] = rule;
        }
        return acc;
    }, {});
    return securityRulesMap;
}
function getSecurityResultsOnly(results, securityRules) {
    const securityResults = results.reduce((acc, result) => {
        const isSecurityResult = securityRules.some((securityRule) => securityRule === (result === null || result === void 0 ? void 0 : result.ruleId));
        if (isSecurityResult) {
            acc.push(result);
        }
        return acc;
    }, []);
    return securityResults;
}
//# sourceMappingURL=analysis.js.map