"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createBundleFromFolders = exports.analyzeGit = exports.extendAnalysis = exports.analyzeFolders = exports.analyzeBundle = void 0;
/* eslint-disable no-await-in-loop */
const lodash_omit_1 = __importDefault(require("lodash.omit"));
const files_1 = require("./files");
const http_1 = require("./http");
const emitter_1 = __importDefault(require("./emitter"));
const constants_1 = require("./constants");
const bundles_1 = require("./bundles");
const sarif_converter_1 = __importDefault(require("./sarif_converter"));
const analysis_result_interface_1 = require("./interfaces/analysis-result.interface");
const utils_1 = require("./lib/utils");
const sleep = (duration) => new Promise(resolve => setTimeout(resolve, duration));
const ANALYSIS_OPTIONS_DEFAULTS = {
    baseURL: constants_1.defaultBaseURL,
    sessionToken: '',
    includeLint: false,
    reachability: false,
    severity: analysis_result_interface_1.AnalysisSeverity.info,
    symlinksEnabled: false,
    maxPayload: constants_1.MAX_PAYLOAD,
    defaultFileIgnores: constants_1.IGNORES_DEFAULT,
    sarif: false,
    source: '',
};
async function pollAnalysis({ baseURL, sessionToken, includeLint, severity, bundleId, oAuthToken, username, limitToFiles, source, reachability, }, requestOptions) {
    let analysisResponse;
    let analysisData;
    emitter_1.default.analyseProgress({
        status: http_1.AnalysisStatus.waiting,
        progress: 0,
    });
    // eslint-disable-next-line no-constant-condition
    while (true) {
        // eslint-disable-next-line no-await-in-loop
        analysisResponse = await http_1.getAnalysis({
            baseURL,
            sessionToken,
            oAuthToken,
            username,
            bundleId,
            includeLint,
            severity,
            limitToFiles,
            source,
            reachability,
        }, requestOptions);
        if (analysisResponse.type === 'error') {
            return analysisResponse;
        }
        analysisData = analysisResponse.value;
        if (analysisData.status === http_1.AnalysisStatus.waiting ||
            analysisData.status === http_1.AnalysisStatus.fetching ||
            analysisData.status === http_1.AnalysisStatus.analyzing ||
            analysisData.status === http_1.AnalysisStatus.dcDone) {
            // Report progress of fetching
            emitter_1.default.analyseProgress(analysisData);
        }
        else if (analysisData.status === http_1.AnalysisStatus.done) {
            // Return data of analysis
            return analysisResponse;
            // deepcode ignore DuplicateIfBody: false positive it seems that interface is not taken into account
        }
        else if (analysisData.status === http_1.AnalysisStatus.failed) {
            // Report failure of analysing
            return analysisResponse;
        }
        await sleep(500);
    }
}
async function analyzeBundle({ baseURL = constants_1.defaultBaseURL, sessionToken = '', includeLint = false, severity = analysis_result_interface_1.AnalysisSeverity.info, bundleId, oAuthToken, username, limitToFiles, source, reachability = false, }, requestOptions) {
    // Call remote bundle for analysis results and emit intermediate progress
    const analysisData = await pollAnalysis({
        baseURL,
        sessionToken,
        oAuthToken,
        username,
        bundleId,
        includeLint,
        severity,
        limitToFiles,
        source,
        reachability,
    }, requestOptions);
    if (analysisData.type === 'error') {
        throw analysisData.error;
    }
    else if (analysisData.value.status === http_1.AnalysisStatus.failed) {
        throw new Error('Analysis has failed');
    }
    const { analysisResults } = analysisData.value;
    // Create bundle instance to handle extensions
    return {
        bundleId,
        analysisResults,
        analysisURL: analysisData.value.analysisURL,
    };
}
exports.analyzeBundle = analyzeBundle;
function normalizeResultFiles(files, baseDir) {
    if (baseDir) {
        return utils_1.fromEntries(Object.entries(files).map(([path, positions]) => {
            const filePath = files_1.resolveBundleFilePath(baseDir, path);
            return [filePath, positions];
        }));
    }
    return files;
}
const moveSuggestionIndexes = (suggestionIndex, suggestions) => {
    const entries = Object.entries(suggestions);
    return utils_1.fromEntries(entries.map(([i, s]) => {
        return [`${parseInt(i, 10) + suggestionIndex + 1}`, s];
    }));
};
function mergeBundleResults(bundle, analysisData, limitToFiles) {
    // Determine max suggestion index in our data
    const suggestionIndex = Math.max(...Object.keys(bundle.analysisResults.suggestions).map(i => parseInt(i, 10))) || -1;
    // Addup all new suggestions' indexes
    const newSuggestions = moveSuggestionIndexes(suggestionIndex, analysisData.analysisResults.suggestions);
    const suggestions = { ...bundle.analysisResults.suggestions, ...newSuggestions };
    const newFiles = utils_1.fromEntries(Object.entries(analysisData.analysisResults.files).map(([fn, s]) => {
        return [fn, moveSuggestionIndexes(suggestionIndex, s)];
    }));
    const files = {
        ...lodash_omit_1.default(bundle.analysisResults.files, limitToFiles),
        ...newFiles,
    };
    const analysisResults = {
        ...analysisData.analysisResults,
        files,
        suggestions,
    };
    return {
        ...bundle,
        ...analysisData,
        analysisResults,
    };
}
async function analyzeFolders(options) {
    const analysisOptions = { ...ANALYSIS_OPTIONS_DEFAULTS, ...options };
    const { baseURL, sessionToken, includeLint, reachability, severity, paths, symlinksEnabled, sarif, defaultFileIgnores, source, } = analysisOptions;
    const supportedFiles = await getSupportedFiles(baseURL, source);
    // Scan directories and find all suitable files
    const baseDir = files_1.determineBaseDir(paths);
    // Scan for custom ignore rules
    const fileIgnores = await files_1.collectIgnoreRules(paths, symlinksEnabled, defaultFileIgnores);
    const remoteBundle = await createBundleFromFolders({ ...analysisOptions, supportedFiles, baseDir, fileIgnores });
    // Analyze bundle
    let analysisData;
    if (remoteBundle === null) {
        analysisData = {
            analysisResults: {
                files: {},
                suggestions: {},
                timing: {
                    analysis: 0,
                    fetchingCode: 0,
                    queue: 0,
                },
                coverage: [],
            },
            analysisURL: '',
            bundleId: '',
        };
    }
    else {
        analysisData = await analyzeBundle({
            baseURL,
            sessionToken,
            includeLint,
            reachability,
            severity,
            bundleId: remoteBundle.bundleId,
            source,
        });
        analysisData.analysisResults.files = normalizeResultFiles(analysisData.analysisResults.files, baseDir);
    }
    const result = {
        baseURL,
        sessionToken,
        includeLint,
        reachability,
        severity,
        supportedFiles,
        baseDir,
        paths,
        fileIgnores,
        symlinksEnabled,
        ...analysisData,
    };
    if (sarif && analysisData.analysisResults) {
        result.sarifResults = sarif_converter_1.default(analysisData.analysisResults);
    }
    return result;
}
exports.analyzeFolders = analyzeFolders;
async function extendAnalysis(bundle, filePaths, maxPayload = constants_1.MAX_PAYLOAD, source) {
    const { files, removedFiles } = await files_1.prepareExtendingBundle(bundle.baseDir, filePaths, bundle.supportedFiles, bundle.fileIgnores, maxPayload, bundle.symlinksEnabled);
    if (!files.length && !removedFiles.length) {
        return null; // nothing to extend, just return null
    }
    // Extend remote bundle
    const remoteBundle = await bundles_1.remoteBundleFactory(bundle.baseURL, bundle.sessionToken, files, removedFiles, bundle.baseDir, bundle.bundleId, maxPayload, source);
    if (remoteBundle === null) {
        // File list is empty
        // nothing to extend, just return null
        return null;
    }
    const analysisData = await analyzeBundle({
        baseURL: bundle.baseURL,
        sessionToken: bundle.sessionToken,
        includeLint: bundle.includeLint,
        severity: bundle.severity,
        bundleId: remoteBundle.bundleId,
        limitToFiles: files.map(f => f.bundlePath),
        source,
    });
    // Transform relative paths into absolute
    analysisData.analysisResults.files = normalizeResultFiles(analysisData.analysisResults.files, bundle.baseDir);
    // Merge into base bundle results
    return mergeBundleResults(bundle, analysisData, files.map(f => f.filePath));
}
exports.extendAnalysis = extendAnalysis;
async function analyzeGit(options, requestOptions) {
    const analysisOptions = { ...ANALYSIS_OPTIONS_DEFAULTS, ...options };
    const { baseURL, sessionToken, oAuthToken, username, includeLint, reachability, severity, gitUri, sarif, source } = analysisOptions;
    const bundleResponse = await http_1.createGitBundle({
        baseURL,
        sessionToken,
        oAuthToken,
        username,
        gitUri,
        source,
    }, requestOptions);
    if (bundleResponse.type === 'error') {
        throw bundleResponse.error;
    }
    const { bundleId } = bundleResponse.value;
    const analysisData = await analyzeBundle({
        baseURL,
        sessionToken,
        oAuthToken,
        username,
        includeLint,
        reachability,
        severity,
        bundleId,
        source,
    }, requestOptions);
    const result = {
        baseURL,
        sessionToken,
        oAuthToken,
        includeLint,
        reachability,
        severity,
        gitUri,
        ...analysisData,
    };
    // Create bundle instance to handle extensions
    if (sarif && analysisData.analysisResults) {
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        result.sarifResults = sarif_converter_1.default(analysisData.analysisResults);
    }
    return result;
}
exports.analyzeGit = analyzeGit;
/**
 * Creates a remote bundle and returns response from the bundle API
 *
 * @param {CreateBundleFromFoldersOptions} options
 * @returns {Promise<RemoteBundle | null>}
 */
async function createBundleFromFolders(options) {
    const analysisOptions = { ...ANALYSIS_OPTIONS_DEFAULTS, ...options };
    const { baseURL, source, paths, symlinksEnabled, defaultFileIgnores, maxPayload, sessionToken, supportedFiles = await getSupportedFiles(baseURL, source), baseDir = files_1.determineBaseDir(paths), fileIgnores = await files_1.collectIgnoreRules(paths, symlinksEnabled, defaultFileIgnores), } = analysisOptions;
    emitter_1.default.scanFilesProgress(0);
    const bundleFiles = [];
    let totalFiles = 0;
    const bundleFileCollector = files_1.collectBundleFiles(baseDir, paths, supportedFiles, fileIgnores, maxPayload, symlinksEnabled);
    for await (const f of bundleFileCollector) {
        bundleFiles.push(f);
        totalFiles += 1;
        emitter_1.default.scanFilesProgress(totalFiles);
    }
    // Create remote bundle
    return bundleFiles.length
        ? await bundles_1.remoteBundleFactory(baseURL, sessionToken, bundleFiles, [], baseDir, null, maxPayload, source)
        : null;
}
exports.createBundleFromFolders = createBundleFromFolders;
/**
 * Get supported filters and test baseURL for correctness and availability
 *
 * @param baseURL
 * @param source
 * @returns
 */
async function getSupportedFiles(baseURL, source) {
    emitter_1.default.supportedFilesLoaded(null);
    const resp = await http_1.getFilters(baseURL, source);
    if (resp.type === 'error') {
        throw resp.error;
    }
    const supportedFiles = resp.value;
    emitter_1.default.supportedFilesLoaded(supportedFiles);
    return supportedFiles;
}
//# sourceMappingURL=analysis.js.map