"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.xfs = exports.extendFs = exports.patchFs = exports.ZipOpenFS = exports.ZipFS = exports.VirtualFS = exports.ProxiedFS = exports.PosixFS = exports.NodeFS = exports.NoFS = exports.LazyFS = exports.JailFS = exports.CwdFS = exports.FakeFS = exports.AliasFS = exports.toFilename = exports.ppath = exports.npath = exports.Filename = exports.PortablePath = exports.DEFAULT_COMPRESSION_LEVEL = exports.normalizeLineEndings = exports.statUtils = exports.opendir = void 0;
const tslib_1 = require("tslib");
const os_1 = tslib_1.__importDefault(require("os"));
const util_1 = require("util");
const NodeFS_1 = require("./NodeFS");
const path_1 = require("./path");
const statUtils = tslib_1.__importStar(require("./statUtils"));
exports.statUtils = statUtils;
var opendir_1 = require("./algorithms/opendir");
Object.defineProperty(exports, "opendir", { enumerable: true, get: function () { return opendir_1.opendir; } });
var FakeFS_1 = require("./FakeFS");
Object.defineProperty(exports, "normalizeLineEndings", { enumerable: true, get: function () { return FakeFS_1.normalizeLineEndings; } });
var ZipFS_1 = require("./ZipFS");
Object.defineProperty(exports, "DEFAULT_COMPRESSION_LEVEL", { enumerable: true, get: function () { return ZipFS_1.DEFAULT_COMPRESSION_LEVEL; } });
var path_2 = require("./path");
Object.defineProperty(exports, "PortablePath", { enumerable: true, get: function () { return path_2.PortablePath; } });
Object.defineProperty(exports, "Filename", { enumerable: true, get: function () { return path_2.Filename; } });
var path_3 = require("./path");
Object.defineProperty(exports, "npath", { enumerable: true, get: function () { return path_3.npath; } });
Object.defineProperty(exports, "ppath", { enumerable: true, get: function () { return path_3.ppath; } });
Object.defineProperty(exports, "toFilename", { enumerable: true, get: function () { return path_3.toFilename; } });
var AliasFS_1 = require("./AliasFS");
Object.defineProperty(exports, "AliasFS", { enumerable: true, get: function () { return AliasFS_1.AliasFS; } });
var FakeFS_2 = require("./FakeFS");
Object.defineProperty(exports, "FakeFS", { enumerable: true, get: function () { return FakeFS_2.FakeFS; } });
var CwdFS_1 = require("./CwdFS");
Object.defineProperty(exports, "CwdFS", { enumerable: true, get: function () { return CwdFS_1.CwdFS; } });
var JailFS_1 = require("./JailFS");
Object.defineProperty(exports, "JailFS", { enumerable: true, get: function () { return JailFS_1.JailFS; } });
var LazyFS_1 = require("./LazyFS");
Object.defineProperty(exports, "LazyFS", { enumerable: true, get: function () { return LazyFS_1.LazyFS; } });
var NoFS_1 = require("./NoFS");
Object.defineProperty(exports, "NoFS", { enumerable: true, get: function () { return NoFS_1.NoFS; } });
var NodeFS_2 = require("./NodeFS");
Object.defineProperty(exports, "NodeFS", { enumerable: true, get: function () { return NodeFS_2.NodeFS; } });
var PosixFS_1 = require("./PosixFS");
Object.defineProperty(exports, "PosixFS", { enumerable: true, get: function () { return PosixFS_1.PosixFS; } });
var ProxiedFS_1 = require("./ProxiedFS");
Object.defineProperty(exports, "ProxiedFS", { enumerable: true, get: function () { return ProxiedFS_1.ProxiedFS; } });
var VirtualFS_1 = require("./VirtualFS");
Object.defineProperty(exports, "VirtualFS", { enumerable: true, get: function () { return VirtualFS_1.VirtualFS; } });
var ZipFS_2 = require("./ZipFS");
Object.defineProperty(exports, "ZipFS", { enumerable: true, get: function () { return ZipFS_2.ZipFS; } });
var ZipOpenFS_1 = require("./ZipOpenFS");
Object.defineProperty(exports, "ZipOpenFS", { enumerable: true, get: function () { return ZipOpenFS_1.ZipOpenFS; } });
function getTempName(prefix) {
    const tmpdir = path_1.npath.toPortablePath(os_1.default.tmpdir());
    const hash = Math.ceil(Math.random() * 0x100000000).toString(16).padStart(8, `0`);
    return path_1.ppath.join(tmpdir, `${prefix}${hash}`);
}
function patchFs(patchedFs, fakeFs) {
    const SYNC_IMPLEMENTATIONS = new Set([
        `accessSync`,
        `appendFileSync`,
        `createReadStream`,
        `chmodSync`,
        `chownSync`,
        `closeSync`,
        `copyFileSync`,
        `linkSync`,
        `lstatSync`,
        `lutimesSync`,
        `mkdirSync`,
        `openSync`,
        `opendirSync`,
        `readSync`,
        `readlinkSync`,
        `readFileSync`,
        `readdirSync`,
        `readlinkSync`,
        `realpathSync`,
        `renameSync`,
        `rmdirSync`,
        `statSync`,
        `symlinkSync`,
        `truncateSync`,
        `unlinkSync`,
        `unwatchFile`,
        `utimesSync`,
        `watch`,
        `watchFile`,
        `writeFileSync`,
        `writeSync`,
    ]);
    const ASYNC_IMPLEMENTATIONS = new Set([
        `accessPromise`,
        `appendFilePromise`,
        `chmodPromise`,
        `chownPromise`,
        `closePromise`,
        `copyFilePromise`,
        `linkPromise`,
        `lstatPromise`,
        `lutimesPromise`,
        `mkdirPromise`,
        `openPromise`,
        `opendirPromise`,
        `readdirPromise`,
        `realpathPromise`,
        `readFilePromise`,
        `readdirPromise`,
        `readlinkPromise`,
        `renamePromise`,
        `rmdirPromise`,
        `statPromise`,
        `symlinkPromise`,
        `truncatePromise`,
        `unlinkPromise`,
        `utimesPromise`,
        `writeFilePromise`,
        `writeSync`,
    ]);
    const FILEHANDLE_IMPLEMENTATIONS = new Set([
        `appendFilePromise`,
        `chmodPromise`,
        `chownPromise`,
        `closePromise`,
        `readPromise`,
        `readFilePromise`,
        `statPromise`,
        `truncatePromise`,
        `utimesPromise`,
        `writePromise`,
        `writeFilePromise`,
    ]);
    const setupFn = (target, name, replacement) => {
        const orig = target[name];
        target[name] = replacement;
        // Preserve any util.promisify implementations
        if (typeof (orig === null || orig === void 0 ? void 0 : orig[util_1.promisify.custom]) !== `undefined`) {
            replacement[util_1.promisify.custom] = orig[util_1.promisify.custom];
        }
    };
    /** Callback implementations */
    {
        setupFn(patchedFs, `exists`, (p, ...args) => {
            const hasCallback = typeof args[args.length - 1] === `function`;
            const callback = hasCallback ? args.pop() : () => { };
            process.nextTick(() => {
                fakeFs.existsPromise(p).then(exists => {
                    callback(exists);
                }, () => {
                    callback(false);
                });
            });
        });
        setupFn(patchedFs, `read`, (p, buffer, ...args) => {
            const hasCallback = typeof args[args.length - 1] === `function`;
            const callback = hasCallback ? args.pop() : () => { };
            process.nextTick(() => {
                fakeFs.readPromise(p, buffer, ...args).then(bytesRead => {
                    callback(null, bytesRead, buffer);
                }, error => {
                    callback(error);
                });
            });
        });
        for (const fnName of ASYNC_IMPLEMENTATIONS) {
            const origName = fnName.replace(/Promise$/, ``);
            if (typeof patchedFs[origName] === `undefined`)
                continue;
            const fakeImpl = fakeFs[fnName];
            if (typeof fakeImpl === `undefined`)
                continue;
            const wrapper = (...args) => {
                const hasCallback = typeof args[args.length - 1] === `function`;
                const callback = hasCallback ? args.pop() : () => { };
                process.nextTick(() => {
                    fakeImpl.apply(fakeFs, args).then((result) => {
                        callback(null, result);
                    }, (error) => {
                        callback(error);
                    });
                });
            };
            setupFn(patchedFs, origName, wrapper);
        }
        patchedFs.realpath.native = patchedFs.realpath;
    }
    /** Sync implementations */
    {
        setupFn(patchedFs, `existsSync`, (p) => {
            try {
                return fakeFs.existsSync(p);
            }
            catch (error) {
                return false;
            }
        });
        for (const fnName of SYNC_IMPLEMENTATIONS) {
            const origName = fnName;
            if (typeof patchedFs[origName] === `undefined`)
                continue;
            const fakeImpl = fakeFs[fnName];
            if (typeof fakeImpl === `undefined`)
                continue;
            setupFn(patchedFs, origName, fakeImpl.bind(fakeFs));
        }
        patchedFs.realpathSync.native = patchedFs.realpathSync;
    }
    /** Promise implementations */
    {
        // `fs.promises` is a getter that returns a reference to require(`fs/promises`),
        // so we can just patch `fs.promises` and both will be updated
        const origEmitWarning = process.emitWarning;
        process.emitWarning = () => { };
        let patchedFsPromises;
        try {
            patchedFsPromises = patchedFs.promises;
        }
        finally {
            process.emitWarning = origEmitWarning;
        }
        if (typeof patchedFsPromises !== `undefined`) {
            // `fs.promises.exists` doesn't exist
            for (const fnName of ASYNC_IMPLEMENTATIONS) {
                const origName = fnName.replace(/Promise$/, ``);
                if (typeof patchedFsPromises[origName] === `undefined`)
                    continue;
                const fakeImpl = fakeFs[fnName];
                if (typeof fakeImpl === `undefined`)
                    continue;
                // Open is a bit particular with fs.promises: it returns a file handle
                // instance instead of the traditional file descriptor number
                if (fnName === `open`)
                    continue;
                setupFn(patchedFsPromises, origName, fakeImpl.bind(fakeFs));
            }
            class FileHandle {
                constructor(fd) {
                    this.fd = fd;
                }
            }
            for (const fnName of FILEHANDLE_IMPLEMENTATIONS) {
                const origName = fnName.replace(/Promise$/, ``);
                const fakeImpl = fakeFs[fnName];
                if (typeof fakeImpl === `undefined`)
                    continue;
                setupFn(FileHandle.prototype, origName, function (...args) {
                    return fakeImpl.call(fakeFs, this.fd, ...args);
                });
            }
            setupFn(patchedFsPromises, `open`, async (...args) => {
                // @ts-expect-error
                const fd = await fakeFs.openPromise(...args);
                return new FileHandle(fd);
            });
            // `fs.promises.realpath` doesn't have a `native` property
        }
    }
    /** util.promisify implementations */
    {
        // Override the promisified version of `fs.read` to return an object as per
        // https://github.com/nodejs/node/blob/dc79f3f37caf6f25b8efee4623bec31e2c20f595/lib/fs.js#L559-L560
        // and
        // https://github.com/nodejs/node/blob/ba684805b6c0eded76e5cd89ee00328ac7a59365/lib/internal/util.js#L293
        // @ts-expect-error
        patchedFs.read[util_1.promisify.custom] = async (p, buffer, ...args) => {
            const res = fakeFs.readPromise(p, buffer, ...args);
            return { bytesRead: await res, buffer };
        };
    }
}
exports.patchFs = patchFs;
function extendFs(realFs, fakeFs) {
    const patchedFs = Object.create(realFs);
    patchFs(patchedFs, fakeFs);
    return patchedFs;
}
exports.extendFs = extendFs;
const tmpdirs = new Set();
let cleanExitRegistered = false;
function registerCleanExit() {
    if (cleanExitRegistered)
        return;
    cleanExitRegistered = true;
    process.once(`exit`, () => {
        exports.xfs.rmtempSync();
    });
}
exports.xfs = Object.assign(new NodeFS_1.NodeFS(), {
    detachTemp(p) {
        tmpdirs.delete(p);
    },
    mktempSync(cb) {
        registerCleanExit();
        while (true) {
            const p = getTempName(`xfs-`);
            try {
                this.mkdirSync(p);
            }
            catch (error) {
                if (error.code === `EEXIST`) {
                    continue;
                }
                else {
                    throw error;
                }
            }
            const realP = this.realpathSync(p);
            tmpdirs.add(realP);
            if (typeof cb !== `undefined`) {
                try {
                    return cb(realP);
                }
                finally {
                    if (tmpdirs.has(realP)) {
                        tmpdirs.delete(realP);
                        try {
                            this.removeSync(realP);
                        }
                        catch (_a) {
                            // Too bad if there's an error
                        }
                    }
                }
            }
            else {
                return p;
            }
        }
    },
    async mktempPromise(cb) {
        registerCleanExit();
        while (true) {
            const p = getTempName(`xfs-`);
            try {
                await this.mkdirPromise(p);
            }
            catch (error) {
                if (error.code === `EEXIST`) {
                    continue;
                }
                else {
                    throw error;
                }
            }
            const realP = await this.realpathPromise(p);
            tmpdirs.add(realP);
            if (typeof cb !== `undefined`) {
                try {
                    return await cb(realP);
                }
                finally {
                    if (tmpdirs.has(realP)) {
                        tmpdirs.delete(realP);
                        try {
                            await this.removePromise(realP);
                        }
                        catch (_a) {
                            // Too bad if there's an error
                        }
                    }
                }
            }
            else {
                return realP;
            }
        }
    },
    async rmtempPromise() {
        await Promise.all(Array.from(tmpdirs.values()).map(async (p) => {
            try {
                await exports.xfs.removePromise(p, { maxRetries: 0 });
                tmpdirs.delete(p);
            }
            catch (_a) {
                // Too bad if there's an error
            }
        }));
    },
    rmtempSync() {
        for (const p of tmpdirs) {
            try {
                exports.xfs.removeSync(p);
                tmpdirs.delete(p);
            }
            catch (_a) {
                // Too bad if there's an error
            }
        }
    },
});
