"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.reportEvent = exports.reportError = exports.getAnalysis = exports.AnalysisStatus = exports.uploadFiles = exports.createGitBundle = exports.extendBundle = exports.checkBundle = exports.createBundle = exports.getFilters = exports.checkSession = exports.startSession = exports.determineErrorCode = void 0;
const uuid_1 = require("uuid");
const constants_1 = require("./constants");
const axios_1 = __importDefault(require("./axios"));
function determineErrorCode(error) {
    // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
    const { response } = error;
    if (response) {
        return response.status;
    }
    // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
    const { errno, code } = error;
    if (errno === 'ECONNREFUSED') {
        // set connectionRefused item
        return constants_1.ErrorCodes.connectionRefused;
    }
    if (errno === 'ECONNRESET') {
        // set connectionRefused item
        return constants_1.ErrorCodes.connectionRefused;
    }
    if (code === 'ENOTFOUND') {
        return constants_1.ErrorCodes.dnsNotFound;
    }
    // We must be strict here and if none of our existing logic recognized this error, just throw it up.
    throw error;
}
exports.determineErrorCode = determineErrorCode;
// The trick to typecast union type alias
function isSubsetErrorCode(code, messages) {
    if (code in messages) {
        return true;
    }
    return false;
}
function generateError(error, messages, apiName) {
    const errorCode = determineErrorCode(error);
    if (!isSubsetErrorCode(errorCode, messages)) {
        throw error;
    }
    const statusCode = errorCode;
    const statusText = messages[errorCode];
    return {
        type: 'error',
        error: {
            apiName,
            statusCode,
            statusText,
        },
    };
}
const GENERIC_ERROR_MESSAGES = {
    [constants_1.ErrorCodes.serverError]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.serverError],
    [constants_1.ErrorCodes.badGateway]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.badGateway],
    [constants_1.ErrorCodes.serviceUnavailable]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.serviceUnavailable],
    [constants_1.ErrorCodes.timeout]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.timeout],
    [constants_1.ErrorCodes.dnsNotFound]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.dnsNotFound],
    [constants_1.ErrorCodes.connectionRefused]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.connectionRefused],
};
function startSession(options) {
    const { source, authHost } = options;
    const draftToken = uuid_1.v4();
    return {
        draftToken,
        loginURL: `${authHost}/login?token=${draftToken}&utm_medium=${source}&utm_source=${source}&utm_campaign=${source}&docker=false`,
    };
}
exports.startSession = startSession;
const CHECK_SESSION_ERROR_MESSAGES = {
    ...GENERIC_ERROR_MESSAGES,
    [constants_1.ErrorCodes.unauthorizedUser]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.unauthorizedUser],
    [constants_1.ErrorCodes.loginInProgress]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.loginInProgress],
};
async function checkSession(options) {
    var _a;
    const { draftToken, authHost } = options;
    const config = {
        url: `${authHost}/api/v1/verify/callback`,
        method: 'POST',
        data: {
            token: draftToken,
        },
    };
    try {
        const response = await axios_1.default.request(config);
        return {
            type: 'success',
            // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment, @typescript-eslint/no-unsafe-member-access
            value: (response.status === 200 && response.data.ok && response.data.api) || '',
        };
    }
    catch (err) {
        if ([constants_1.ErrorCodes.loginInProgress, constants_1.ErrorCodes.unauthorizedContent, constants_1.ErrorCodes.unauthorizedUser].includes(
        // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
        (_a = err.response) === null || _a === void 0 ? void 0 : _a.status)) {
            return { type: 'success', value: '' };
        }
        return generateError(err, CHECK_SESSION_ERROR_MESSAGES, 'checkSession');
    }
}
exports.checkSession = checkSession;
async function getFilters(baseURL, source) {
    const apiName = 'filters';
    const config = {
        headers: { source },
        url: `${baseURL}${constants_1.apiPath}/${apiName}`,
        method: 'GET',
    };
    try {
        const response = await axios_1.default.request(config);
        return { type: 'success', value: response.data };
    }
    catch (error) {
        return generateError(error, GENERIC_ERROR_MESSAGES, apiName);
    }
}
exports.getFilters = getFilters;
const CREATE_BUNDLE_ERROR_MESSAGES = {
    ...GENERIC_ERROR_MESSAGES,
    [constants_1.ErrorCodes.unauthorizedUser]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.unauthorizedUser],
    [constants_1.ErrorCodes.unauthorizedBundleAccess]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.unauthorizedBundleAccess],
    [constants_1.ErrorCodes.bigPayload]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.bigPayload],
    [constants_1.ErrorCodes.unauthorizedContent]: `Request content doesn't match the specifications`,
    [constants_1.ErrorCodes.notFound]: 'Unable to resolve requested oid',
};
async function createBundle(options) {
    const { baseURL, sessionToken, files, source } = options;
    const config = {
        headers: { 'Session-Token': sessionToken, source },
        url: `${baseURL}${constants_1.apiPath}/bundle`,
        method: 'POST',
        data: {
            files,
        },
    };
    try {
        const response = await axios_1.default.request(config);
        return { type: 'success', value: response.data };
    }
    catch (error) {
        return generateError(error, CREATE_BUNDLE_ERROR_MESSAGES, 'createBundle');
    }
}
exports.createBundle = createBundle;
const CHECK_BUNDLE_ERROR_MESSAGES = {
    ...GENERIC_ERROR_MESSAGES,
    [constants_1.ErrorCodes.unauthorizedUser]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.unauthorizedUser],
    [constants_1.ErrorCodes.unauthorizedBundleAccess]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.unauthorizedBundleAccess],
    [constants_1.ErrorCodes.notFound]: 'Uploaded bundle has expired',
};
async function checkBundle(options) {
    const { baseURL, sessionToken, bundleId } = options;
    const config = {
        headers: { 'Session-Token': sessionToken },
        url: `${baseURL}${constants_1.apiPath}/bundle/${bundleId}`,
        method: 'GET',
    };
    try {
        const response = await axios_1.default.request(config);
        return { type: 'success', value: response.data };
    }
    catch (error) {
        return generateError(error, CHECK_BUNDLE_ERROR_MESSAGES, 'checkBundle');
    }
}
exports.checkBundle = checkBundle;
const EXTEND_BUNDLE_ERROR_MESSAGES = {
    ...GENERIC_ERROR_MESSAGES,
    [constants_1.ErrorCodes.unauthorizedUser]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.unauthorizedUser],
    [constants_1.ErrorCodes.bigPayload]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.bigPayload],
    [constants_1.ErrorCodes.unauthorizedContent]: `Attempted to extend a git bundle, or ended up with an empty bundle after the extension`,
    [constants_1.ErrorCodes.unauthorizedBundleAccess]: 'Unauthorized access to parent bundle',
    [constants_1.ErrorCodes.notFound]: 'Parent bundle has expired',
};
async function extendBundle(options) {
    const { baseURL, sessionToken, bundleId, files, removedFiles = [] } = options;
    const config = {
        headers: { 'Session-Token': sessionToken },
        url: `${baseURL}${constants_1.apiPath}/bundle/${bundleId}`,
        method: 'PUT',
        data: {
            files,
            removedFiles,
        },
    };
    try {
        const response = await axios_1.default.request(config);
        return { type: 'success', value: response.data };
    }
    catch (error) {
        return generateError(error, EXTEND_BUNDLE_ERROR_MESSAGES, 'extendBundle');
    }
}
exports.extendBundle = extendBundle;
const CREATE_GIT_BUNDLE_ERROR_MESSAGES = {
    ...GENERIC_ERROR_MESSAGES,
    [constants_1.ErrorCodes.unauthorizedUser]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.unauthorizedUser],
    [constants_1.ErrorCodes.unauthorizedBundleAccess]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.unauthorizedBundleAccess],
    [constants_1.ErrorCodes.notFound]: 'Unable to found requested repository or commit hash',
};
async function createGitBundle(options, requestOptions) {
    const { baseURL, sessionToken, oAuthToken, username, gitUri, source } = options;
    const headers = { ...requestOptions === null || requestOptions === void 0 ? void 0 : requestOptions.headers, 'Session-Token': sessionToken, source };
    if (oAuthToken) {
        headers['X-OAuthToken'] = oAuthToken;
    }
    if (username) {
        headers['X-UserName'] = username;
    }
    const config = {
        headers,
        url: `${baseURL}${constants_1.apiPath}/bundle`,
        method: 'POST',
        data: { gitURI: gitUri },
    };
    try {
        const response = await axios_1.default.request(config);
        return { type: 'success', value: response.data };
    }
    catch (error) {
        return generateError(error, CREATE_GIT_BUNDLE_ERROR_MESSAGES, 'createBundle');
    }
}
exports.createGitBundle = createGitBundle;
const UPLOAD_BUNDLE_ERROR_MESSAGES = {
    ...GENERIC_ERROR_MESSAGES,
    [constants_1.ErrorCodes.unauthorizedUser]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.unauthorizedUser],
    [constants_1.ErrorCodes.unauthorizedBundleAccess]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.unauthorizedBundleAccess],
    [constants_1.ErrorCodes.notFound]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.notFound],
    [constants_1.ErrorCodes.bigPayload]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.bigPayload],
    [constants_1.ErrorCodes.unauthorizedContent]: `Invalid request, attempted to extend a git bundle, or ended up with an empty bundle after the extension`,
};
async function uploadFiles(options) {
    const { baseURL, sessionToken, bundleId, content } = options;
    const config = {
        headers: { 'Session-Token': sessionToken },
        url: `${baseURL}${constants_1.apiPath}/file/${bundleId}`,
        method: 'POST',
        data: content,
    };
    try {
        await axios_1.default.request(config);
        return { type: 'success', value: true };
    }
    catch (error) {
        return generateError(error, UPLOAD_BUNDLE_ERROR_MESSAGES, 'uploadFiles');
    }
}
exports.uploadFiles = uploadFiles;
// eslint-disable-next-line no-shadow
var AnalysisStatus;
(function (AnalysisStatus) {
    AnalysisStatus["waiting"] = "WAITING";
    AnalysisStatus["fetching"] = "FETCHING";
    AnalysisStatus["analyzing"] = "ANALYZING";
    AnalysisStatus["dcDone"] = "DC_DONE";
    AnalysisStatus["done"] = "DONE";
    AnalysisStatus["failed"] = "FAILED";
})(AnalysisStatus = exports.AnalysisStatus || (exports.AnalysisStatus = {}));
const GET_ANALYSIS_ERROR_MESSAGES = {
    ...GENERIC_ERROR_MESSAGES,
    [constants_1.ErrorCodes.unauthorizedUser]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.unauthorizedUser],
    [constants_1.ErrorCodes.unauthorizedBundleAccess]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.unauthorizedBundleAccess],
    [constants_1.ErrorCodes.notFound]: constants_1.DEFAULT_ERROR_MESSAGES[constants_1.ErrorCodes.notFound],
    [constants_1.ErrorCodes.serverError]: 'Getting analysis failed',
};
async function getAnalysis(options, requestOptions) {
    const { baseURL, sessionToken, oAuthToken, username, bundleId, includeLint, severity, limitToFiles, source, reachability, } = options;
    // ?linters=false is still a truthy query value, if(includeLint === false) we have to avoid sending the value altogether
    // the same applies for reachability
    const params = { severity, linters: includeLint || undefined, reachability: reachability || undefined };
    const headers = { ...requestOptions === null || requestOptions === void 0 ? void 0 : requestOptions.headers, 'Session-Token': sessionToken, source };
    if (oAuthToken) {
        headers['X-OAuthToken'] = oAuthToken;
    }
    if (username) {
        headers['X-UserName'] = username;
    }
    const config = {
        headers,
        params,
        url: `${baseURL}${constants_1.apiPath}/analysis/${bundleId}`,
        method: 'GET',
    };
    if (limitToFiles && limitToFiles.length) {
        config.data = { files: limitToFiles };
        config.method = 'POST';
    }
    try {
        const response = await axios_1.default.request(config);
        return { type: 'success', value: response.data };
    }
    catch (error) {
        return generateError(error, GET_ANALYSIS_ERROR_MESSAGES, 'getAnalysis');
    }
}
exports.getAnalysis = getAnalysis;
async function reportError(options) {
    const { baseURL, sessionToken, source, type, message, path, bundleId, version, environmentVersion, data } = options;
    const config = {
        url: `${baseURL}${constants_1.apiPath}/error`,
        method: 'POST',
        data: {
            sessionToken,
            source,
            type,
            message,
            path,
            bundleId,
            version,
            environmentVersion,
            data,
        },
    };
    try {
        await axios_1.default.request(config);
        return { type: 'success', value: undefined };
    }
    catch (error) {
        return generateError(error, GENERIC_ERROR_MESSAGES, 'reportError');
    }
}
exports.reportError = reportError;
async function reportEvent(options) {
    const { baseURL, sessionToken, source, type, message, path, bundleId, version, environmentVersion, data } = options;
    const config = {
        url: `${baseURL}${constants_1.apiPath}/track`,
        method: 'POST',
        data: {
            sessionToken,
            source,
            type,
            message,
            path,
            bundleId,
            version,
            environmentVersion,
            data,
        },
    };
    try {
        await axios_1.default.request(config);
        return { type: 'success', value: undefined };
    }
    catch (error) {
        return generateError(error, GENERIC_ERROR_MESSAGES, 'reportEvent');
    }
}
exports.reportEvent = reportEvent;
//# sourceMappingURL=http.js.map