"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.nodeFilesToScannedProjects = void 0;
const dep_graph_1 = require("@snyk/dep-graph");
const path = require("path");
const lockFileParser = require("snyk-nodejs-lockfile-parser");
async function nodeFilesToScannedProjects(filePathToContent) {
    const scanResults = [];
    /**
     * TODO: Add support for Yarn workspaces!
     * https://github.com/snyk/nodejs-lockfile-parser/blob/af8ba81930e950156b539281ecf41c1bc63dacf4/test/lib/yarn-workflows.test.ts#L7-L17
     *
     * When building the ScanResult ensure the workspace is stored in scanResult.identity.args:
     * args: {
     *   rootWorkspace: <path-of-workspace>,
     * };
     */
    const filePairs = findManifestLockPairsInSameDirectory(filePathToContent);
    const shouldIncludeDevDependencies = false;
    const shouldBeStrictForManifestAndLockfileOutOfSync = false;
    for (const pathPair of filePairs) {
        // TODO: initially generate as DepGraph
        const parserResult = await lockFileParser.buildDepTree(filePathToContent[pathPair.manifest], filePathToContent[pathPair.lock], shouldIncludeDevDependencies, pathPair.lockType, shouldBeStrictForManifestAndLockfileOutOfSync);
        const strippedLabelsParserResult = stripUndefinedLabels(parserResult);
        const depGraph = await dep_graph_1.legacy.depTreeToGraph(strippedLabelsParserResult, pathPair.lockType);
        const depGraphFact = {
            type: "depGraph",
            data: depGraph,
        };
        const testedFilesFact = {
            type: "testedFiles",
            data: [path.basename(pathPair.manifest), path.basename(pathPair.lock)],
        };
        scanResults.push({
            facts: [depGraphFact, testedFilesFact],
            identity: {
                type: depGraph.pkgManager.name,
                targetFile: pathPair.manifest,
            },
        });
    }
    return scanResults;
}
exports.nodeFilesToScannedProjects = nodeFilesToScannedProjects;
function findManifestLockPairsInSameDirectory(filePathToContent) {
    const fileNamesGroupedByDirectory = groupFilesByDirectory(filePathToContent);
    const manifestLockPathPairs = [];
    for (const directoryPath of Object.keys(fileNamesGroupedByDirectory)) {
        const filesInDirectory = fileNamesGroupedByDirectory[directoryPath];
        if (filesInDirectory.length !== 2) {
            // either a missing file or too many files, ignore
            continue;
        }
        const hasPackageJson = filesInDirectory.includes("package.json");
        const hasPackageLockJson = filesInDirectory.includes("package-lock.json");
        const hasYarnLock = filesInDirectory.includes("yarn.lock");
        if (hasPackageJson && hasPackageLockJson) {
            manifestLockPathPairs.push({
                manifest: path.join(directoryPath, "package.json"),
                lock: path.join(directoryPath, "package-lock.json"),
                lockType: lockFileParser.LockfileType.npm,
            });
            continue;
        }
        if (hasPackageJson && hasYarnLock) {
            manifestLockPathPairs.push({
                manifest: path.join(directoryPath, "package.json"),
                lock: path.join(directoryPath, "yarn.lock"),
                lockType: lockFileParser.LockfileType.yarn,
            });
            continue;
        }
    }
    return manifestLockPathPairs;
}
// assumption: we only care about manifest+lock files if they are in the same directory
function groupFilesByDirectory(filePathToContent) {
    const fileNamesGroupedByDirectory = {};
    for (const filePath of Object.keys(filePathToContent)) {
        const directory = path.dirname(filePath);
        const fileName = path.basename(filePath);
        if (!fileNamesGroupedByDirectory[directory]) {
            fileNamesGroupedByDirectory[directory] = [];
        }
        fileNamesGroupedByDirectory[directory].push(fileName);
    }
    return fileNamesGroupedByDirectory;
}
function stripUndefinedLabels(parserResult) {
    const optionalLabels = parserResult.labels;
    const mandatoryLabels = {};
    if (optionalLabels) {
        for (const currentLabelName of Object.keys(optionalLabels)) {
            if (optionalLabels[currentLabelName] !== undefined) {
                mandatoryLabels[currentLabelName] = optionalLabels[currentLabelName];
            }
        }
    }
    const parserResultWithProperLabels = Object.assign({}, parserResult, {
        labels: mandatoryLabels,
    });
    return parserResultWithProperLabels;
}
//# sourceMappingURL=node.js.map