"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Docker = void 0;
const snyk_docker_pull_1 = require("@snyk/snyk-docker-pull");
const Debug = require("debug");
const Modem = require("docker-modem");
const fs_1 = require("fs");
const subProcess = require("./sub-process");
const debug = Debug("snyk");
class Docker {
    static async binaryExists() {
        try {
            await subProcess.execute("docker", ["version"]);
            return true;
        }
        catch (e) {
            return false;
        }
    }
    async pull(registry, repo, tag, imageSavePath, username, password) {
        const dockerPull = new snyk_docker_pull_1.DockerPull();
        const opt = {
            username,
            password,
            loadImage: false,
            imageSavePath,
        };
        return await dockerPull.pull(registry, repo, tag, opt);
    }
    async isDockerExperimentalEnabled() {
        const output = await subProcess.execute("docker", ["system", "info"]);
        return output.stdout.includes("Experimental: true");
    }
    async pullCli(targetImage, options) {
        let platformOption = "";
        if (options === null || options === void 0 ? void 0 : options.platform) {
            if (await this.isDockerExperimentalEnabled()) {
                platformOption = `--platform=${options.platform}`;
            }
            else {
                throw new Error('"--platform" is only supported on a Docker daemon with experimental features enabled');
            }
        }
        return subProcess.execute("docker", ["pull", platformOption, targetImage]);
    }
    async save(targetImage, destination) {
        const request = {
            path: `/images/${targetImage}/get?`,
            method: "GET",
            isStream: true,
            statusCodes: {
                200: true,
                400: "bad request",
                404: "not found",
                500: "server error",
            },
        };
        debug(`Docker.save: targetImage: ${targetImage}, destination: ${destination}`);
        const modem = new Modem();
        return new Promise((resolve, reject) => {
            modem.dial(request, (err, stream) => {
                if (err) {
                    return reject(err);
                }
                const writeStream = fs_1.createWriteStream(destination);
                writeStream.on("error", (err) => {
                    reject(err);
                });
                writeStream.on("finish", () => {
                    resolve();
                });
                stream.on("error", (err) => {
                    reject(err);
                });
                stream.on("end", () => {
                    writeStream.end();
                });
                stream.pipe(writeStream);
            });
        });
    }
    async inspectImage(targetImage) {
        return subProcess.execute("docker", ["inspect", targetImage]);
    }
}
exports.Docker = Docker;
//# sourceMappingURL=docker.js.map