"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseGoBinary = void 0;
const depGraph = require("@snyk/dep-graph");
const event_loop_spinner_1 = require("event-loop-spinner");
const _1 = require("./");
const str_tab_parser_1 = require("./str-tab-parser");
const version_parser_1 = require("./version-parser");
/**
 * Parse `.strtab` to get Go packages
 * Parse `.go.buildinfo` to get addresses
 * to go modules and their versions
 * @param goBinary
 */
async function parseGoBinary(goBinary) {
    // Non-stripped binaries contain data on ".strtab" section
    const strTab = goBinary.body.sections.find((section) => section.name === ".strtab");
    // TODO: stripped binaries are not supported in this iteration
    if (!strTab) {
        return undefined;
    }
    const { name, modules } = version_parser_1.extractModulesFromBinary(goBinary);
    const packages = str_tab_parser_1.parserStrTab(strTab.data);
    // If there is no packages or modules, return empty result
    if (Object.keys(modules).length === 0 || packages.length === 0) {
        return undefined;
    }
    const packageVersionTable = matchModuleToPackage(modules, packages);
    return await createDepGraph(name, packageVersionTable);
}
exports.parseGoBinary = parseGoBinary;
/**
 * Package name consist of module name and path to package.
 * This function matches package to modules and their versions
 * @param moduleVersionTable
 * @param packages
 */
function matchModuleToPackage(moduleVersionTable, packages) {
    const resultTable = {};
    for (const pack of packages) {
        let moduleVersion = moduleVersionTable[pack];
        if (!moduleVersion) {
            const [packageModule] = Object.keys(moduleVersionTable)
                // Find all modules that this package can be from
                .filter((moduleName) => pack.startsWith(moduleName))
                // The longest string will be the closest match
                .sort((a, b) => b.length - a.length);
            if (!packageModule || !moduleVersionTable[packageModule]) {
                continue;
            }
            moduleVersion = moduleVersionTable[packageModule];
        }
        resultTable[pack] = moduleVersion;
    }
    return resultTable;
}
async function createDepGraph(name, packageVersionTable) {
    const goModulesDepGraph = new depGraph.DepGraphBuilder({ name: _1.DEP_GRAPH_TYPE }, { name });
    for (const [name, version] of Object.entries(packageVersionTable)) {
        if (event_loop_spinner_1.eventLoopSpinner.isStarving()) {
            await event_loop_spinner_1.eventLoopSpinner.spin();
        }
        const nodeId = `${name}@${version}`;
        goModulesDepGraph.addPkgNode({ name, version }, nodeId);
        goModulesDepGraph.connectDep(goModulesDepGraph.rootNodeId, nodeId);
    }
    return goModulesDepGraph.build();
}
//# sourceMappingURL=parser.js.map