"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.appendLatestTagIfMissing = exports.scan = exports.mergeEnvVarsIntoCredentials = void 0;
const fs = require("fs");
const path = require("path");
const image_inspector_1 = require("./analyzer/image-inspector");
const dockerfile_1 = require("./dockerfile");
const image_save_path_1 = require("./image-save-path");
const image_type_1 = require("./image-type");
const staticModule = require("./static");
const types_1 = require("./types");
// Registry credentials may also be provided by env vars. When both are set, flags take precedence.
function mergeEnvVarsIntoCredentials(options) {
    options.username = options.username || process.env.SNYK_REGISTRY_USERNAME;
    options.password = options.password || process.env.SNYK_REGISTRY_PASSWORD;
}
exports.mergeEnvVarsIntoCredentials = mergeEnvVarsIntoCredentials;
async function scan(options) {
    if (!options) {
        throw new Error("No plugin options provided");
    }
    mergeEnvVarsIntoCredentials(options);
    if (!options.path) {
        throw new Error("No image identifier or path provided");
    }
    const targetImage = appendLatestTagIfMissing(options.path);
    const dockerfilePath = options.file;
    const dockerfileAnalysis = await dockerfile_1.readDockerfileAndAnalyse(dockerfilePath);
    const imageType = image_type_1.getImageType(targetImage);
    switch (imageType) {
        case types_1.ImageType.DockerArchive:
        case types_1.ImageType.OciArchive:
            return localArchiveAnalysis(targetImage, imageType, dockerfileAnalysis, options);
        case types_1.ImageType.Identifier:
            return imageIdentifierAnalysis(targetImage, imageType, dockerfileAnalysis, options);
        default:
            throw new Error("Unhandled image type for image " + targetImage);
    }
}
exports.scan = scan;
async function localArchiveAnalysis(targetImage, imageType, dockerfileAnalysis, options) {
    var _a, _b;
    const excludeBaseImageVulns = isTrue(options["exclude-base-image-vulns"]);
    const appScan = isTrue(options["app-vulns"]);
    const globToFind = {
        include: ((_a = options.globsToFind) === null || _a === void 0 ? void 0 : _a.include) || [],
        exclude: ((_b = options.globsToFind) === null || _b === void 0 ? void 0 : _b.exclude) || [],
    };
    const archivePath = image_type_1.getArchivePath(targetImage);
    if (!fs.existsSync(archivePath)) {
        throw new Error("The provided archive path does not exist on the filesystem");
    }
    if (!fs.lstatSync(archivePath).isFile()) {
        throw new Error("The provided archive path is not a file");
    }
    const imageIdentifier = options.imageNameAndTag ||
        // The target image becomes the base of the path, e.g. "archive.tar" for "/var/tmp/archive.tar"
        path.basename(archivePath);
    return await staticModule.analyzeStatically(imageIdentifier, dockerfileAnalysis, imageType, archivePath, excludeBaseImageVulns, globToFind, appScan);
}
async function imageIdentifierAnalysis(targetImage, imageType, dockerfileAnalysis, options) {
    var _a, _b;
    const excludeBaseImageVulns = isTrue(options["exclude-base-image-vulns"]);
    const appScan = isTrue(options["app-vulns"]);
    const globToFind = {
        include: ((_a = options.globsToFind) === null || _a === void 0 ? void 0 : _a.include) || [],
        exclude: ((_b = options.globsToFind) === null || _b === void 0 ? void 0 : _b.exclude) || [],
    };
    const imageSavePath = image_save_path_1.fullImageSavePath(options.imageSavePath);
    const archiveResult = await image_inspector_1.getImageArchive(targetImage, imageSavePath, options.username, options.password, options.platform);
    const imagePath = archiveResult.path;
    try {
        return await staticModule.analyzeStatically(targetImage, dockerfileAnalysis, imageType, imagePath, excludeBaseImageVulns, globToFind, appScan);
    }
    finally {
        archiveResult.removeArchive();
    }
}
function isTrue(value) {
    return String(value).toLowerCase() === "true";
}
function appendLatestTagIfMissing(targetImage) {
    if (image_type_1.getImageType(targetImage) === types_1.ImageType.Identifier &&
        !targetImage.includes(":")) {
        return `${targetImage}:latest`;
    }
    return targetImage;
}
exports.appendLatestTagIfMissing = appendLatestTagIfMissing;
//# sourceMappingURL=scan.js.map