"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.scan = void 0;
const path = require("upath");
const fs = require("fs");
const mix_parser_1 = require("@snyk/mix-parser");
const subProcess = require("./sub-process");
const debug_1 = require("./debug");
const copy_elixir_code_to_temp_dir_1 = require("./copy-elixir-code-to-temp-dir");
const MANIFEST_FILE_NAME = 'mix.exs';
async function scan(options) {
    debug_1.init(options.debug);
    const targetFile = path.parse(path.resolve(options.path, options.targetFile || MANIFEST_FILE_NAME));
    if (targetFile.base !== MANIFEST_FILE_NAME) {
        throw new Error("target file must be 'mix.exs'.");
    }
    await verifyMixInstalled();
    const mixResult = await getMixResult(targetFile.dir);
    const depGraphMap = mix_parser_1.buildDepGraphs(mixResult, !!options.dev, true, options.allProjects);
    const scanResults = Object.entries(depGraphMap).map(([name, depGraph], index) => {
        const isRoot = index === 0;
        const relativePathToManifest = getRelativePathToManifest(options, targetFile, isRoot, name);
        return {
            identity: {
                type: 'hex',
                targetFile: relativePathToManifest,
            },
            facts: [
                {
                    type: 'depGraph',
                    data: depGraph,
                },
            ],
            ...(options.projectName
                ? { name: getProjectNamePath(options, relativePathToManifest) }
                : {}),
        };
    });
    return { scanResults };
}
exports.scan = scan;
async function verifyMixInstalled() {
    try {
        const mixVersion = await subProcess.execute('mix', ['-v']);
        debug_1.debug(`mix version: `, mixVersion);
    }
    catch (_a) {
        throw new Error('mix is not installed. please make sure Elixir is installed and try again.');
    }
}
async function getMixResult(root) {
    const elixirTmpDir = copy_elixir_code_to_temp_dir_1.copyElixirCodeToTempDir();
    const cwd = elixirTmpDir.name;
    let filePath;
    try {
        const output = await subProcess.execute('mix', ['read.mix', root], { cwd });
        debug_1.debug(`read.mix output: ${output}`);
        const fileName = output.trim().split('\n').pop();
        debug_1.debug(`fileName: ${fileName}`);
        filePath = path.resolve(cwd, fileName);
        const result = (await fs.promises.readFile(filePath, 'utf8'));
        return JSON.parse(result);
    }
    catch (err) {
        const errorMessage = `Error parsing manifest file on ${root}`;
        debug_1.debug(errorMessage, err);
        throw new Error(errorMessage);
    }
    finally {
        try {
            elixirTmpDir.removeCallback();
        }
        catch (err) {
            debug_1.debug(`can't remove ${elixirTmpDir.name}`, err);
        }
    }
}
function normalizePath(filePath) {
    const parts = filePath.split(path.sep);
    return parts.join(path.posix.sep);
}
function getRelativePathToManifest(options, targetFile, isRoot, name) {
    return normalizePath(path.relative(options.path, path.resolve(targetFile.dir, isRoot ? '' : name, targetFile.base)));
}
function getProjectNamePath(options, relativePathToManifest) {
    return [
        options.projectName,
        ...relativePathToManifest.split('/').slice(0, -1),
    ].join('/');
}
//# sourceMappingURL=scan.js.map