"use strict";
const Debug = require("debug");
const snykFix = require("@snyk/fix");
const ora = require("ora");
const snyk = require("../../../lib");
const convert_legacy_tests_results_to_fix_entities_1 = require("./convert-legacy-tests-results-to-fix-entities");
const format_test_error_1 = require("../test/format-test-error");
const process_command_args_1 = require("../process-command-args");
const validate_credentials_1 = require("../test/validate-credentials");
const validate_test_options_1 = require("../test/validate-test-options");
const set_default_test_options_1 = require("../test/set-default-test-options");
const validate_fix_command_is_supported_1 = require("./validate-fix-command-is-supported");
const get_display_path_1 = require("./get-display-path");
const chalk_1 = require("chalk");
const debug = Debug('snyk-fix');
const snykFixFeatureFlag = 'cliSnykFix';
async function fix(...args) {
    const { options: rawOptions, paths } = await process_command_args_1.processCommandArgs(...args);
    const options = set_default_test_options_1.setDefaultTestOptions(rawOptions);
    debug(options);
    await validate_fix_command_is_supported_1.validateFixCommandIsSupported(options);
    validate_test_options_1.validateTestOptions(options);
    validate_credentials_1.validateCredentials(options);
    const results = [];
    results.push(...(await runSnykTestLegacy(options, paths)));
    // fix
    debug(`Organization has ${snykFixFeatureFlag} feature flag enabled for experimental Snyk fix functionality`);
    const vulnerableResults = results.filter((res) => Object.keys(res.testResult.issues).length);
    const { dryRun, quiet } = options;
    const { fixSummary, meta } = await snykFix.fix(results, { dryRun, quiet });
    // `snyk test` did not return any test results
    if (results.length === 0) {
        throw new Error(fixSummary);
    }
    // `snyk test` returned no vulnerable results, so nothing to fix
    if (vulnerableResults.length === 0) {
        return fixSummary;
    }
    // `snyk test` returned vulnerable results
    // however some errors occurred during `snyk fix` and nothing was fixed in the end
    const anyFailed = meta.failed > 0;
    const noneFixed = meta.fixed === 0;
    if (anyFailed && noneFixed) {
        throw new Error(fixSummary);
    }
    return fixSummary;
}
/* @deprecated
 * TODO: once project envelope is default all code below will be deleted
 * we should be calling test via new Ecosystems instead
 */
async function runSnykTestLegacy(options, paths) {
    const results = [];
    const stdOutSpinner = ora({
        isSilent: options.quiet,
        stream: process.stdout,
    });
    const stdErrSpinner = ora({
        isSilent: options.quiet,
        stream: process.stdout,
    });
    stdErrSpinner.start();
    stdOutSpinner.start();
    for (const path of paths) {
        let displayPath = path;
        const spinnerMessage = `Running \`snyk test\` for ${displayPath}`;
        try {
            displayPath = get_display_path_1.getDisplayPath(path);
            stdOutSpinner.text = spinnerMessage;
            stdOutSpinner.render();
            // Create a copy of the options so a specific test can
            // modify them i.e. add `options.file` etc. We'll need
            // these options later.
            const snykTestOptions = Object.assign(Object.assign({}, options), { path, projectName: options['project-name'] });
            const testResults = [];
            const testResultForPath = await snyk.test(path, Object.assign(Object.assign({}, snykTestOptions), { quiet: true }));
            testResults.push(...(Array.isArray(testResultForPath)
                ? testResultForPath
                : [testResultForPath]));
            const newRes = convert_legacy_tests_results_to_fix_entities_1.convertLegacyTestResultToFixEntities(testResults, path, options);
            results.push(...newRes);
            stdOutSpinner.stopAndPersist({
                text: spinnerMessage,
                symbol: '\n►',
            });
        }
        catch (error) {
            const testError = format_test_error_1.formatTestError(error);
            const userMessage = chalk_1.default.red(`Failed! ${testError.message}.`) +
                `\n  Tip: run \`snyk test ${displayPath} -d\` for more information.`;
            stdOutSpinner.stopAndPersist({
                text: spinnerMessage,
                symbol: '\n►',
            });
            stdErrSpinner.stopAndPersist({
                text: userMessage,
                symbol: chalk_1.default.red(' '),
            });
            debug(userMessage);
        }
    }
    stdOutSpinner.stop();
    stdErrSpinner.stop();
    return results;
}
module.exports = fix;
//# sourceMappingURL=index.js.map