"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FailedToExecutePolicyEngine = exports.FailedToBuildPolicyEngine = exports.clearPolicyEngineCache = exports.scanFiles = void 0;
const types_1 = require("./types");
const opa_wasm_1 = require("@open-policy-agent/opa-wasm");
const fs = require("fs");
const local_cache_1 = require("./local-cache");
const errors_1 = require("../../../../lib/errors");
const error_utils_1 = require("./error-utils");
async function scanFiles(parsedFiles) {
    // TODO: gracefully handle failed scans
    const scanResults = [];
    for (const parsedFile of parsedFiles) {
        const policyEngine = await getPolicyEngine(parsedFile.engineType);
        const result = policyEngine.scanFile(parsedFile);
        scanResults.push(result);
    }
    return scanResults;
}
exports.scanFiles = scanFiles;
async function getPolicyEngine(engineType) {
    if (policyEngineCache[engineType]) {
        return policyEngineCache[engineType];
    }
    policyEngineCache[engineType] = await buildPolicyEngine(engineType);
    return policyEngineCache[engineType];
}
// used in tests only
function clearPolicyEngineCache() {
    policyEngineCache = {
        [types_1.EngineType.Kubernetes]: null,
        [types_1.EngineType.Terraform]: null,
        [types_1.EngineType.CloudFormation]: null,
        [types_1.EngineType.Custom]: null,
    };
}
exports.clearPolicyEngineCache = clearPolicyEngineCache;
let policyEngineCache = {
    [types_1.EngineType.Kubernetes]: null,
    [types_1.EngineType.Terraform]: null,
    [types_1.EngineType.CloudFormation]: null,
    [types_1.EngineType.Custom]: null,
};
async function buildPolicyEngine(engineType) {
    const [policyEngineCoreDataPath, policyEngineMetaDataPath,] = local_cache_1.getLocalCachePath(engineType);
    try {
        const wasmFile = fs.readFileSync(policyEngineCoreDataPath);
        const policyMetaData = fs.readFileSync(policyEngineMetaDataPath);
        const policyMetadataAsJson = JSON.parse(policyMetaData.toString());
        const opaWasmInstance = await opa_wasm_1.loadPolicy(Buffer.from(wasmFile));
        opaWasmInstance.setData(policyMetadataAsJson);
        return new PolicyEngine(opaWasmInstance);
    }
    catch (err) {
        throw new FailedToBuildPolicyEngine();
    }
}
class PolicyEngine {
    constructor(opaWasmInstance) {
        this.opaWasmInstance = opaWasmInstance;
        this.opaWasmInstance = opaWasmInstance;
    }
    evaluate(data) {
        return this.opaWasmInstance.evaluate(data)[0].result;
    }
    scanFile(iacFile) {
        try {
            const violatedPolicies = this.evaluate(iacFile.jsonContent);
            return Object.assign(Object.assign({}, iacFile), { violatedPolicies });
        }
        catch (err) {
            // TODO: to distinguish between different failure reasons
            throw new FailedToExecutePolicyEngine();
        }
    }
}
class FailedToBuildPolicyEngine extends errors_1.CustomError {
    constructor(message) {
        super(message || 'Failed to build policy engine');
        this.code = types_1.IaCErrorCodes.FailedToBuildPolicyEngine;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage =
            'We were unable run the test. Please run the command again with the `-d` flag and contact support@snyk.io with the contents of the output.';
    }
}
exports.FailedToBuildPolicyEngine = FailedToBuildPolicyEngine;
class FailedToExecutePolicyEngine extends errors_1.CustomError {
    constructor(message) {
        super(message || 'Failed to execute policy engine');
        this.code = types_1.IaCErrorCodes.FailedToExecutePolicyEngine;
        this.strCode = error_utils_1.getErrorStringCode(this.code);
        this.userMessage =
            'We were unable run the test. Please run the command again with the `-d` flag and contact support@snyk.io with the contents of the output.';
    }
}
exports.FailedToExecutePolicyEngine = FailedToExecutePolicyEngine;
//# sourceMappingURL=file-scanner.js.map