"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getWorkspacesMap = exports.processYarnWorkspaces = void 0;
const baseDebug = require("debug");
const pathUtil = require("path");
// import * as _ from 'lodash';
const sortBy = require('lodash.sortby');
const groupBy = require('lodash.groupby');
const micromatch = require("micromatch");
const debug = baseDebug('snyk-yarn-workspaces');
const lockFileParser = require("snyk-nodejs-lockfile-parser");
const path = require("path");
const errors_1 = require("../../errors");
const get_file_contents_1 = require("../../get-file-contents");
async function processYarnWorkspaces(root, settings, targetFiles) {
    // the order of folders is important
    // must have the root level most folders at the top
    // const yarnTargetFiles: {
    //   [dir: string]: Array<{
    //     path: string;
    //     base: string;
    //     dir: string;
    //   }>;
    // } = targetFiles
    //   .map((p) => ({ path: p, ...pathUtil.parse(p) }))
    //   .filter((res) => ['package.json'].includes(res.base))
    //   .sortBy('dir')
    //   .groupBy('dir')
    //   .value();
    const mappedAndFiltered = targetFiles
        .map((p) => (Object.assign({ path: p }, pathUtil.parse(p))))
        .filter((res) => ['package.json'].includes(res.base));
    const sorted = sortBy(mappedAndFiltered, 'dir');
    const grouped = groupBy(sorted, 'dir');
    const yarnTargetFiles = grouped;
    debug(`Processing potential Yarn workspaces (${targetFiles.length})`);
    if (Object.keys(yarnTargetFiles).length === 0) {
        throw errors_1.NoSupportedManifestsFoundError([root]);
    }
    let yarnWorkspacesMap = {};
    const yarnWorkspacesFilesMap = {};
    const result = {
        plugin: {
            name: 'snyk-nodejs-yarn-workspaces',
            runtime: process.version,
        },
        scannedProjects: [],
    };
    // the folders must be ordered highest first
    for (const directory of Object.keys(yarnTargetFiles)) {
        let isYarnWorkspacePackage = false;
        let isRootPackageJson = false;
        const packageJsonFileName = pathUtil.join(directory, 'package.json');
        const packageJson = get_file_contents_1.getFileContents(root, packageJsonFileName);
        yarnWorkspacesMap = Object.assign(Object.assign({}, yarnWorkspacesMap), getWorkspacesMap(packageJson));
        for (const workspaceRoot of Object.keys(yarnWorkspacesMap)) {
            const match = packageJsonBelongsToWorkspace(packageJsonFileName, yarnWorkspacesMap, workspaceRoot);
            if (match) {
                debug(`${packageJsonFileName} matches an existing workspace pattern`);
                yarnWorkspacesFilesMap[packageJsonFileName] = {
                    root: workspaceRoot,
                };
                isYarnWorkspacePackage = true;
            }
            if (packageJsonFileName === workspaceRoot) {
                isRootPackageJson = true;
            }
        }
        if (isYarnWorkspacePackage || isRootPackageJson) {
            const rootDir = isYarnWorkspacePackage
                ? path.dirname(yarnWorkspacesFilesMap[packageJsonFileName].root)
                : path.dirname(packageJsonFileName);
            const rootYarnLockfileName = path.join(rootDir, 'yarn.lock');
            const yarnLock = await get_file_contents_1.getFileContents(root, rootYarnLockfileName);
            const res = await lockFileParser.buildDepTree(packageJson.content, yarnLock.content, settings.dev, lockFileParser.LockfileType.yarn, settings.strictOutOfSync !== false);
            const project = {
                packageManager: 'yarn',
                targetFile: path.relative(root, packageJson.fileName),
                depTree: res,
                plugin: {
                    name: 'snyk-nodejs-lockfile-parser',
                    runtime: process.version,
                },
            };
            result.scannedProjects.push(project);
        }
        else {
            debug(`${packageJsonFileName} is not part of any detected workspace, skipping`);
        }
    }
    if (!result.scannedProjects.length) {
        debug(`No yarn workspaces detected in any of the ${targetFiles.length} target files.`);
    }
    return result;
}
exports.processYarnWorkspaces = processYarnWorkspaces;
function getWorkspacesMap(file) {
    const yarnWorkspacesMap = {};
    if (!file) {
        return yarnWorkspacesMap;
    }
    try {
        const rootFileWorkspacesDefinitions = lockFileParser.getYarnWorkspaces(file.content);
        if (rootFileWorkspacesDefinitions && rootFileWorkspacesDefinitions.length) {
            yarnWorkspacesMap[file.fileName] = {
                workspaces: rootFileWorkspacesDefinitions,
            };
        }
    }
    catch (e) {
        debug('Failed to process a workspace', e.message);
    }
    return yarnWorkspacesMap;
}
exports.getWorkspacesMap = getWorkspacesMap;
function packageJsonBelongsToWorkspace(packageJsonFileName, yarnWorkspacesMap, workspaceRoot) {
    const workspaceRootFolder = path.dirname(workspaceRoot);
    const workspacesGlobs = (yarnWorkspacesMap[workspaceRoot].workspaces || []).map((p) => path.join(workspaceRootFolder, p));
    const filePath = path.normalize(packageJsonFileName);
    const match = micromatch.isMatch(filePath, workspacesGlobs.map((p) => (p.endsWith('/**') ? p : p + '/**')));
    return match;
}
//# sourceMappingURL=yarn-workspaces-parser.js.map