"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildDepGraphs = void 0;
const dep_graph_1 = require("@snyk/dep-graph");
const out_of_sync_error_1 = require("./out-of-sync-error");
const is_dev_dependency_1 = require("./is-dev-dependency");
const get_non_hex_version_1 = require("./get-non-hex-version");
function buildDepGraphs(mixJsonResult, includeDev = false, strict = true, allProjects = false) {
    const manifest = mixJsonResult === null || mixJsonResult === void 0 ? void 0 : mixJsonResult.manifest;
    if (!manifest)
        throw new Error('No manifest found');
    const lock = mixJsonResult.lock[0];
    if (!lock)
        throw new Error('No lock file found');
    const lockDepMap = Object.entries(lock).reduce((acc, [key, dep]) => {
        const [packageManager, name, version, hash, , dependencies] = dep;
        acc[key] = {
            packageManager,
            name,
            version,
            hash,
            dependencies,
        };
        return acc;
    }, {});
    if (mixJsonResult.parent_umbrella_manifest) {
        const umbrella = mixJsonResult.parent_umbrella_manifest;
        const prefix = `${getManifestName(umbrella)}/${umbrella.apps_path}/`;
        const name = `${umbrella.apps_path}/${getManifestName(manifest)}`;
        return {
            [name]: getDepGraph(prefix, manifest, includeDev, lockDepMap, strict),
        };
    }
    const apps = allProjects || isEmpty(mixJsonResult.apps) ? {} : mixJsonResult.apps;
    const projects = { root: manifest, ...apps };
    const appsPrefix = `${getManifestName(manifest)}/${manifest.apps_path}/`;
    return Object.entries(projects).reduce((acc, [key, manifest]) => {
        const prefix = key === 'root' ? '' : appsPrefix;
        acc[key] = getDepGraph(prefix, manifest, includeDev, lockDepMap, strict);
        return acc;
    }, {});
}
exports.buildDepGraphs = buildDepGraphs;
function isEmpty(obj) {
    return !obj || Object.keys(obj).length === 0;
}
function getDepGraph(prefix, manifest, includeDev, lockDepMap, strict) {
    const builder = new dep_graph_1.DepGraphBuilder({ name: 'hex' }, getRootPkg(prefix, manifest));
    if (!manifest.deps)
        return builder.build();
    const transitivesQueue = [];
    const deps = getTopLevelDeps(manifest);
    for (const topLevelDep of deps) {
        // eslint-disable-next-line prefer-const
        let [depName, depVersionSpec, options] = topLevelDep;
        if (typeof depVersionSpec === 'object') {
            options = depVersionSpec;
            depVersionSpec = undefined;
        }
        const isDev = is_dev_dependency_1.isDevDependency(options);
        if (!includeDev && isDev)
            continue;
        const scope = isDev ? 'dev' : 'prod';
        const parentNodeId = builder.rootNodeId;
        let dep = lockDepMap[depName];
        let labels;
        if (!dep) {
            const nonHexVersion = get_non_hex_version_1.getNonHexVersion(options);
            if (!nonHexVersion && strict)
                throw new out_of_sync_error_1.OutOfSyncError(depName);
            labels = {
                missingLockFileEntry: 'true',
                ...((nonHexVersion === null || nonHexVersion === void 0 ? void 0 : nonHexVersion.labels) || {}),
            };
            dep = { name: depName, version: (nonHexVersion === null || nonHexVersion === void 0 ? void 0 : nonHexVersion.title) || depVersionSpec };
        }
        transitivesQueue.push({ dep, parentNodeId, scope, labels });
    }
    while (transitivesQueue.length > 0) {
        const { dep, parentNodeId, scope, labels } = transitivesQueue.shift();
        const nodeId = addNode(dep.name, dep.version, scope, labels);
        builder.connectDep(parentNodeId, nodeId);
        if (!dep.dependencies)
            continue;
        for (const [childName, , childOptions] of dep.dependencies) {
            const lockDep = lockDepMap[childName];
            if (!lockDep && childOptions && !childOptions.optional)
                throw new out_of_sync_error_1.OutOfSyncError(childName);
            else if (!lockDep)
                continue;
            transitivesQueue.push({ parentNodeId: nodeId, dep: lockDep, scope });
        }
    }
    return builder.build();
    function addNode(name, version, scope, labels) {
        const nodeInfo = {
            labels: {
                scope,
                ...(labels || {}),
            },
        };
        const nodeId = `${name}@${version || ''}`;
        builder.addPkgNode({ name, version }, nodeId, nodeInfo);
        return nodeId;
    }
}
function getRootPkg(prefix, manifest) {
    const name = getManifestName(manifest);
    return { name: `${prefix}${name}`, version: manifest.version || '0.0.0' };
}
function getManifestName(manifest) {
    var _a;
    return (manifest.app || ((_a = manifest.module_name) === null || _a === void 0 ? void 0 : _a.replace(/\.Mix\w{4,}$/, '').toLowerCase()) ||
        'no_name');
}
function getTopLevelDeps(manifest) {
    return Array.isArray(manifest.deps)
        ? manifest.deps
        : Object.entries(manifest.deps).map(([key, value]) => Array.isArray(value) ? [key, ...value] : [key, value]);
}
//# sourceMappingURL=build-dep-graphs.js.map