"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAuthTokenForEndpoint = void 0;
const contentTypes = require("./content-types");
const registry_call_1 = require("./registry-call");
const needle_1 = require("./needle");
/**
 * getAuthTokenForEndpoint attempts to retrieve a token for an unauthorized endpoint.
 * It will make a request to the endpoint and in case it returns an Unauthorized (401)
 * response code with the 'www-Authenticate' header, it will make a consecutive call according
 * to the details within this header.
 * More info: https://docs.docker.com/registry/spec/auth/token/#how-to-authenticate
 * @param registryBase - the container registry base URL
 * @param endpoint - the endpoint to which we would like to make the request
 * @param username - the username for authentication
 * @param password - the password for authentication
 * @param reqOptions - http request options
 * @returns an auth token string if authentication succeeded or null if
 * authorization is not required/not bases on bearer token
 * @throws an error if http error status code is not 401
 */
async function getAuthTokenForEndpoint(registryBase, endpoint, username, password, reqOptions = {}) {
    const reqConfig = registry_call_1.buildUnauthenticatedV2RequestConfig(registryBase, endpoint, contentTypes.JSON, reqOptions);
    try {
        await needle_1.needleWrapper(reqConfig);
        return null;
    }
    catch (err) {
        if (err.statusCode !== 401) {
            throw err;
        }
        const challengeHeaders = err.headers["www-authenticate"];
        if (!challengeHeaders) {
            return null;
        }
        const [authBase, service, scope] = registry_call_1.parseChallengeHeaders(challengeHeaders);
        if (!authBase) {
            return null;
        }
        else {
            // Bearer token
            const token = await registry_call_1.getToken(registryBase, authBase, service, scope, username, password, reqOptions);
            return token;
        }
    }
}
exports.getAuthTokenForEndpoint = getAuthTokenForEndpoint;
//# sourceMappingURL=get-auth-token-for-endpoint.js.map