"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractImageLayer = void 0;
const Debug = require("debug");
const gunzip = require("gunzip-maybe");
const path = require("path");
const tar_stream_1 = require("tar-stream");
const callbacks_1 = require("./callbacks");
const debug = Debug("snyk");
/**
 * Extract key files from the specified TAR stream.
 * @param layerTarStream image layer as a Readable TAR stream. Note: consumes the stream.
 * @param extractActions array of pattern, callbacks pairs
 * @returns extracted file products
 */
async function extractImageLayer(layerTarStream, extractActions) {
    return new Promise((resolve, reject) => {
        const result = {};
        const tarExtractor = tar_stream_1.extract();
        tarExtractor.on("entry", async (headers, stream, next) => {
            if (headers.type === "file") {
                const absoluteFileName = path.join(path.sep, headers.name);
                const matchedActions = extractActions.filter((action) => action.filePathMatches(absoluteFileName));
                if (matchedActions.length > 0) {
                    try {
                        const callbackResult = await callbacks_1.applyCallbacks(matchedActions, stream, headers.size);
                        if (!callbacks_1.isResultEmpty(callbackResult)) {
                            result[absoluteFileName] = callbackResult;
                        }
                    }
                    catch (error) {
                        // An ExtractAction has thrown an uncaught exception, likely a bug in the code!
                        debug("Exception thrown while applying callbacks during image layer extraction", JSON.stringify(error));
                        reject(error);
                    }
                }
            }
            stream.resume(); // auto drain the stream
            next(); // ready for next entry
        });
        tarExtractor.on("finish", () => {
            // all layer level entries read
            resolve(result);
        });
        tarExtractor.on("error", (error) => reject(error));
        layerTarStream.pipe(gunzip()).pipe(tarExtractor);
    });
}
exports.extractImageLayer = extractImageLayer;
//# sourceMappingURL=layer.js.map