"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PathHelpers = void 0;

var _lodash = require("lodash");

var PathHelpers = {
  circle: function circle(x, y, size) {
    return "M ".concat(x, ", ").concat(y, "\n      m ").concat(-size, ", 0\n      a ").concat(size, ", ").concat(size, " 0 1,0 ").concat(size * 2, ",0\n      a ").concat(size, ", ").concat(size, " 0 1,0 ").concat(-size * 2, ",0");
  },
  dash: function dash(x, y, size) {
    var baseSize = 1.1 * size;
    var lineHeight = baseSize - baseSize * 0.3;
    var x0 = x - baseSize;
    var y1 = y + lineHeight / 2;
    var distance = (x + baseSize - x0) * 0.3;
    var padding = distance / 3;
    return "M ".concat(x0, ", ").concat(y1, "\n      h").concat(distance, "\n      v-").concat(lineHeight, "\n      h-").concat(distance, "\n      z\n      M ").concat(x0 + distance + padding, ", ").concat(y1, "\n      h").concat(distance, "\n      v-").concat(lineHeight, "\n      h-").concat(distance, "\n      z\n      M ").concat(x0 + distance * 2 + padding * 2, ", ").concat(y1, "\n      h").concat(distance, "\n      v-").concat(lineHeight, "\n      h-").concat(distance, "\n      z");
  },
  diamond: function diamond(x, y, size) {
    var baseSize = 0.87 * size;
    var length = Math.sqrt(2 * (baseSize * baseSize));
    return "M ".concat(x, ", ").concat(y + length, "\n      l ").concat(length, ", -").concat(length, "\n      l -").concat(length, ", -").concat(length, "\n      l -").concat(length, ", ").concat(length, "\n      l ").concat(length, ", ").concat(length, "\n      z");
  },
  // The eye slash symbol is used to hide / show each data series individually
  // See https://github.com/patternfly/patternfly-next/issues/2352
  eyeSlash: function eyeSlash(x, y, size) {
    var baseSize = 0.87 * size;
    var x0 = x - baseSize;
    var y1 = y - size * 1.25; // For converting to relative paths, see https://aydos.com/svgedit/
    // Data taken from path of assets/Eyecon.svg

    var data = '.013 .013 0 0 2.179 2.219c.7-.204 1.418-.307 2.152-.307 2.859 0 5.464 1.551 7.814 4.654.243 .321.268 .753.073 1.097l-.073.111-.236.305c-.632.801-1.282 1.491-1.951 2.071l1.773 1.806c.382.389 .382 1.012 0 1.401l-.058.059c-.387.394-1.02.4-1.414.013l-.013-.013-11.732-11.956c-.382-.389-.382-1.012 0-1.401l.058-.059c.387-.394 1.02-.4 1.414-.013zm-.674 3.71 1.407 1.436c-.329.604-.516 1.298-.516 2.038 0 2.323 1.848 4.206 4.127 4.206.726 0 1.408-.191 2-.526l.966.984c-.956.396-1.945.593-2.966.593-2.859 0-5.464-1.551-7.814-4.654-.243-.321-.268-.753-.073-1.097l.073-.111.236-.305c.823-1.042 1.676-1.897 2.56-2.565zm2.177 2.22 4.072 4.149c-.377.167-.793.259-1.23.259-1.71 0-3.096-1.412-3.096-3.155 0-.445.091-.869.254-1.253zm2.842-2.953c-.43 0-.845.067-1.234.191l.865.882c.121-.015.244-.022.369-.022 1.71 0 3.096 1.412 3.096 3.155 0 .127-.007.252-.022.375l.866.882c.122-.397.187-.819.187-1.257 0-2.323-1.848-4.206-4.127-4.206z';
    return "m".concat(x0, ", ").concat(y1, " ").concat(data);
  },
  minus: function minus(x, y, size) {
    var baseSize = 1.1 * size;
    var lineHeight = baseSize - baseSize * 0.3;
    var x0 = x - baseSize;
    var y1 = y + lineHeight / 2;
    var distance = x + baseSize - x0;
    return "M ".concat(x0, ", ").concat(y1, "\n      h").concat(distance, "\n      v-").concat(lineHeight, "\n      h-").concat(distance, "\n      z");
  },
  plus: function plus(x, y, size) {
    var baseSize = 1.1 * size;
    var distance = baseSize / 1.5;
    return "\n      M ".concat(x - distance / 2, ", ").concat(y + baseSize, "\n      v-").concat(distance, "\n      h-").concat(distance, "\n      v-").concat(distance, "\n      h").concat(distance, "\n      v-").concat(distance, "\n      h").concat(distance, "\n      v").concat(distance, "\n      h").concat(distance, "\n      v").concat(distance, "\n      h-").concat(distance, "\n      v").concat(distance, "\n      z");
  },
  star: function star(x, y, size) {
    var baseSize = 1.35 * size;
    var angle = Math.PI / 5;
    var starCoords = (0, _lodash.range)(10).map(function (index) {
      var length = index % 2 === 0 ? baseSize : baseSize / 2;
      return "".concat(length * Math.sin(angle * (index + 1)) + x, ",\n        ").concat(length * Math.cos(angle * (index + 1)) + y);
    });
    return "M ".concat(starCoords.join('L'), " z");
  },
  square: function square(x, y, size) {
    var baseSize = 0.87 * size;
    var x0 = x - baseSize;
    var y1 = y + baseSize;
    var distance = x + baseSize - x0;
    return "M ".concat(x0, ", ").concat(y1, "\n      h").concat(distance, "\n      v-").concat(distance, "\n      h-").concat(distance, "\n      z");
  },
  threshold: function threshold(x, y, size) {
    var baseSize = 1.1 * size;
    var lineHeight = baseSize - baseSize * 0.3;
    var x0 = x - baseSize;
    var y1 = y + lineHeight / 2;
    var distance = (x + baseSize - x0) * 0.5;
    var padding = distance / 3;
    return "M ".concat(x0, ", ").concat(y1, "\n      h").concat(distance, "\n      v-").concat(lineHeight, "\n      h-").concat(distance, "\n      z\n      M ").concat(x0 + distance + padding, ", ").concat(y1, "\n      h").concat(distance, "\n      v-").concat(lineHeight, "\n      h-").concat(distance, "\n      z");
  },
  triangleDown: function triangleDown(x, y, size) {
    var height = size / 2 * Math.sqrt(3);
    var x0 = x - size;
    var x1 = x + size;
    var y0 = y - size;
    var y1 = y + height;
    return "M ".concat(x0, ", ").concat(y0, "\n      L ").concat(x1, ", ").concat(y0, "\n      L ").concat(x, ", ").concat(y1, "\n      z");
  },
  triangleUp: function triangleUp(x, y, size) {
    var height = size / 2 * Math.sqrt(3);
    var x0 = x - size;
    var x1 = x + size;
    var y0 = y - height;
    var y1 = y + size;
    return "M ".concat(x0, ", ").concat(y1, "\n      L ").concat(x1, ", ").concat(y1, "\n      L ").concat(x, ", ").concat(y0, "\n      z");
  }
};
exports.PathHelpers = PathHelpers;
//# sourceMappingURL=path-helpers.js.map