/**
 * @fileoverview Forbid certain props on components
 * @author Joe Lencioni
 */
'use strict';

const docsUrl = require('../util/docsUrl');

// ------------------------------------------------------------------------------
// Constants
// ------------------------------------------------------------------------------

const DEFAULTS = ['className', 'style'];

// ------------------------------------------------------------------------------
// Rule Definition
// ------------------------------------------------------------------------------

module.exports = {
  meta: {
    docs: {
      description: 'Forbid certain props on components',
      category: 'Best Practices',
      recommended: false,
      url: docsUrl('forbid-component-props')
    },

    schema: [{
      type: 'object',
      properties: {
        forbid: {
          type: 'array',
          items: {
            oneOf: [{
              type: 'string'
            }, {
              type: 'object',
              properties: {
                propName: {
                  type: 'string'
                },
                allowedFor: {
                  type: 'array',
                  uniqueItems: true,
                  items: {
                    type: 'string'
                  }
                }
              }
            }]
          }
        }
      }
    }]
  },

  create: function(context) {
    const configuration = context.options[0] || {};
    const forbid = new Map((configuration.forbid || DEFAULTS).map(value => {
      const propName = typeof value === 'string' ? value : value.propName;
      const whitelist = typeof value === 'string' ? [] : (value.allowedFor || []);
      return [propName, whitelist];
    }));

    function isForbidden(prop, tagName) {
      const whitelist = forbid.get(prop);
      // if the tagName is undefined (`<this.something>`), we assume it's a forbidden element
      return typeof whitelist !== 'undefined' && (typeof tagName === 'undefined' || whitelist.indexOf(tagName) === -1);
    }

    return {
      JSXAttribute: function(node) {
        const tag = node.parent.name.name;
        if (tag && tag[0] !== tag[0].toUpperCase()) {
          // This is a DOM node, not a Component, so exit.
          return;
        }

        const prop = node.name.name;

        if (!isForbidden(prop, tag)) {
          return;
        }

        context.report({
          node: node,
          message: `Prop \`${prop}\` is forbidden on Components`
        });
      }
    };
  }
};
