// Package embed implements all file embedding logic for github.com/mjibson/esc.
package embed

import (
	"bytes"
	"compress/gzip"
	"encoding/base64"
	"io"
	"io/ioutil"
	"strings"
	"testing"
)

func Test_canonicFileName(t *testing.T) {
	tests := []struct {
		name   string
		fname  string
		prefix string
		want   string
	}{
		{"simple", "ololo", "", "/ololo"},
		{"simple with prefix", "/ololo", "trololo", "/ololo"},
		{"simple start with prefix", "trololo/ololo", "trololo", "/ololo"},
		{"prefix in the middle", "start/trololo/ololo", "trololo", "/start/trololo/ololo"},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {

			if got := canonicFileName(tt.fname, tt.prefix); got != tt.want {
				t.Errorf("%q. canonicFileName() = %v, want %v", tt.name, got, tt.want)
			}
		})
	}
}

func TestRun(t *testing.T) {
	o := ioutil.Discard

	tests := []struct {
		name    string
		conf    *Config
		out     io.Writer
		wantErr bool
	}{
		{"empty conf - must err", &Config{}, nil, true},
		{"wrong modtime - must err", &Config{ModTime: "xxx"}, o, true},
		{"wrong ignore regexp - must err", &Config{Ignore: "**/xxx/**"}, o, true},
		{"wrong include regexp - must err", &Config{Include: "**/xxx/**"}, o, true},
		{"testdata", &Config{Package: "main", Files: []string{"../testdata"}}, o, false},
		{"testdata with ignore", &Config{
			Package: "main",
			Files:   []string{"../testdata"},
			Ignore:  "-responsive",
		}, o, false},
		{"testdata with include", &Config{
			Package: "main",
			Files:   []string{"../testdata"},
			Include: `.*min.js`,
		}, o, false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if err := Run(tt.conf, tt.out); (err != nil) != tt.wantErr {
				t.Errorf("%q. Run() error = %v, wantErr %v", tt.name, err, tt.wantErr)
			}
		})
	}
}

func TestDirOrder(t *testing.T) {
	var buf bytes.Buffer
	config := &Config{
		Package: "main",
		Files:   []string{"../testdata/empty"},
		ModTime: "0",
	}
	if err := Run(config, &buf); err != nil {
		t.Fatal(err)
	}
	got := buf.String()
	expect, _ := ioutil.ReadFile("../testdata/empty.expect")

	if got != string(expect) {
		t.Fatalf("got %s\nexpected %s", got, expect)
	}
}

func Test_escFile_fillCompressed(t *testing.T) {
	tests := []struct {
		name           string
		content        []byte
		gzipLevel      int
		wantCompressed string
		wantErr        bool
	}{
		{"empty", []byte(""), gzip.NoCompression, "\nH4sIAAAAAAAA/wEAAP//AAAAAAAAAAA=\n", false},
		{"short", []byte("ololo"), gzip.NoCompression, "\nH4sIAAAAAAAA/wAFAPr/b2xvbG8BAAD//0WLfvsFAAAA\n", false},
		{"wrong gzip level", []byte("ololo"), 40, "", true},
		{"some big file", bigFile, gzip.BestCompression, compressedBigFile, false},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			f := &_escFile{
				Data: tt.content,
			}
			if err := f.fillCompressed(tt.gzipLevel); (err != nil) != tt.wantErr {
				t.Errorf("%q. _escFile.fillCompressed() error = %v, wantErr %v", tt.name, err, tt.wantErr)
			}
			if tt.wantErr {
				return
			}
			if strings.Compare(tt.wantCompressed, f.Compressed) != 0 {
				t.Errorf("%q. _escFile.fillCompressed() compress to  = %v, want %v", tt.name, f.Compressed, tt.wantCompressed)
			}
			if !bytes.Equal(tt.content, decompress(f.Compressed)) {
				t.Errorf("%q. _escFile.fillCompressed() decompress  = %v, want %v", tt.name, decompress(f.Compressed), tt.content)
			}
		})
	}
}

func decompress(compressed string) []byte {
	var gr *gzip.Reader
	b64 := base64.NewDecoder(base64.StdEncoding, bytes.NewBufferString(compressed))
	gr, err := gzip.NewReader(b64)
	if err != nil {
		panic("error occured in decompress, gzip.NewReader: " + err.Error())
	}
	data, err := ioutil.ReadAll(gr)
	if err != nil {
		panic("error occured in decompress, ioutil.ReadAll: " + err.Error())
	}
	return data
}

var (
	bigFile = []byte(`
/*!
 * GitHub Light v0.4.1
 * Copyright (c) 2012 - 2017 GitHub, Inc.
 * Licensed under MIT (https://github.com/primer/github-syntax-theme-generator/blob/master/LICENSE)
 */.pl-c{color:#6a737d}.pl-c1,.pl-s .pl-v{color:#005cc5}.pl-e,.pl-en{color:#6f42c1}.pl-smi,.pl-s .pl-s1{color:#24292e}.pl-ent{color:#22863a}.pl-k{color:#d73a49}.pl-s,.pl-pds,.pl-s .pl-pse .pl-s1,.pl-sr,.pl-sr .pl-cce,.pl-sr .pl-sre,.pl-sr .pl-sra{color:#032f62}.pl-v,.pl-smw{color:#e36209}.pl-bu{color:#b31d28}.pl-ii{color:#fafbfc;background-color:#b31d28}.pl-c2{color:#fafbfc;background-color:#d73a49}.pl-c2::before{content:"^M"}.pl-sr .pl-cce{font-weight:bold;color:#22863a}.pl-ml{color:#735c0f}.pl-mh,.pl-mh .pl-en,.pl-ms{font-weight:bold;color:#005cc5}.pl-mi{font-style:italic;color:#24292e}.pl-mb{font-weight:bold;color:#24292e}.pl-md{color:#b31d28;background-color:#ffeef0}.pl-mi1{color:#22863a;background-color:#f0fff4}.pl-mc{color:#e36209;background-color:#ffebda}.pl-mi2{color:#f6f8fa;background-color:#005cc5}.pl-mdr{font-weight:bold;color:#6f42c1}.pl-ba{color:#586069}.pl-sg{color:#959da5}.pl-corl{text-decoration:underline;color:#032f62}.CodeMirror{font-family:monospace;height:300px;color:black;direction:ltr}.CodeMirror-lines{padding:4px 0}.CodeMirror pre{padding:0 4px}.CodeMirror-scrollbar-filler,.CodeMirror-gutter-filler{background-color:white}.CodeMirror-gutters{border-right:1px solid #ddd;background-color:#f7f7f7;white-space:nowrap}.CodeMirror-linenumber{padding:0 3px 0 5px;min-width:20px;text-align:right;color:#999;white-space:nowrap}.CodeMirror-guttermarker{color:black}.CodeMirror-guttermarker-subtle{color:#999}.CodeMirror-cursor{border-left:1px solid black;border-right:none;width:0}.CodeMirror div.CodeMirror-secondarycursor{border-left:1px solid silver}.cm-fat-cursor .CodeMirror-cursor{width:auto;border:0 !important;background:#7e7}.cm-fat-cursor div.CodeMirror-cursors{z-index:1}.cm-fat-cursor-mark{background-color:rgba(20,255,20,0.5);animation:blink 1.06s steps(1) infinite}.cm-animate-fat-cursor{width:auto;border:0;animation:blink 1.06s steps(1) infinite;background-color:#7e7}@keyframes blink{0%{}50%{background-color:transparent}100%{}}.cm-tab{display:inline-block;text-decoration:inherit}.CodeMirror-rulers{position:absolute;left:0;right:0;top:-50px;bottom:-20px;overflow:hidden}.CodeMirror-ruler{border-left:1px solid #ccc;top:0;bottom:0;position:absolute}.cm-s-default .cm-header{color:blue}.cm-s-default .cm-quote{color:#090}.cm-negative{color:#d44}.cm-positive{color:#292}.cm-header,.cm-strong{font-weight:bold}.cm-em{font-style:italic}.cm-link{text-decoration:underline}.cm-strikethrough{text-decoration:line-through}.cm-s-default .cm-keyword{color:#708}.cm-s-default .cm-atom{color:#219}.cm-s-default .cm-number{color:#164}.cm-s-default .cm-def{color:#00f}.cm-s-default .cm-variable-2{color:#05a}.cm-s-default .cm-variable-3,.cm-s-default .cm-type{color:#085}.cm-s-default .cm-comment{color:#a50}.cm-s-default .cm-string{color:#a11}.cm-s-default .cm-string-2{color:#f50}.cm-s-default .cm-meta{color:#555}.cm-s-default .cm-qualifier{color:#555}.cm-s-default .cm-builtin{color:#30a}.cm-s-default .cm-bracket{color:#997}.cm-s-default .cm-tag{color:#170}.cm-s-default .cm-attribute{color:#00c}.cm-s-default .cm-hr{color:#999}.cm-s-default .cm-link{color:#00c}.cm-s-default .cm-error{color:#f00}.cm-invalidchar{color:#f00}.CodeMirror-composing{border-bottom:2px solid}div.CodeMirror span.CodeMirror-matchingbracket{color:#0b0}div.CodeMirror span.CodeMirror-nonmatchingbracket{color:#a22}.CodeMirror-matchingtag{background:rgba(255,150,0,0.3)}.CodeMirror-activeline-background{background:#e8f2ff}.CodeMirror{position:relative;overflow:hidden;background:white}.CodeMirror-scroll{overflow:scroll !important;margin-bottom:-30px;margin-right:-30px;padding-bottom:30px;height:100%;outline:none;position:relative}.CodeMirror-sizer{position:relative;border-right:30px solid transparent}.CodeMirror-vscrollbar,.CodeMirror-hscrollbar,.CodeMirror-scrollbar-filler,.CodeMirror-gutter-filler{position:absolute;z-index:6;display:none}.CodeMirror-vscrollbar{right:0;top:0;overflow-x:hidden;overflow-y:scroll}.CodeMirror-hscrollbar{bottom:0;left:0;overflow-y:hidden;overflow-x:scroll}.CodeMirror-scrollbar-filler{right:0;bottom:0}.CodeMirror-gutter-filler{left:0;bottom:0}.CodeMirror-gutters{position:absolute;left:0;top:0;min-height:100%;z-index:3}.CodeMirror-gutter{white-space:normal;height:100%;display:inline-block;vertical-align:top;margin-bottom:-30px}.CodeMirror-gutter-wrapper{position:absolute;z-index:4;background:none !important;border:none !important}.CodeMirror-gutter-background{position:absolute;top:0;bottom:0;z-index:4}.CodeMirror-gutter-elt{position:absolute;cursor:default;z-index:4}.CodeMirror-gutter-wrapper ::-moz-selection{background-color:transparent}.CodeMirror-gutter-wrapper ::selection{background-color:transparent}.CodeMirror-gutter-wrapper ::-moz-selection{background-color:transparent}.CodeMirror-lines{cursor:text;min-height:1px}.CodeMirror pre{border-radius:0;border-width:0;background:transparent;font-family:inherit;font-size:inherit;margin:0;white-space:pre;word-wrap:normal;line-height:inherit;color:inherit;z-index:2;position:relative;overflow:visible;-webkit-tap-highlight-color:transparent;-webkit-font-variant-ligatures:contextual;font-variant-ligatures:contextual}.CodeMirror-wrap pre{word-wrap:break-word;white-space:pre-wrap;word-break:normal}.CodeMirror-linebackground{position:absolute;left:0;right:0;top:0;bottom:0;z-index:0}.CodeMirror-linewidget{position:relative;z-index:2;padding:0.1px}.CodeMirror-rtl pre{direction:rtl}.CodeMirror-code{outline:none}.CodeMirror-scroll,.CodeMirror-sizer,.CodeMirror-gutter,.CodeMirror-gutters,.CodeMirror-linenumber{box-sizing:content-box}.CodeMirror-measure{position:absolute;width:100%;height:0;overflow:hidden;visibility:hidden}.CodeMirror-cursor{position:absolute;pointer-events:none}.CodeMirror-measure pre{position:static}div.CodeMirror-cursors{visibility:hidden;position:relative;z-index:3}div.CodeMirror-dragcursors{visibility:visible}.CodeMirror-focused div.CodeMirror-cursors{visibility:visible}.CodeMirror-selected{background:#d9d9d9}.CodeMirror-focused .CodeMirror-selected{background:#d7d4f0}.CodeMirror-crosshair{cursor:crosshair}.CodeMirror-line::-moz-selection,.CodeMirror-line>span::-moz-selection,.CodeMirror-line>span>span::-moz-selection{background:#d7d4f0}.CodeMirror-line::selection,.CodeMirror-line>span::selection,.CodeMirror-line>span>span::selection{background:#d7d4f0}.CodeMirror-line::-moz-selection,.CodeMirror-line>span::-moz-selection,.CodeMirror-line>span>span::-moz-selection{background:#d7d4f0}.cm-searching{background-color:#ffa;background-color:rgba(255,255,0,0.4)}.cm-force-border{padding-right:.1px}@media print{.CodeMirror div.CodeMirror-cursors{visibility:hidden}}.cm-tab-wrap-hack:after{content:''}span.CodeMirror-selectedtext{background:none}.CodeMirror-dialog{position:absolute;left:0;right:0;background:inherit;z-index:15;padding:.1em .8em;overflow:hidden;color:inherit}.CodeMirror-dialog-top{border-bottom:1px solid #eee;top:0}.CodeMirror-dialog-bottom{border-top:1px solid #eee;bottom:0}.CodeMirror-dialog input{border:none;outline:none;background:transparent;width:20em;color:inherit;font-family:monospace}.CodeMirror-dialog button{font-size:70%}.CodeMirror-merge{position:relative;border:1px solid #ddd;white-space:pre}.CodeMirror-merge,.CodeMirror-merge .CodeMirror{height:350px}.CodeMirror-merge-2pane .CodeMirror-merge-pane{width:47%}.CodeMirror-merge-2pane .CodeMirror-merge-gap{width:6%}.CodeMirror-merge-3pane .CodeMirror-merge-pane{width:31%}.CodeMirror-merge-3pane .CodeMirror-merge-gap{width:3.5%}.CodeMirror-merge-pane{display:inline-block;white-space:normal;vertical-align:top}.CodeMirror-merge-pane-rightmost{position:absolute;right:0px;z-index:1}.CodeMirror-merge-gap{z-index:2;display:inline-block;height:100%;box-sizing:border-box;overflow:hidden;border-left:1px solid #ddd;border-right:1px solid #ddd;position:relative;background:#f8f8f8}.CodeMirror-merge-scrolllock-wrap{position:absolute;bottom:0;left:50%}.CodeMirror-merge-scrolllock{position:relative;left:-50%;cursor:pointer;color:#555;line-height:1}.CodeMirror-merge-scrolllock:after{content:"\21db\00a0\00a0\21da"}.CodeMirror-merge-scrolllock.CodeMirror-merge-scrolllock-enabled:after{content:"\21db\21da"}.CodeMirror-merge-copybuttons-left,.CodeMirror-merge-copybuttons-right{position:absolute;left:0;top:0;right:0;bottom:0;line-height:1}.CodeMirror-merge-copy{position:absolute;cursor:pointer;color:#44c;z-index:3}.CodeMirror-merge-copy-reverse{position:absolute;cursor:pointer;color:#44c}.CodeMirror-merge-copybuttons-left .CodeMirror-merge-copy{left:2px}.CodeMirror-merge-copybuttons-right .CodeMirror-merge-copy{right:2px}.CodeMirror-merge-r-inserted,.CodeMirror-merge-l-inserted{background-image:url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAMAAAACCAYAAACddGYaAAAAGUlEQVQI12MwuCXy3+CWyH8GBgYGJgYkAABZbAQ9ELXurwAAAABJRU5ErkJggg==);background-position:bottom left;background-repeat:repeat-x}.CodeMirror-merge-r-deleted,.CodeMirror-merge-l-deleted{background-image:url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAMAAAACCAYAAACddGYaAAAAGUlEQVQI12M4Kyb2/6yY2H8GBgYGJgYkAABURgPz6Ks7wQAAAABJRU5ErkJggg==);background-position:bottom left;background-repeat:repeat-x}.CodeMirror-merge-r-chunk{background:#ffffe0}.CodeMirror-merge-r-chunk-start{border-top:1px solid #ee8}.CodeMirror-merge-r-chunk-end{border-bottom:1px solid #ee8}.CodeMirror-merge-r-connect{fill:#ffffe0;stroke:#ee8;stroke-width:1px}.CodeMirror-merge-l-chunk{background:#eef}.CodeMirror-merge-l-chunk-start{border-top:1px solid #88e}.CodeMirror-merge-l-chunk-end{border-bottom:1px solid #88e}.CodeMirror-merge-l-connect{fill:#eef;stroke:#88e;stroke-width:1px}.CodeMirror-merge-l-chunk.CodeMirror-merge-r-chunk{background:#dfd}.CodeMirror-merge-l-chunk-start.CodeMirror-merge-r-chunk-start{border-top:1px solid #4e4}.CodeMirror-merge-l-chunk-end.CodeMirror-merge-r-chunk-end{border-bottom:1px solid #4e4}.CodeMirror-merge-collapsed-widget:before{content:"(...)"}.CodeMirror-merge-collapsed-widget{cursor:pointer;color:#88b;background:#eef;border:1px solid #ddf;font-size:90%;padding:0 3px;border-radius:4px}.CodeMirror-merge-collapsed-line .CodeMirror-gutter-elt{display:none}/*!
 * GitHub Light v0.4.2
 * Copyright (c) 2012 - 2017 GitHub, Inc.
 * Licensed under MIT (https://github.com/primer/github-syntax-theme-generator/blob/master/LICENSE)
 */.cm-s-github-light.CodeMirror{background:#fff;color:#24292e}.cm-s-github-light .CodeMirror-gutters{background:#fff;border-right-width:0}.cm-s-github-light .CodeMirror-guttermarker{color:white}.cm-s-github-light .CodeMirror-guttermarker-subtle{color:#d0d0d0}.cm-s-github-light .CodeMirror-linenumber{color:#959da5;padding:0 16px 0 16px}.cm-s-github-light .CodeMirror-cursor{border-left:1px solid #24292e}.cm-s-github-light div.CodeMirror-selected,.cm-s-github-light .CodeMirror-line::-moz-selection,.cm-s-github-light .CodeMirror-line>span::-moz-selection,.cm-s-github-light .CodeMirror-line>span>span::-moz-selection,.cm-s-github-light .CodeMirror-line::-moz-selection,.cm-s-github-light .CodeMirror-line>span::-moz-selection,.cm-s-github-light .CodeMirror-line>span>span::-moz-selection{background:#c8c8fa}.cm-s-github-light div.CodeMirror-selected,.cm-s-github-light .CodeMirror-line::selection,.cm-s-github-light .CodeMirror-line>span::selection,.cm-s-github-light .CodeMirror-line>span>span::selection,.cm-s-github-light .CodeMirror-line::-moz-selection,.cm-s-github-light .CodeMirror-line>span::-moz-selection,.cm-s-github-light .CodeMirror-line>span>span::-moz-selection{background:#c8c8fa}.cm-s-github-light .CodeMirror-activeline-background{background:#fafbfc}.cm-s-github-light .CodeMirror-matchingbracket{text-decoration:underline;color:#24292e !important}.cm-s-github-light .CodeMirror-lines{font-family:"SFMono-Regular", Consolas, "Liberation Mono", Menlo, Courier, monospace;font-size:12px;background:#fff;line-height:1.5}.cm-s-github-light .cm-comment{color:#6a737d}.cm-s-github-light .cm-constant{color:#005cc5}.cm-s-github-light .cm-entity{font-weight:normal;font-style:normal;text-decoration:none;color:#6f42c1}.cm-s-github-light .cm-keyword{font-weight:normal;font-style:normal;text-decoration:none;color:#d73a49}.cm-s-github-light .cm-storage{color:#d73a49}.cm-s-github-light .cm-string{font-weight:normal;font-style:normal;text-decoration:none;color:#032f62}.cm-s-github-light .cm-support{font-weight:normal;font-style:normal;text-decoration:none;color:#005cc5}.cm-s-github-light .cm-variable{font-weight:normal;font-style:normal;text-decoration:none;color:#e36209}details-dialog{position:fixed;margin:10vh auto;top:0;left:50%;transform:translateX(-50%);z-index:999;max-height:80vh;width:448px}.octicon{display:inline-block;vertical-align:text-top;fill:currentColor}body.intent-mouse [role="button"]:focus,body.intent-mouse button:focus,body.intent-mouse summary:focus{outline:none}.CodeMirror{height:calc(100vh - 1px)}.file-editor-textarea{width:100%;padding:5px 4px;font:12px "SFMono-Regular",Consolas,"Liberation Mono",Menlo,Courier,monospace;resize:vertical;border:0;border-radius:0;outline:none}.container-preview .tabnav-tabs{margin:-6px 0 -6px -11px}.container-preview .tabnav-tabs .tabnav-tab{padding:12px 15px;border-radius:0}.container-preview .tabnav-tabs>.selected:first-child{border-top-left-radius:3px}.container-preview .tabnav-tabs .selected{font-weight:600}.container-preview.template-editor .commit-create,.container-preview.template-editor .file-actions{display:block}.container-preview.template-editor .show-code,.container-preview.template-editor .commit-preview,.container-preview.template-editor .loading-preview-msg,.container-preview.template-editor .no-changes-preview-msg,.container-preview.template-editor .error-preview-msg{display:none}.container-preview.render-editor .commit-create,.container-preview.render-editor .file-actions{display:block}.container-preview.render-editor .template-editor,.container-preview.render-editor .show-code,.container-preview.render-editor .commit-preview,.container-preview.render-editor .loading-preview-msg,.container-preview.render-editor .no-changes-preview-msg,.container-preview.render-editor .error-preview-msg{display:none}.container-preview.show-code .commit-create,.container-preview.show-code .file-actions{display:block}.container-preview.show-code .template-editor,.container-preview.show-code .render-editor,.container-preview.show-code .commit-preview,.container-preview.show-code .loading-preview-msg,.container-preview.show-code .no-changes-preview-msg,.container-preview.show-code .error-preview-msg{display:none}.container-preview:not(.show-code) .commit-create,.container-preview:not(.show-code) .file-actions{display:none}.container-preview.loading-preview .loading-preview-msg{display:block}.container-preview.loading-preview .template-editor,.container-preview.loading-preview .render-editor,.container-preview.loading-preview .no-changes-preview-msg,.container-preview.loading-preview .error-preview-msg,.container-preview.loading-preview .commit-preview{display:none}.container-preview.show-preview .commit-preview{display:block}.container-preview.show-preview .template-editor,.container-preview.show-preview .render-editor,.container-preview.show-preview .loading-preview-msg,.container-preview.show-preview .no-changes-preview-msg,.container-preview.show-preview .error-preview-msg{display:none}.container-preview.no-changes-preview .no-changes-preview-msg{display:block}.container-preview.no-changes-preview .template-editor,.container-preview.no-changes-preview .render-editor,.container-preview.no-changes-preview .loading-preview-msg,.container-preview.no-changes-preview .error-preview-msg,.container-preview.no-changes-preview .commit-preview{display:none}.container-preview.error-preview .error-preview-msg{display:block}.container-preview.error-preview .template-editor,.container-preview.error-preview .render-editor,.container-preview.error-preview .loading-preview-msg,.container-preview.error-preview .no-changes-preview-msg,.container-preview.error-preview .commit-preview{display:none}.container-preview p.preview-msg{padding:30px;font-size:16px}.CodeMirror-merge-header{height:30px}.CodeMirror-merge-header .CodeMirror-merge-pane{height:30px;line-height:30px}.cm-s-github-light .merge-gutter{width:14px}.conflict-background+.CodeMirror-gutter-wrapper .CodeMirror-linenumber{background-color:#fffbdd}.conflict-gutter-marker{background-color:#fffbdd}.conflict-gutter-marker::after,.conflict-gutter-marker::before{position:absolute;left:-1px;content:"";background-color:#d73a49}.conflict-gutter-marker-start::after,.conflict-gutter-marker-end::after{width:1px;height:10px}.conflict-gutter-marker-start::before,.conflict-gutter-marker-middle::before,.conflict-gutter-marker-end::before{width:10px;height:1px}.conflict-gutter-marker-start::after{bottom:0}.conflict-gutter-marker-end::after{top:0}.conflict-gutter-marker-start::before{top:7px}.conflict-gutter-marker-end::before{bottom:7px}.conflict-gutter-marker-line::after,.conflict-gutter-marker-middle::after{width:1px;height:18px}.conflict-gutter-marker-middle::before{top:9px}.form-group .draggable-box{height:20px;opacity:0}.form-group .draggable-box:hover{cursor:move}.form-group .edit-action{opacity:0.6}.form-group .draggable-box-icon{margin-left:-4px}.form-group .form-field-hover{background-color:none;border:1px solid rgba(149,157,165,0)}.form-group:hover{cursor:pointer}.form-group:hover .draggable-box{opacity:0.2}.form-group:hover .edit-action{opacity:0.7}.form-group:hover .form-field-hover{border:1px solid #959da5}.placeholder-box{border:1px solid #959da5}.template-previews{max-width:768px}.template-previews .Box .expand-group{display:none;height:0}.template-previews .Box .dismiss-preview-button{display:none}.template-previews .Box.expand-preview .expand-group{display:block;height:100%;transition:height 3s}.template-previews .Box.expand-preview .preview-button{display:none}.template-previews .Box.expand-preview .dismiss-preview-button{display:inline}.additional-options{display:none}.edit-labels{display:none}.preview-section{display:block}.edit-section{display:none}.Box .section-focus .preview-section{display:none}.Box .section-focus .edit-section{display:block}.user-select-contain{-ms-user-select:element;-webkit-user-select:contain;-moz-user-select:contain;user-select:contain}.zeroclipboard-link{padding:0;margin:0;color:#0366d6;cursor:pointer;background:none;border:0}.zeroclipboard-link .octicon{display:block}image-crop{display:block}image-crop.nesw{cursor:nesw-resize}image-crop.nwse{cursor:nwse-resize}image-crop.nesw .crop-box,image-crop.nwse .crop-box{cursor:inherit}image-crop[loaded] .crop-image{display:block}image-crop[loaded] [data-loading-slot],image-crop .crop-image{display:none}image-crop .crop-wrapper{position:relative;font-size:0}image-crop .crop-container{user-select:none;-ms-user-select:none;-moz-user-select:none;-webkit-user-select:none;position:absolute;overflow:hidden;z-index:1;top:0;width:100%;height:100%}image-crop .crop-box{position:absolute;border:1px dashed #fff;box-sizing:border-box;cursor:move}image-crop .crop-outline{position:absolute;top:0;bottom:0;left:0;right:0;outline:600px solid rgba(0,0,0,0.3)}image-crop .handle{position:absolute}image-crop .handle:before{position:absolute;display:block;padding:4px;transform:translate(-50%, -50%);content:' ';background:#fff;border:1px solid #767676}image-crop .ne{top:0;right:0;cursor:nesw-resize}image-crop .nw{top:0;left:0;cursor:nwse-resize}image-crop .se{bottom:0;right:0;cursor:nwse-resize}image-crop .sw{bottom:0;left:0;cursor:nesw-resize}/*!
 * Primer-core
 * http://primer.github.io
 *
 * Released under MIT license. Copyright (c) 2018 GitHub Inc.
 *//*! normalize.css v4.1.1 | MIT License | github.com/necolas/normalize.css */html{font-family:sans-serif;-ms-text-size-adjust:100%;-webkit-text-size-adjust:100%}body{margin:0}article,aside,details,figcaption,figure,footer,header,main,menu,nav,section{display:block}summary{display:list-item}audio,canvas,progress,video{display:inline-block}audio:not([controls]){display:none;height:0}progress{vertical-align:baseline}template,[hidden]{display:none}a{background-color:transparent}a:active,a:hover{outline-width:0}abbr[title]{border-bottom:none;text-decoration:underline;-webkit-text-decoration:underline dotted;text-decoration:underline dotted}b,strong{font-weight:inherit}b,strong{font-weight:bolder}dfn{font-style:italic}h1{font-size:2em;margin:0.67em 0}mark{background-color:#ff0;color:#000}small{font-size:80%}sub,sup{font-size:75%;line-height:0;position:relative;vertical-align:baseline}sub{bottom:-0.25em}sup{top:-0.5em}img{border-style:none}svg:not(:root){overflow:hidden}code,kbd,pre,samp{font-family:monospace, monospace;font-size:1em}figure{margin:1em 40px}hr{box-sizing:content-box;height:0;overflow:visible}button,input,select,textarea{font:inherit;margin:0}optgroup{font-weight:bold}button,input{overflow:visible}button,select{text-transform:none}button,html [type="button"],[type="reset"],[type="submit"]{-webkit-appearance:button}button::-moz-focus-inner,[type="button"]::-moz-focus-inner,[type="reset"]::-moz-focus-inner,[type="submit"]::-moz-focus-inner{border-style:none;padding:0}button:-moz-focusring,[type="button"]:-moz-focusring,[type="reset"]:-moz-focusring,[type="submit"]:-moz-focusring{outline:1px dotted ButtonText}fieldset{border:1px solid #c0c0c0;margin:0 2px;padding:0.35em 0.625em 0.75em}legend{box-sizing:border-box;color:inherit;display:table;max-width:100%;padding:0;white-space:normal}textarea{overflow:auto}[type="checkbox"],[type="radio"]{box-sizing:border-box;padding:0}[type="number"]::-webkit-inner-spin-button,[type="number"]::-webkit-outer-spin-button{height:auto}[type="search"]{-webkit-appearance:textfield;outline-offset:-2px}[type="search"]::-webkit-search-cancel-button,[type="search"]::-webkit-search-decoration{-webkit-appearance:none}::-webkit-input-placeholder{color:inherit;opacity:0.54}::-webkit-file-upload-button{-webkit-appearance:button;font:inherit}*{box-sizing:border-box}input,select,textarea,button{font-family:inherit;font-size:inherit;line-height:inherit}body{font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Helvetica,Arial,sans-serif,"Apple Color Emoji","Segoe UI Emoji","Segoe UI Symbol";font-size:14px;line-height:1.5;color:#24292e;background-color:#fff}a{color:#0366d6;text-decoration:none}a:hover{text-decoration:underline}b,strong{font-weight:600}hr,.rule{height:0;margin:15px 0;overflow:hidden;background:transparent;border:0;border-bottom:1px solid #dfe2e5}hr::before,.rule::before{display:table;content:""}hr::after,.rule::after{display:table;clear:both;content:""}table{border-spacing:0;border-collapse:collapse}td,th{padding:0}button{cursor:pointer;border-radius:0}details summary{cursor:pointer}details:not([open])>*:not(summary){display:none !important}h1,h2,h3,h4,h5,h6{margin-top:0;margin-bottom:0}h1{font-size:32px;font-weight:600}h2{font-size:24px;font-weight:600}h3{font-size:20px;font-weight:600}h4{font-size:16px;font-weight:600}h5{font-size:14px;font-weight:600}h6{font-size:12px;font-weight:600}p{margin-top:0;margin-bottom:10px}small{font-size:90%}blockquote{margin:0}ul,ol{padding-left:0;margin-top:0;margin-bottom:0}ol ol,ul ol{list-style-type:lower-roman}ul ul ol,ul ol ol,ol ul ol,ol ol ol{list-style-type:lower-alpha}dd{margin-left:0}tt,code{font-family:"SFMono-Regular",Consolas,"Liberation Mono",Menlo,Courier,monospace;font-size:12px}pre{margin-top:0;margin-bottom:0;font-family:"SFMono-Regular",Consolas,"Liberation Mono",Menlo,Courier,monospace;font-size:12px}.octicon{vertical-align:text-bottom}.Box{background-color:#fff;border:1px solid #d1d5da;border-radius:3px}.Box--condensed{line-height:1.25}.Box--condensed .Box-header{padding:8px 16px}.Box--condensed .Box-body{padding:8px 16px}.Box--condensed .Box-footer{padding:8px 16px}.Box--condensed .Box-btn-octicon.btn-octicon{padding:8px 16px;margin:-8px -16px;line-height:1.25}.Box--condensed .Box-row{padding:8px 16px}.Box--spacious .Box-header{padding:24px;line-height:1.25}.Box--spacious .Box-title{font-size:20px}.Box--spacious .Box-body{padding:24px}.Box--spacious .Box-footer{padding:24px}.Box--spacious .Box-btn-octicon.btn-octicon{padding:24px;margin:-24px -24px}.Box--spacious .Box-row{padding:24px}.Box-header{padding:16px;margin:-1px -1px 0 -1px;background-color:#f6f8fa;border-color:#d1d5da;border-style:solid;border-width:1px;border-top-left-radius:3px;border-top-right-radius:3px}.Box-title{font-size:14px;font-weight:600}.Box-body{padding:16px;border-bottom:1px solid #e1e4e8}.Box-body:last-of-type{margin-bottom:-1px;border-bottom-right-radius:2px;border-bottom-left-radius:2px}.Box-row{padding:16px;margin-top:-1px;list-style-type:none;border-top:1px solid #e1e4e8}.Box-row:first-of-type{border-top-color:transparent;border-top-left-radius:2px;border-top-right-radius:2px}.Box-row:last-of-type{border-bottom-right-radius:2px;border-bottom-left-radius:2px}.Box-row.Box-row--unread,.Box-row.unread{box-shadow:2px 0 0 #0366d6 inset}.Box-row.navigation-focus .Box-row--drag-button{color:#0366d6;cursor:-webkit-grab;cursor:grab;opacity:100}.Box-row.navigation-focus.is-dragging .Box-row--drag-button{cursor:-webkit-grabbing;cursor:grabbing}.Box-row.navigation-focus.sortable-chosen{background-color:#fafbfc}.Box-row.navigation-focus.sortable-ghost{background-color:#f6f8fa}.Box-row.navigation-focus.sortable-ghost .Box-row--drag-hide{opacity:0}.Box-row--focus-gray.navigation-focus{background-color:#f6f8fa}.Box-row--focus-blue.navigation-focus{background-color:#f1f8ff}.Box-row--hover-gray:hover{background-color:#f6f8fa}.Box-row--hover-blue:hover{background-color:#f1f8ff}@media (min-width: 768px){.Box-row-link{color:#24292e;text-decoration:none}.Box-row-link:hover{color:#0366d6;text-decoration:none}}.Box-row--drag-button{opacity:0}.Box-footer{padding:16px;margin-top:-1px;border-top:1px solid #e1e4e8}.Box--scrollable{max-height:324px;overflow:scroll}.Box--blue{border-color:#c8e1ff}.Box--blue .Box-header{background-color:#f1f8ff;border-color:#c8e1ff}.Box--blue .Box-body{border-color:#c8e1ff}.Box--blue .Box-row{border-color:#c8e1ff}.Box--blue .Box-footer{border-color:#c8e1ff}.Box--danger{border-color:#d73a49}.Box--danger .Box-row:first-of-type{border-color:#d73a49}.Box--danger .Box-body:last-of-type{border-color:#d73a49}.Box-header--blue{background-color:#f1f8ff;border-color:#c8e1ff}.Box-row--yellow{background-color:#fffbdd}.Box-row--blue{background-color:#f1f8ff}.Box-row--gray{background-color:#f6f8fa}.Box-btn-octicon.btn-octicon{padding:16px 16px;margin:-16px -16px;line-height:1.5}.breadcrumb-item{display:inline-block;margin-left:-0.35em;white-space:nowrap;list-style:none}.breadcrumb-item::after{padding-right:.5em;padding-left:.5em;color:#e1e4e8;content:"/"}.breadcrumb-item:first-child{margin-left:0}.breadcrumb-item-selected::after{content:none}.btn{position:relative;display:inline-block;padding:6px 12px;font-size:14px;font-weight:600;line-height:20px;white-space:nowrap;vertical-align:middle;cursor:pointer;-webkit-user-select:none;-moz-user-select:none;-ms-user-select:none;user-select:none;background-repeat:repeat-x;background-position:-1px -1px;background-size:110% 110%;border:1px solid rgba(27,31,35,0.2);border-radius:0.25em;-webkit-appearance:none;-moz-appearance:none;appearance:none}.btn i{font-style:normal;font-weight:500;opacity:0.75}.btn .octicon{vertical-align:text-top}.btn .Counter{color:#586069;text-shadow:none;background-color:rgba(27,31,35,0.1)}.btn:hover{text-decoration:none;background-repeat:repeat-x}.btn:focus{outline:0}.btn:disabled,.btn.disabled{cursor:default;background-position:0 0}.btn:active,.btn.selected{background-image:none}.btn{color:#24292e;background-color:#eff3f6;background-image:linear-gradient(-180deg, #fafbfc 0%, #eff3f6 90%)}.btn:focus,.btn.focus{box-shadow:0 0 0 0.2em rgba(3,102,214,0.3)}.btn:hover,.btn.hover{background-color:#e6ebf1;background-image:linear-gradient(-180deg, #f0f3f6 0%, #e6ebf1 90%);background-position:-.5em;border-color:rgba(27,31,35,0.35)}.btn:active,.btn.selected,[open]>.btn{background-color:#e9ecef;background-image:none;border-color:rgba(27,31,35,0.35);box-shadow:inset 0 0.15em 0.3em rgba(27,31,35,0.15)}.btn:disabled,.btn.disabled{color:rgba(36,41,46,0.4);background-color:#eff3f6;background-image:none;border-color:rgba(27,31,35,0.2);box-shadow:none}.btn-primary{color:#fff;background-color:#28a745;background-image:linear-gradient(-180deg, #34d058 0%, #28a745 90%)}.btn-primary:focus,.btn-primary.focus{box-shadow:0 0 0 0.2em rgba(52,208,88,0.4)}.btn-primary:hover,.btn-primary.hover{background-color:#269f42;background-image:linear-gradient(-180deg, #2fcb53 0%, #269f42 90%);background-position:-.5em;border-color:rgba(27,31,35,0.5)}.btn-primary:active,.btn-primary.selected,[open]>.btn-primary{background-color:#279f43;background-image:none;border-color:rgba(27,31,35,0.5);box-shadow:inset 0 0.15em 0.3em rgba(27,31,35,0.15)}.btn-primary:disabled,.btn-primary.disabled{color:rgba(255,255,255,0.75);background-color:#94d3a2;background-image:none;border-color:rgba(27,31,35,0.2);box-shadow:none}.btn-primary .Counter{color:#29b249;background-color:#fff}.btn-purple{color:#fff;background-color:#643ab0;background-image:linear-gradient(-180deg, #7e55c7 0%, #643ab0 90%)}.btn-purple:focus,.btn-purple.focus{box-shadow:0 0 0 0.2em rgba(126,85,199,0.4)}.btn-purple:hover,.btn-purple.hover{background-color:#5f37a8;background-image:linear-gradient(-180deg, #784ec5 0%, #5f37a8 90%);background-position:-.5em;border-color:rgba(27,31,35,0.5)}.btn-purple:active,.btn-purple.selected,[open]>.btn-purple{background-color:#613ca4;background-image:none;border-color:rgba(27,31,35,0.5);box-shadow:inset 0 0.15em 0.3em rgba(27,31,35,0.15)}.btn-purple:disabled,.btn-purple.disabled{color:rgba(255,255,255,0.75);background-color:#b19cd7;background-image:none;border-color:rgba(27,31,35,0.2);box-shadow:none}.btn-purple .Counter{color:#683cb8;background-color:#fff}.btn-blue{color:#fff;background-color:#0361cc;background-image:linear-gradient(-180deg, #0679fc 0%, #0361cc 90%)}.btn-blue:focus,.btn-blue.focus{box-shadow:0 0 0 0.2em rgba(6,121,252,0.4)}.btn-blue:hover,.btn-blue.hover{background-color:#035cc2;background-image:linear-gradient(-180deg, #0374f4 0%, #035cc2 90%);background-position:-.5em;border-color:rgba(27,31,35,0.5)}.btn-blue:active,.btn-blue.selected,[open]>.btn-blue{background-color:#045cc1;background-image:none;border-color:rgba(27,31,35,0.5);box-shadow:inset 0 0.15em 0.3em rgba(27,31,35,0.15)}.btn-blue:disabled,.btn-blue.disabled{color:rgba(255,255,255,0.75);background-color:#81b0e5;background-image:none;border-color:rgba(27,31,35,0.2);box-shadow:none}.btn-blue .Counter{color:#0366d6;background-color:#fff}.btn-danger{color:#cb2431;background-color:#fafbfc;background-image:linear-gradient(-180deg, #fafbfc 0%, #eff3f6 90%)}.btn-danger:focus{box-shadow:0 0 0 0.2em rgba(203,36,49,0.4)}.btn-danger:hover{color:#fff;background-color:#cb2431;background-image:linear-gradient(-180deg, #de4450 0%, #cb2431 90%);border-color:rgba(27,31,35,0.5)}.btn-danger:hover .Counter{color:#fff}.btn-danger:active,.btn-danger.selected,[open]>.btn-danger{color:#fff;background-color:#b5202c;background-image:none;border-color:rgba(27,31,35,0.5);box-shadow:inset 0 0.15em 0.3em rgba(27,31,35,0.15)}.btn-danger:disabled,.btn-danger.disabled{color:rgba(203,36,49,0.4);background-color:#eff3f6;background-image:none;border-color:rgba(27,31,35,0.2);box-shadow:none}.btn-outline{color:#0366d6;background-color:#fff;background-image:none}.btn-outline .Counter{background-color:rgba(27,31,35,0.07)}.btn-outline:hover,.btn-outline:active,.btn-outline.selected,[open]>.btn-outline{color:#fff;background-color:#0366d6;background-image:none;border-color:#0366d6}.btn-outline:hover .Counter,.btn-outline:active .Counter,.btn-outline.selected .Counter,[open]>.btn-outline .Counter{color:#0366d6;background-color:#fff}.btn-outline:focus{border-color:#0366d6;box-shadow:0 0 0 0.2em rgba(3,102,214,0.4)}.btn-outline:disabled,.btn-outline.disabled{color:rgba(27,31,35,0.3);background-color:#fff;border-color:rgba(27,31,35,0.15);box-shadow:none}.btn-with-count{float:left;border-top-right-radius:0;border-bottom-right-radius:0}.btn-sm{padding:3px 10px;font-size:12px;line-height:20px}.btn-large{padding:.75em 1.25em;font-size:inherit;border-radius:6px}.hidden-text-expander{display:block}.hidden-text-expander.inline{position:relative;top:-1px;display:inline-block;margin-left:5px;line-height:0}.hidden-text-expander a,.ellipsis-expander{display:inline-block;height:12px;padding:0 5px 5px;font-size:12px;font-weight:600;line-height:6px;color:#444d56;text-decoration:none;vertical-align:middle;background:#dfe2e5;border:0;border-radius:1px}.hidden-text-expander a:hover,.ellipsis-expander:hover{text-decoration:none;background-color:#c6cbd1}.hidden-text-expander a:active,.ellipsis-expander:active{color:#fff;background-color:#2188ff}.social-count{float:left;padding:3px 10px;font-size:12px;font-weight:600;line-height:20px;color:#24292e;vertical-align:middle;background-color:#fff;border:1px solid rgba(27,31,35,0.2);border-left:0;border-top-right-radius:3px;border-bottom-right-radius:3px}.social-count:hover,.social-count:active{text-decoration:none}.social-count:hover{color:#0366d6;cursor:pointer}.btn-block{display:block;width:100%;text-align:center}.btn-link{display:inline-block;padding:0;font-size:inherit;color:#0366d6;text-decoration:none;white-space:nowrap;cursor:pointer;-webkit-user-select:none;-moz-user-select:none;-ms-user-select:none;user-select:none;background-color:transparent;border:0;-webkit-appearance:none;-moz-appearance:none;appearance:none}.btn-link:hover{text-decoration:underline}.btn-link:disabled,.btn-link:disabled:hover{color:rgba(88,96,105,0.5);cursor:default}.details-reset>summary{list-style:none}.details-reset>summary::before{display:none}.details-reset>summary::-webkit-details-marker{display:none}.BtnGroup{display:inline-block;vertical-align:middle}.BtnGroup::before{display:table;content:""}.BtnGroup::after{display:table;clear:both;content:""}.BtnGroup+.BtnGroup,.BtnGroup+.btn{margin-left:5px}.BtnGroup-item{position:relative;float:left;border-right-width:0;border-radius:0}.BtnGroup-item:first-child{border-top-left-radius:3px;border-bottom-left-radius:3px}.BtnGroup-item:last-child{border-right-width:1px;border-top-right-radius:3px;border-bottom-right-radius:3px}.BtnGroup-item.selected,.BtnGroup-item:focus,.BtnGroup-item:active,.BtnGroup-item:hover{border-right-width:1px}.BtnGroup-item.selected+.BtnGroup-item,.BtnGroup-item.selected+.BtnGroup-form .BtnGroup-item,.BtnGroup-item:focus+.BtnGroup-item,.BtnGroup-item:focus+.BtnGroup-form .BtnGroup-item,.BtnGroup-item:active+.BtnGroup-item,.BtnGroup-item:active+.BtnGroup-form .BtnGroup-item,.BtnGroup-item:hover+.BtnGroup-item,.BtnGroup-item:hover+.BtnGroup-form .BtnGroup-item{border-left-width:0}.BtnGroup-form{float:left}.BtnGroup-form:first-child .BtnGroup-item{border-top-left-radius:3px;border-bottom-left-radius:3px}.BtnGroup-form:last-child .BtnGroup-item{border-right-width:1px;border-top-right-radius:3px;border-bottom-right-radius:3px}.BtnGroup-form .BtnGroup-item{border-right-width:0;border-radius:0}.BtnGroup-form.selected .BtnGroup-item,.BtnGroup-form:focus .BtnGroup-item,.BtnGroup-form:active .BtnGroup-item,.BtnGroup-form:hover .BtnGroup-item{border-right-width:1px}.BtnGroup-form.selected+.BtnGroup-item,.BtnGroup-form.selected+.BtnGroup-form .BtnGroup-item,.BtnGroup-form:focus+.BtnGroup-item,.BtnGroup-form:focus+.BtnGroup-form .BtnGroup-item,.BtnGroup-form:active+.BtnGroup-item,.BtnGroup-form:active+.BtnGroup-form .BtnGroup-item,.BtnGroup-form:hover+.BtnGroup-item,.BtnGroup-form:hover+.BtnGroup-form .BtnGroup-item{border-left-width:0}.TableObject{display:table}.TableObject-item{display:table-cell;width:1%;white-space:nowrap;vertical-align:middle}.TableObject-item--primary{width:99%}fieldset{padding:0;margin:0;border:0}label{font-weight:600}.form-control,.form-select{min-height:34px;padding:6px 8px;font-size:16px;line-height:20px;color:#24292e;vertical-align:middle;background-color:#fff;background-repeat:no-repeat;background-position:right 8px center;border:1px solid #d1d5da;border-radius:3px;outline:none;box-shadow:inset 0 1px 2px rgba(27,31,35,0.075)}.form-control.focus,.form-control:focus,.form-select.focus,.form-select:focus{border-color:#2188ff;outline:none;box-shadow:inset 0 1px 2px rgba(27,31,35,0.075),0 0 0 0.2em rgba(3,102,214,0.3)}@media (min-width: 768px){.form-control,.form-select{font-size:14px}}.input-contrast{background-color:#fafbfc}.input-contrast:focus{background-color:#fff}:-ms-input-placeholder{color:#6a737d}::-ms-input-placeholder{color:#6a737d}::placeholder{color:#6a737d}.input-sm{min-height:28px;padding-top:3px;padding-bottom:3px;font-size:12px;line-height:20px}.input-lg{padding:4px 10px;font-size:16px}.input-block{display:block;width:100%}.input-monospace{font-family:"SFMono-Regular",Consolas,"Liberation Mono",Menlo,Courier,monospace}.input-hide-webkit-autofill::-webkit-contacts-auto-fill-button{position:absolute;right:0;display:none !important;pointer-events:none;visibility:hidden}.form-checkbox{padding-left:20px;margin:15px 0;vertical-align:middle}.form-checkbox label em.highlight{position:relative;left:-4px;padding:2px 4px;font-style:normal;background:#fffbdd;border-radius:3px}.form-checkbox input[type=checkbox],.form-checkbox input[type=radio]{float:left;margin:5px 0 0 -20px;vertical-align:middle}.form-checkbox .note{display:block;margin:0;font-size:12px;font-weight:400;color:#586069}.form-checkbox-details{display:none}.form-checkbox-details-trigger:checked ~ * .form-checkbox-details,.form-checkbox-details-trigger:checked ~ .form-checkbox-details{display:block}.hfields{margin:15px 0}.hfields::before{display:table;content:""}.hfields::after{display:table;clear:both;content:""}.hfields .form-group{float:left;margin:0 30px 0 0}.hfields .form-group dt label{display:inline-block;margin:5px 0 0;color:#586069}.hfields .form-group dt img{position:relative;top:-2px}.hfields .btn{float:left;margin:28px 25px 0 -20px}.hfields .form-select{margin-top:5px}input::-webkit-outer-spin-button,input::-webkit-inner-spin-button{margin:0;-webkit-appearance:none;appearance:none}.form-actions::before{display:table;content:""}.form-actions::after{display:table;clear:both;content:""}.form-actions .btn{float:right}.form-actions .btn+.btn{margin-right:5px}.form-warning{padding:8px 10px;margin:10px 0;font-size:14px;color:#735c0f;background:#fffbdd;border:1px solid #d9d0a5;border-radius:3px}.form-warning p{margin:0;line-height:1.5}.form-warning a{font-weight:600}.form-select{display:inline-block;max-width:100%;height:34px;padding-right:24px;padding-right:8px \9;background:#fff url("data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAUCAMAAACzvE1FAAAADFBMVEUzMzMzMzMzMzMzMzMKAG/3AAAAA3RSTlMAf4C/aSLHAAAAPElEQVR42q3NMQ4AIAgEQTn//2cLdRKppSGzBYwzVXvznNWs8C58CiussPJj8h6NwgorrKRdTvuV9v16Afn0AYFOB7aYAAAAAElFTkSuQmCC") no-repeat right 8px center;background-image:none \9;background-size:8px 10px;-webkit-appearance:none;-moz-appearance:none;appearance:none}.form-select::-ms-expand{opacity:0}.form-select[multiple]{height:auto}.select-sm{height:28px;min-height:28px;padding-top:3px;padding-bottom:3px;font-size:12px}.select-sm[multiple]{height:auto;min-height:0}.form-group{margin:15px 0}.form-group .form-control{width:440px;max-width:100%;margin-right:5px;background-color:#fafbfc}.form-group .form-control:focus{background-color:#fff}.form-group .form-control.shorter{width:130px}.form-group .form-control.short{width:250px}.form-group .form-control.long{width:100%}.form-group textarea.form-control{width:100%;height:200px;min-height:200px}.form-group textarea.form-control.short{height:50px;min-height:50px}.form-group dt{margin:0 0 6px}.form-group label{position:relative}.form-group.flattened dt{float:left;margin:0;line-height:32px}.form-group.flattened dd{line-height:32px}.form-group dd h4{margin:4px 0 0}.form-group dd h4.is-error{color:#cb2431}.form-group dd h4.is-success{color:#28a745}.form-group dd h4+.note{margin-top:0}.form-group.required dt label::after{padding-left:5px;color:#cb2431;content:"*"}.form-group .success,.form-group .error,.form-group .indicator{display:none;font-size:12px;font-weight:600}.form-group.loading{opacity:0.5}.form-group.loading .indicator{display:inline}.form-group.loading .spinner{display:inline-block;vertical-align:middle}.form-group.successful .success{display:inline;color:#28a745}.form-group.warn .warning,.form-group.warn .error,.form-group.errored .warning,.form-group.errored .error{position:absolute;z-index:10;display:block;max-width:450px;padding:5px 8px;margin:4px 0 0;font-size:13px;font-weight:400;border-style:solid;border-width:1px;border-radius:3px}.form-group.warn .warning::after,.form-group.warn .warning::before,.form-group.warn .error::after,.form-group.warn .error::before,.form-group.errored .warning::after,.form-group.errored .warning::before,.form-group.errored .error::after,.form-group.errored .error::before{position:absolute;bottom:100%;left:10px;z-index:15;width:0;height:0;pointer-events:none;content:" ";border:solid transparent}.form-group.warn .warning::after,.form-group.warn .error::after,.form-group.errored .warning::after,.form-group.errored .error::after{border-width:5px}.form-group.warn .warning::before,.form-group.warn .error::before,.form-group.errored .warning::before,.form-group.errored .error::before{margin-left:-1px;border-width:6px}.form-group.warn .warning{color:#735c0f;background-color:#fffbdd;border-color:#d9d0a5}.form-group.warn .warning::after{border-bottom-color:#fffbdd}.form-group.warn .warning::before{border-bottom-color:#d9d0a5}.form-group.errored label{color:#cb2431}.form-group.errored .error{color:#86181d;background-color:#ffdce0;border-color:#cea0a5}.form-group.errored .error::after{border-bottom-color:#ffdce0}.form-group.errored .error::before{border-bottom-color:#cea0a5}.note{min-height:17px;margin:4px 0 2px;font-size:12px;color:#586069}.note .spinner{margin-right:3px;vertical-align:middle}dl.form-group>dd .form-control.is-autocheck-loading,dl.form-group>dd .form-control.is-autocheck-successful,dl.form-group>dd .form-control.is-autocheck-errored{padding-right:30px}dl.form-group>dd .form-control.is-autocheck-loading{background-image:url("/images/spinners/octocat-spinner-16px.gif")}dl.form-group>dd .form-control.is-autocheck-successful{background-image:url("/images/modules/ajax/success.png")}dl.form-group>dd .form-control.is-autocheck-errored{background-image:url("/images/modules/ajax/error.png")}@media only screen and (-webkit-min-device-pixel-ratio: 2), only screen and (min--moz-device-pixel-ratio: 2), only screen and (-moz-min-device-pixel-ratio: 2), only screen and (min-device-pixel-ratio: 2), only screen and (min-resolution: 192dpi), only screen and (min-resolution: 2dppx){dl.form-group>dd .form-control.is-autocheck-loading,dl.form-group>dd .form-control.is-autocheck-successful,dl.form-group>dd .form-control.is-autocheck-errored{background-size:16px 16px}dl.form-group>dd .form-control.is-autocheck-loading{background-image:url("/images/spinners/octocat-spinner-32.gif")}dl.form-group>dd .form-control.is-autocheck-successful{background-image:url("/images/modules/ajax/success@2x.png")}dl.form-group>dd .form-control.is-autocheck-errored{background-image:url("/images/modules/ajax/error@2x.png")}}.status-indicator{display:inline-block;width:16px;height:16px;margin-left:5px}.status-indicator .octicon{display:none}.status-indicator-success::before{content:""}.status-indicator-success .octicon-check{display:inline-block;color:#28a745;fill:#28a745}.status-indicator-success .octicon-x{display:none}.status-indicator-failed::before{content:""}.status-indicator-failed .octicon-check{display:none}.status-indicator-failed .octicon-x{display:inline-block;color:#cb2431;fill:#d73a49}.status-indicator-loading{width:16px;background:url("/images/spinners/octocat-spinner-32-EAF2F5.gif") 0 0 no-repeat;background-size:16px}.inline-form{display:inline-block}.inline-form .btn-plain{background-color:transparent;border:0}.drag-and-drop{padding:7px 10px;margin:0;font-size:13px;line-height:16px;color:#586069;background-color:#fafbfc;border:1px solid #c3c8cf;border-top:0;border-bottom-right-radius:3px;border-bottom-left-radius:3px}.drag-and-drop .default,.drag-and-drop .loading,.drag-and-drop .error{display:none}.drag-and-drop .error{color:#cb2431}.drag-and-drop img{vertical-align:top}.is-default .drag-and-drop .default{display:inline-block}.is-uploading .drag-and-drop .loading{display:inline-block}.is-bad-file .drag-and-drop .bad-file{display:inline-block}.is-duplicate-filename .drag-and-drop .duplicate-filename{display:inline-block}.is-too-big .drag-and-drop .too-big{display:inline-block}.is-hidden-file .drag-and-drop .hidden-file{display:inline-block}.is-empty .drag-and-drop .empty{display:inline-block}.is-bad-permissions .drag-and-drop .bad-permissions{display:inline-block}.is-repository-required .drag-and-drop .repository-required{display:inline-block}.drag-and-drop-error-info{font-weight:400;color:#586069}.drag-and-drop-error-info a{color:#0366d6}.is-failed .drag-and-drop .failed-request{display:inline-block}.manual-file-chooser{position:absolute;width:240px;padding:5px;margin-left:-80px;cursor:pointer;opacity:0.0001}.manual-file-chooser:hover+.manual-file-chooser-text{text-decoration:underline}.btn .manual-file-chooser{top:0;padding:0;line-height:34px}.upload-enabled textarea{display:block;border-bottom:1px dashed #dfe2e5;border-bottom-right-radius:0;border-bottom-left-radius:0}.upload-enabled.focused{border-radius:3px;box-shadow:inset 0 1px 2px rgba(27,31,35,0.075),0 0 0 0.2em rgba(3,102,214,0.3)}.upload-enabled.focused .form-control{box-shadow:none}.upload-enabled.focused .drag-and-drop{border-color:#4a9eff}.dragover textarea,.dragover .drag-and-drop{box-shadow:#c9ff00 0 0 3px}.write-content{position:relative}.previewable-comment-form{position:relative}.previewable-comment-form .tabnav{position:relative;padding:8px 8px 0}.previewable-comment-form .comment{border:1px solid #c3c8cf}.previewable-comment-form .comment-form-error{margin-bottom:8px}.previewable-comment-form .write-content,.previewable-comment-form .preview-content{display:none;margin:0 8px 8px}.previewable-comment-form.write-selected .write-content,.previewable-comment-form.preview-selected .preview-content{display:block}.previewable-comment-form textarea{display:block;width:100%;min-height:100px;max-height:500px;padding:8px;resize:vertical}.form-action-spacious{margin-top:10px}div.composer{margin-top:0;border:0}.composer .comment-form-textarea{height:200px;min-height:200px}.composer .tabnav{margin:0 0 10px}h2.account{margin:15px 0 0;font-size:18px;font-weight:400;color:#586069}p.explain{position:relative;font-size:12px;color:#586069}p.explain strong{color:#24292e}p.explain .octicon{margin-right:5px;color:#959da5}p.explain .minibutton{top:-4px;float:right}.form-group label{position:static}.input-group{display:table}.input-group .form-control{position:relative;width:100%}.input-group .form-control:focus{z-index:2}.input-group .form-control+.btn{margin-left:0}.input-group.inline{display:inline-table}.input-group .form-control,.input-group-button{display:table-cell}.input-group-button{width:1%;vertical-align:middle}.input-group .form-control:first-child,.input-group-button:first-child .btn{border-top-right-radius:0;border-bottom-right-radius:0}.input-group-button:first-child .btn{margin-right:-1px}.input-group .form-control:last-child,.input-group-button:last-child .btn{border-top-left-radius:0;border-bottom-left-radius:0}.input-group-button:last-child .btn{margin-left:-1px}.container{width:980px;margin-right:auto;margin-left:auto}.container::before{display:table;content:""}.container::after{display:table;clear:both;content:""}.container-md{max-width:768px;margin-right:auto;margin-left:auto}.container-lg{max-width:1012px;margin-right:auto;margin-left:auto}.container-xl{max-width:1280px;margin-right:auto;margin-left:auto}.columns{margin-right:-10px;margin-left:-10px}.columns::before{display:table;content:""}.columns::after{display:table;clear:both;content:""}.column{float:left;padding-right:10px;padding-left:10px}.one-third{width:33.333333%}.two-thirds{width:66.666667%}.one-fourth{width:25%}.one-half{width:50%}.three-fourths{width:75%}.one-fifth{width:20%}.four-fifths{width:80%}.centered{display:block;float:none;margin-right:auto;margin-left:auto}.col-1{width:8.33333%}.col-2{width:16.66667%}.col-3{width:25%}.col-4{width:33.33333%}.col-5{width:41.66667%}.col-6{width:50%}.col-7{width:58.33333%}.col-8{width:66.66667%}.col-9{width:75%}.col-10{width:83.33333%}.col-11{width:91.66667%}.col-12{width:100%}@media (min-width: 544px){.col-sm-1{width:8.33333%}.col-sm-2{width:16.66667%}.col-sm-3{width:25%}.col-sm-4{width:33.33333%}.col-sm-5{width:41.66667%}.col-sm-6{width:50%}.col-sm-7{width:58.33333%}.col-sm-8{width:66.66667%}.col-sm-9{width:75%}.col-sm-10{width:83.33333%}.col-sm-11{width:91.66667%}.col-sm-12{width:100%}}@media (min-width: 768px){.col-md-1{width:8.33333%}.col-md-2{width:16.66667%}.col-md-3{width:25%}.col-md-4{width:33.33333%}.col-md-5{width:41.66667%}.col-md-6{width:50%}.col-md-7{width:58.33333%}.col-md-8{width:66.66667%}.col-md-9{width:75%}.col-md-10{width:83.33333%}.col-md-11{width:91.66667%}.col-md-12{width:100%}}@media (min-width: 1012px){.col-lg-1{width:8.33333%}.col-lg-2{width:16.66667%}.col-lg-3{width:25%}.col-lg-4{width:33.33333%}.col-lg-5{width:41.66667%}.col-lg-6{width:50%}.col-lg-7{width:58.33333%}.col-lg-8{width:66.66667%}.col-lg-9{width:75%}.col-lg-10{width:83.33333%}.col-lg-11{width:91.66667%}.col-lg-12{width:100%}}@media (min-width: 1280px){.col-xl-1{width:8.33333%}.col-xl-2{width:16.66667%}.col-xl-3{width:25%}.col-xl-4{width:33.33333%}.col-xl-5{width:41.66667%}.col-xl-6{width:50%}.col-xl-7{width:58.33333%}.col-xl-8{width:66.66667%}.col-xl-9{width:75%}.col-xl-10{width:83.33333%}.col-xl-11{width:91.66667%}.col-xl-12{width:100%}}.gutter{margin-right:-16px;margin-left:-16px}.gutter>[class*="col-"]{padding-right:16px !important;padding-left:16px !important}.gutter-condensed{margin-right:-8px;margin-left:-8px}.gutter-condensed>[class*="col-"]{padding-right:8px !important;padding-left:8px !important}.gutter-spacious{margin-right:-24px;margin-left:-24px}.gutter-spacious>[class*="col-"]{padding-right:24px !important;padding-left:24px !important}@media (min-width: 544px){.gutter-sm{margin-right:-16px;margin-left:-16px}.gutter-sm>[class*="col-"]{padding-right:16px !important;padding-left:16px !important}.gutter-sm-condensed{margin-right:-8px;margin-left:-8px}.gutter-sm-condensed>[class*="col-"]{padding-right:8px !important;padding-left:8px !important}.gutter-sm-spacious{margin-right:-24px;margin-left:-24px}.gutter-sm-spacious>[class*="col-"]{padding-right:24px !important;padding-left:24px !important}}@media (min-width: 768px){.gutter-md{margin-right:-16px;margin-left:-16px}.gutter-md>[class*="col-"]{padding-right:16px !important;padding-left:16px !important}.gutter-md-condensed{margin-right:-8px;margin-left:-8px}.gutter-md-condensed>[class*="col-"]{padding-right:8px !important;padding-left:8px !important}.gutter-md-spacious{margin-right:-24px;margin-left:-24px}.gutter-md-spacious>[class*="col-"]{padding-right:24px !important;padding-left:24px !important}}@media (min-width: 1012px){.gutter-lg{margin-right:-16px;margin-left:-16px}.gutter-lg>[class*="col-"]{padding-right:16px !important;padding-left:16px !important}.gutter-lg-condensed{margin-right:-8px;margin-left:-8px}.gutter-lg-condensed>[class*="col-"]{padding-right:8px !important;padding-left:8px !important}.gutter-lg-spacious{margin-right:-24px;margin-left:-24px}.gutter-lg-spacious>[class*="col-"]{padding-right:24px !important;padding-left:24px !important}}@media (min-width: 1280px){.gutter-xl{margin-right:-16px;margin-left:-16px}.gutter-xl>[class*="col-"]{padding-right:16px !important;padding-left:16px !important}.gutter-xl-condensed{margin-right:-8px;margin-left:-8px}.gutter-xl-condensed>[class*="col-"]{padding-right:8px !important;padding-left:8px !important}.gutter-xl-spacious{margin-right:-24px;margin-left:-24px}.gutter-xl-spacious>[class*="col-"]{padding-right:24px !important;padding-left:24px !important}}.offset-1{margin-left:8.33333%}.offset-2{margin-left:16.66667%}.offset-3{margin-left:25%}.offset-4{margin-left:33.33333%}.offset-5{margin-left:41.66667%}.offset-6{margin-left:50%}.offset-7{margin-left:58.33333%}.offset-8{margin-left:66.66667%}.offset-9{margin-left:75%}.offset-10{margin-left:83.33333%}.offset-11{margin-left:91.66667%}@media (min-width: 544px){.offset-sm-1{margin-left:8.33333%}.offset-sm-2{margin-left:16.66667%}.offset-sm-3{margin-left:25%}.offset-sm-4{margin-left:33.33333%}.offset-sm-5{margin-left:41.66667%}.offset-sm-6{margin-left:50%}.offset-sm-7{margin-left:58.33333%}.offset-sm-8{margin-left:66.66667%}.offset-sm-9{margin-left:75%}.offset-sm-10{margin-left:83.33333%}.offset-sm-11{margin-left:91.66667%}}@media (min-width: 768px){.offset-md-1{margin-left:8.33333%}.offset-md-2{margin-left:16.66667%}.offset-md-3{margin-left:25%}.offset-md-4{margin-left:33.33333%}.offset-md-5{margin-left:41.66667%}.offset-md-6{margin-left:50%}.offset-md-7{margin-left:58.33333%}.offset-md-8{margin-left:66.66667%}.offset-md-9{margin-left:75%}.offset-md-10{margin-left:83.33333%}.offset-md-11{margin-left:91.66667%}}@media (min-width: 1012px){.offset-lg-1{margin-left:8.33333%}.offset-lg-2{margin-left:16.66667%}.offset-lg-3{margin-left:25%}.offset-lg-4{margin-left:33.33333%}.offset-lg-5{margin-left:41.66667%}.offset-lg-6{margin-left:50%}.offset-lg-7{margin-left:58.33333%}.offset-lg-8{margin-left:66.66667%}.offset-lg-9{margin-left:75%}.offset-lg-10{margin-left:83.33333%}.offset-lg-11{margin-left:91.66667%}}@media (min-width: 1280px){.offset-xl-1{margin-left:8.33333%}.offset-xl-2{margin-left:16.66667%}.offset-xl-3{margin-left:25%}.offset-xl-4{margin-left:33.33333%}.offset-xl-5{margin-left:41.66667%}.offset-xl-6{margin-left:50%}.offset-xl-7{margin-left:58.33333%}.offset-xl-8{margin-left:66.66667%}.offset-xl-9{margin-left:75%}.offset-xl-10{margin-left:83.33333%}.offset-xl-11{margin-left:91.66667%}}.menu{margin-bottom:15px;list-style:none;background-color:#fff;border:1px solid #d1d5da;border-radius:3px}.menu-item{position:relative;display:block;padding:8px 10px;border-bottom:1px solid #e1e4e8}.menu-item:first-child{border-top:0;border-top-left-radius:2px;border-top-right-radius:2px}.menu-item:first-child::before{border-top-left-radius:2px}.menu-item:last-child{border-bottom:0;border-bottom-right-radius:2px;border-bottom-left-radius:2px}.menu-item:last-child::before{border-bottom-left-radius:2px}.menu-item:hover{text-decoration:none;background-color:#f6f8fa}.menu-item.selected{font-weight:600;color:#24292e;cursor:default;background-color:#fff}.menu-item.selected::before{position:absolute;top:0;bottom:0;left:0;width:2px;content:"";background-color:#e36209}.menu-item .octicon{width:16px;margin-right:5px;color:#24292e;text-align:center}.menu-item .Counter{float:right;margin-left:5px}.menu-item .menu-warning{float:right;color:#86181d}.menu-item .avatar{float:left;margin-right:5px}.menu-item.alert .Counter{color:#cb2431}.menu-heading{display:block;padding:8px 10px;margin-top:0;margin-bottom:0;font-size:13px;font-weight:600;line-height:20px;color:#586069;background-color:#f3f5f8;border-bottom:1px solid #e1e4e8}.menu-heading:hover{text-decoration:none}.menu-heading:first-child{border-top-left-radius:2px;border-top-right-radius:2px}.menu-heading:last-child{border-bottom:0;border-bottom-right-radius:2px;border-bottom-left-radius:2px}.tabnav{margin-top:0;margin-bottom:15px;border-bottom:1px solid #d1d5da}.tabnav .Counter{margin-left:5px}.tabnav-tabs{margin-bottom:-1px}.tabnav-tab{display:inline-block;padding:8px 12px;font-size:14px;line-height:20px;color:#586069;text-decoration:none;background-color:transparent;border:1px solid transparent;border-bottom:0}.tabnav-tab.selected{color:#24292e;background-color:#fff;border-color:#d1d5da;border-radius:3px 3px 0 0}.tabnav-tab:hover,.tabnav-tab:focus{color:#24292e;text-decoration:none}.tabnav-extra{display:inline-block;padding-top:10px;margin-left:10px;font-size:12px;color:#586069}.tabnav-extra>.octicon{margin-right:2px}a.tabnav-extra:hover{color:#0366d6;text-decoration:none}.tabnav-btn{margin-left:10px}.filter-list{list-style-type:none}.filter-list.small .filter-item{padding:4px 10px;margin:0 0 2px;font-size:12px}.filter-list.pjax-active .filter-item{color:#586069;background-color:transparent}.filter-list.pjax-active .filter-item.pjax-active{color:#fff;background-color:#0366d6}.filter-item{position:relative;display:block;padding:8px 10px;margin-bottom:5px;overflow:hidden;font-size:14px;color:#586069;text-decoration:none;text-overflow:ellipsis;white-space:nowrap;cursor:pointer;border-radius:3px}.filter-item:hover{text-decoration:none;background-color:#eaecef}.filter-item.selected{color:#fff;background-color:#0366d6}.filter-item .count{float:right;font-weight:600}.filter-item .bar{position:absolute;top:2px;right:0;bottom:2px;z-index:-1;display:inline-block;background-color:#eff3f6}.subnav{margin-bottom:20px}.subnav::before{display:table;content:""}.subnav::after{display:table;clear:both;content:""}.subnav-bordered{padding-bottom:20px;border-bottom:1px solid #eaecef}.subnav-flush{margin-bottom:0}.subnav-item{position:relative;float:left;padding:6px 14px;font-weight:600;line-height:20px;color:#586069;border:1px solid #e1e4e8}.subnav-item+.subnav-item{margin-left:-1px}.subnav-item:hover,.subnav-item:focus{text-decoration:none;background-color:#f6f8fa}.subnav-item.selected,.subnav-item.selected:hover,.subnav-item.selected:focus{z-index:2;color:#fff;background-color:#0366d6;border-color:#0366d6}.subnav-item:first-child{border-top-left-radius:3px;border-bottom-left-radius:3px}.subnav-item:last-child{border-top-right-radius:3px;border-bottom-right-radius:3px}.subnav-search{position:relative;margin-left:10px}.subnav-search-input{width:320px;padding-left:30px;color:#586069}.subnav-search-input-wide{width:500px}.subnav-search-icon{position:absolute;top:9px;left:8px;display:block;color:#c6cbd1;text-align:center;pointer-events:none}.subnav-search-context .btn{color:#444d56;border-top-right-radius:0;border-bottom-right-radius:0}.subnav-search-context .btn:hover,.subnav-search-context .btn:focus,.subnav-search-context .btn:active,.subnav-search-context .btn.selected{z-index:2}.subnav-search-context+.subnav-search{margin-left:-1px}.subnav-search-context+.subnav-search .subnav-search-input{border-top-left-radius:0;border-bottom-left-radius:0}.subnav-search-context .select-menu-modal-holder{z-index:30}.subnav-search-context .select-menu-modal{width:220px}.subnav-search-context .select-menu-item-icon{color:inherit}.subnav-spacer-right{padding-right:10px}.UnderlineNav{display:flex;justify-content:space-between;border-bottom:1px solid #e1e4e8}.UnderlineNav-body{display:flex;margin-bottom:-1px}.UnderlineNav-item{padding:16px 8px;margin-right:16px;font-size:14px;line-height:1.5;color:#586069;text-align:center;border-bottom:2px solid transparent}.UnderlineNav-item:hover,.UnderlineNav-item:focus{color:#24292e;text-decoration:none;border-bottom-color:#d1d5da;transition:0.2s ease}.UnderlineNav-item:hover .UnderlineNav-octicon,.UnderlineNav-item:focus .UnderlineNav-octicon{color:#6a737d}.UnderlineNav-item.selected{font-weight:600;color:#24292e;border-bottom-color:#e36209}.UnderlineNav-item.selected .UnderlineNav-octicon{color:#6a737d}.UnderlineNav--right{justify-content:flex-end}.UnderlineNav--right .UnderlineNav-item{margin-right:0;margin-left:16px}.UnderlineNav--right .UnderlineNav-actions{flex:1 1 auto}.UnderlineNav-actions{align-self:center}.UnderlineNav--full{display:block}.UnderlineNav-octicon{color:#959da5}.UnderlineNav-container{display:flex;justify-content:space-between}.pagination::before{display:table;content:""}.pagination::after{display:table;clear:both;content:""}.pagination a,.pagination span,.pagination em{position:relative;float:left;padding:7px 12px;margin-left:-1px;font-size:13px;font-style:normal;font-weight:600;color:#0366d6;white-space:nowrap;vertical-align:middle;cursor:pointer;-webkit-user-select:none;-moz-user-select:none;-ms-user-select:none;user-select:none;background:#fff;border:1px solid #e1e4e8}.pagination a:first-child,.pagination span:first-child,.pagination em:first-child{margin-left:0;border-top-left-radius:3px;border-bottom-left-radius:3px}.pagination a:last-child,.pagination span:last-child,.pagination em:last-child{border-top-right-radius:3px;border-bottom-right-radius:3px}.pagination a:hover,.pagination a:focus,.pagination span:hover,.pagination span:focus,.pagination em:hover,.pagination em:focus{z-index:2;text-decoration:none;background-color:#eff3f6;border-color:#e1e4e8}.pagination .selected{z-index:3}.pagination .current,.pagination .current:hover{z-index:3;color:#fff;background-color:#0366d6;border-color:#0366d6}.pagination .gap,.pagination .disabled,.pagination .gap:hover,.pagination .disabled:hover{color:#d1d5da;cursor:default;background-color:#fafbfc}.paginate-container{margin-top:20px;margin-bottom:15px;text-align:center}.paginate-container .pagination{display:inline-block}.tooltipped{position:relative}.tooltipped::after{position:absolute;z-index:1000000;display:none;padding:.5em .75em;font:normal normal 11px/1.5 -apple-system,BlinkMacSystemFont,"Segoe UI",Helvetica,Arial,sans-serif,"Apple Color Emoji","Segoe UI Emoji","Segoe UI Symbol";-webkit-font-smoothing:subpixel-antialiased;color:#fff;text-align:center;text-decoration:none;text-shadow:none;text-transform:none;letter-spacing:normal;word-wrap:break-word;white-space:pre;pointer-events:none;content:attr(aria-label);background:#1b1f23;border-radius:3px;opacity:0}.tooltipped::before{position:absolute;z-index:1000001;display:none;width:0;height:0;color:#1b1f23;pointer-events:none;content:"";border:6px solid transparent;opacity:0}@keyframes tooltip-appear{from{opacity:0}to{opacity:1}}.tooltipped:hover::before,.tooltipped:hover::after,.tooltipped:active::before,.tooltipped:active::after,.tooltipped:focus::before,.tooltipped:focus::after{display:inline-block;text-decoration:none;animation-name:tooltip-appear;animation-duration:.1s;animation-fill-mode:forwards;animation-timing-function:ease-in;animation-delay:.4s}.tooltipped-no-delay:hover::before,.tooltipped-no-delay:hover::after,.tooltipped-no-delay:active::before,.tooltipped-no-delay:active::after,.tooltipped-no-delay:focus::before,.tooltipped-no-delay:focus::after{animation-delay:0s}.tooltipped-multiline:hover::after,.tooltipped-multiline:active::after,.tooltipped-multiline:focus::after{display:table-cell}.tooltipped-s::after,.tooltipped-se::after,.tooltipped-sw::after{top:100%;right:50%;margin-top:6px}.tooltipped-s::before,.tooltipped-se::before,.tooltipped-sw::before{top:auto;right:50%;bottom:-7px;margin-right:-6px;border-bottom-color:#1b1f23}.tooltipped-se::after{right:auto;left:50%;margin-left:-16px}.tooltipped-sw::after{margin-right:-16px}.tooltipped-n::after,.tooltipped-ne::after,.tooltipped-nw::after{right:50%;bottom:100%;margin-bottom:6px}.tooltipped-n::before,.tooltipped-ne::before,.tooltipped-nw::before{top:-7px;right:50%;bottom:auto;margin-right:-6px;border-top-color:#1b1f23}.tooltipped-ne::after{right:auto;left:50%;margin-left:-16px}.tooltipped-nw::after{margin-right:-16px}.tooltipped-s::after,.tooltipped-n::after{transform:translateX(50%)}.tooltipped-w::after{right:100%;bottom:50%;margin-right:6px;transform:translateY(50%)}.tooltipped-w::before{top:50%;bottom:50%;left:-7px;margin-top:-6px;border-left-color:#1b1f23}.tooltipped-e::after{bottom:50%;left:100%;margin-left:6px;transform:translateY(50%)}.tooltipped-e::before{top:50%;right:-7px;bottom:50%;margin-top:-6px;border-right-color:#1b1f23}.tooltipped-align-right-1::after,.tooltipped-align-right-2::after{right:0;margin-right:0}.tooltipped-align-right-1::before{right:10px}.tooltipped-align-right-2::before{right:15px}.tooltipped-align-left-1::after,.tooltipped-align-left-2::after{left:0;margin-left:0}.tooltipped-align-left-1::before{left:5px}.tooltipped-align-left-2::before{left:10px}.tooltipped-multiline::after{width:-webkit-max-content;width:-moz-max-content;width:max-content;max-width:250px;word-wrap:break-word;white-space:pre-line;border-collapse:separate}.tooltipped-multiline.tooltipped-s::after,.tooltipped-multiline.tooltipped-n::after{right:auto;left:50%;transform:translateX(-50%)}.tooltipped-multiline.tooltipped-w::after,.tooltipped-multiline.tooltipped-e::after{right:100%}@media screen and (min-width: 0\0){.tooltipped-multiline::after{width:250px}}.tooltipped-sticky::before,.tooltipped-sticky::after{display:inline-block}.tooltipped-sticky.tooltipped-multiline::after{display:table-cell}.css-truncate.css-truncate-target,.css-truncate .css-truncate-target{display:inline-block;max-width:125px;overflow:hidden;text-overflow:ellipsis;white-space:nowrap;vertical-align:top}.css-truncate.expandable.zeroclipboard-is-hover .css-truncate-target,.css-truncate.expandable.zeroclipboard-is-hover.css-truncate-target,.css-truncate.expandable:hover .css-truncate-target,.css-truncate.expandable:hover.css-truncate-target{max-width:10000px !important}.anim-fade-in{animation-name:fade-in;animation-duration:1s;animation-timing-function:ease-in-out}.anim-fade-in.fast{animation-duration:300ms}@keyframes fade-in{0%{opacity:0}100%{opacity:1}}.anim-fade-out{animation-name:fade-out;animation-duration:1s;animation-timing-function:ease-out}.anim-fade-out.fast{animation-duration:0.3s}@keyframes fade-out{0%{opacity:1}100%{opacity:0}}.anim-fade-up{opacity:0;animation-name:fade-up;animation-duration:0.3s;animation-fill-mode:forwards;animation-timing-function:ease-out;animation-delay:1s}@keyframes fade-up{0%{opacity:0.8;transform:translateY(100%)}100%{opacity:1;transform:translateY(0)}}.anim-fade-down{animation-name:fade-down;animation-duration:0.3s;animation-fill-mode:forwards;animation-timing-function:ease-in}@keyframes fade-down{0%{opacity:1;transform:translateY(0)}100%{opacity:0.5;transform:translateY(100%)}}.anim-grow-x{width:0%;animation-name:grow-x;animation-duration:0.3s;animation-fill-mode:forwards;animation-timing-function:ease;animation-delay:0.5s}@keyframes grow-x{to{width:100%}}.anim-shrink-x{animation-name:shrink-x;animation-duration:0.3s;animation-fill-mode:forwards;animation-timing-function:ease-in-out;animation-delay:0.5s}@keyframes shrink-x{to{width:0%}}.anim-scale-in{animation-name:scale-in;animation-duration:0.15s;animation-timing-function:cubic-bezier(0.2, 0, 0.13, 1.5)}@keyframes scale-in{0%{opacity:0;transform:scale(0.5)}100%{opacity:1;transform:scale(1)}}.anim-pulse{animation-name:pulse;animation-duration:2s;animation-timing-function:linear;animation-iteration-count:infinite}@keyframes pulse{0%{opacity:0.3}10%{opacity:1}100%{opacity:0.3}}.anim-pulse-in{animation-name:pulse-in;animation-duration:0.5s}@keyframes pulse-in{0%{transform:scale3d(1, 1, 1)}50%{transform:scale3d(1.1, 1.1, 1.1)}100%{transform:scale3d(1, 1, 1)}}.hover-grow{transition:transform 0.3s;-webkit-backface-visibility:hidden;backface-visibility:hidden}.hover-grow:hover{transform:scale(1.025)}.border{border:1px #e1e4e8 solid !important}.border-top{border-top:1px #e1e4e8 solid !important}.border-right{border-right:1px #e1e4e8 solid !important}.border-bottom{border-bottom:1px #e1e4e8 solid !important}.border-left{border-left:1px #e1e4e8 solid !important}.border-y{border-top:1px #e1e4e8 solid !important;border-bottom:1px #e1e4e8 solid !important}.border-dashed{border-style:dashed !important}.border-blue{border-color:#0366d6 !important}.border-blue-light{border-color:#c8e1ff !important}.border-green{border-color:#34d058 !important}.border-green-light{border-color:#a2cbac !important}.border-red{border-color:#d73a49 !important}.border-red-light{border-color:#cea0a5 !important}.border-purple{border-color:#6f42c1 !important}.border-yellow{border-color:#d9d0a5 !important}.border-gray-light{border-color:#eaecef !important}.border-gray-dark{border-color:#d1d5da !important}.border-black-fade{border-color:rgba(27,31,35,0.15) !important}.border-0{border:0 !important}.border-top-0{border-top:0 !important}.border-right-0{border-right:0 !important}.border-bottom-0{border-bottom:0 !important}.border-left-0{border-left:0 !important}.rounded-0{border-radius:0 !important}.rounded-1{border-radius:3px !important}.rounded-2{border-radius:6px !important}.circle{border-radius:50% !important}.box-shadow{box-shadow:0 1px 1px rgba(27,31,35,0.1) !important}.box-shadow-medium{box-shadow:0 1px 5px rgba(27,31,35,0.15) !important}.box-shadow-large{box-shadow:0 1px 15px rgba(27,31,35,0.15) !important}.box-shadow-extra-large{box-shadow:0 10px 50px rgba(27,31,35,0.07) !important}.box-shadow-none{box-shadow:none !important}.bg-white{background-color:#fff !important}.bg-blue{background-color:#0366d6 !important}.bg-blue-light{background-color:#f1f8ff !important}.bg-gray-dark{background-color:#24292e !important}.bg-gray{background-color:#f6f8fa !important}.bg-gray-light{background-color:#fafbfc !important}.bg-green{background-color:#28a745 !important}.bg-green-light{background-color:#dcffe4 !important}.bg-red{background-color:#d73a49 !important}.bg-red-light{background-color:#ffdce0 !important}.bg-yellow{background-color:#ffd33d !important}.bg-yellow-light{background-color:#fff5b1 !important}.bg-purple{background-color:#6f42c1 !important}.bg-purple-light{background-color:#f5f0ff !important}.bg-shade-gradient{background-image:linear-gradient(180deg, rgba(27,31,35,0.065), rgba(27,31,35,0)) !important;background-repeat:no-repeat !important;background-size:100% 200px !important}.text-blue{color:#0366d6 !important}.text-red{color:#cb2431 !important}.text-gray-light{color:#6a737d !important}.text-gray{color:#586069 !important}.text-gray-dark{color:#24292e !important}.text-green{color:#28a745 !important}.text-orange{color:#a04100 !important}.text-orange-light{color:#e36209 !important}.text-purple{color:#6f42c1 !important}.text-white{color:#fff !important}.text-inherit{color:inherit !important}.text-pending{color:#b08800 !important}.bg-pending{color:#dbab09 !important}.link-gray{color:#586069 !important}.link-gray:hover{color:#0366d6 !important}.link-gray-dark{color:#24292e !important}.link-gray-dark:hover{color:#0366d6 !important}.link-hover-blue:hover{color:#0366d6 !important}.muted-link{color:#586069 !important}.muted-link:hover{color:#0366d6 !important;text-decoration:none}.details-overlay[open]>summary::before{position:fixed;top:0;right:0;bottom:0;left:0;z-index:80;display:block;cursor:default;content:" ";background:transparent}.details-overlay-dark[open]>summary::before{z-index:99;background:rgba(27,31,35,0.5)}.flex-row{flex-direction:row !important}.flex-row-reverse{flex-direction:row-reverse !important}.flex-column{flex-direction:column !important}.flex-wrap{flex-wrap:wrap !important}.flex-nowrap{flex-wrap:nowrap !important}.flex-justify-start{justify-content:flex-start !important}.flex-justify-end{justify-content:flex-end !important}.flex-justify-center{justify-content:center !important}.flex-justify-between{justify-content:space-between !important}.flex-justify-around{justify-content:space-around !important}.flex-items-start{align-items:flex-start !important}.flex-items-end{align-items:flex-end !important}.flex-items-center{align-items:center !important}.flex-items-baseline{align-items:baseline !important}.flex-items-stretch{align-items:stretch !important}.flex-content-start{align-content:flex-start !important}.flex-content-end{align-content:flex-end !important}.flex-content-center{align-content:center !important}.flex-content-between{align-content:space-between !important}.flex-content-around{align-content:space-around !important}.flex-content-stretch{align-content:stretch !important}.flex-auto{flex:1 1 auto !important}.flex-shrink-0{flex-shrink:0 !important}.flex-self-auto{align-self:auto !important}.flex-self-start{align-self:flex-start !important}.flex-self-end{align-self:flex-end !important}.flex-self-center{align-self:center !important}.flex-self-baseline{align-self:baseline !important}.flex-self-stretch{align-self:stretch !important}.flex-item-equal{flex-grow:1;flex-basis:0}@media (min-width: 544px){.flex-sm-row{flex-direction:row !important}.flex-sm-row-reverse{flex-direction:row-reverse !important}.flex-sm-column{flex-direction:column !important}.flex-sm-wrap{flex-wrap:wrap !important}.flex-sm-nowrap{flex-wrap:nowrap !important}.flex-sm-justify-start{justify-content:flex-start !important}.flex-sm-justify-end{justify-content:flex-end !important}.flex-sm-justify-center{justify-content:center !important}.flex-sm-justify-between{justify-content:space-between !important}.flex-sm-justify-around{justify-content:space-around !important}.flex-sm-items-start{align-items:flex-start !important}.flex-sm-items-end{align-items:flex-end !important}.flex-sm-items-center{align-items:center !important}.flex-sm-items-baseline{align-items:baseline !important}.flex-sm-items-stretch{align-items:stretch !important}.flex-sm-content-start{align-content:flex-start !important}.flex-sm-content-end{align-content:flex-end !important}.flex-sm-content-center{align-content:center !important}.flex-sm-content-between{align-content:space-between !important}.flex-sm-content-around{align-content:space-around !important}.flex-sm-content-stretch{align-content:stretch !important}.flex-sm-auto{flex:1 1 auto !important}.flex-sm-shrink-0{flex-shrink:0 !important}.flex-sm-self-auto{align-self:auto !important}.flex-sm-self-start{align-self:flex-start !important}.flex-sm-self-end{align-self:flex-end !important}.flex-sm-self-center{align-self:center !important}.flex-sm-self-baseline{align-self:baseline !important}.flex-sm-self-stretch{align-self:stretch !important}.flex-sm-item-equal{flex-grow:1;flex-basis:0}}@media (min-width: 768px){.flex-md-row{flex-direction:row !important}.flex-md-row-reverse{flex-direction:row-reverse !important}.flex-md-column{flex-direction:column !important}.flex-md-wrap{flex-wrap:wrap !important}.flex-md-nowrap{flex-wrap:nowrap !important}.flex-md-justify-start{justify-content:flex-start !important}.flex-md-justify-end{justify-content:flex-end !important}.flex-md-justify-center{justify-content:center !important}.flex-md-justify-between{justify-content:space-between !important}.flex-md-justify-around{justify-content:space-around !important}.flex-md-items-start{align-items:flex-start !important}.flex-md-items-end{align-items:flex-end !important}.flex-md-items-center{align-items:center !important}.flex-md-items-baseline{align-items:baseline !important}.flex-md-items-stretch{align-items:stretch !important}.flex-md-content-start{align-content:flex-start !important}.flex-md-content-end{align-content:flex-end !important}.flex-md-content-center{align-content:center !important}.flex-md-content-between{align-content:space-between !important}.flex-md-content-around{align-content:space-around !important}.flex-md-content-stretch{align-content:stretch !important}.flex-md-auto{flex:1 1 auto !important}.flex-md-shrink-0{flex-shrink:0 !important}.flex-md-self-auto{align-self:auto !important}.flex-md-self-start{align-self:flex-start !important}.flex-md-self-end{align-self:flex-end !important}.flex-md-self-center{align-self:center !important}.flex-md-self-baseline{align-self:baseline !important}.flex-md-self-stretch{align-self:stretch !important}.flex-md-item-equal{flex-grow:1;flex-basis:0}}@media (min-width: 1012px){.flex-lg-row{flex-direction:row !important}.flex-lg-row-reverse{flex-direction:row-reverse !important}.flex-lg-column{flex-direction:column !important}.flex-lg-wrap{flex-wrap:wrap !important}.flex-lg-nowrap{flex-wrap:nowrap !important}.flex-lg-justify-start{justify-content:flex-start !important}.flex-lg-justify-end{justify-content:flex-end !important}.flex-lg-justify-center{justify-content:center !important}.flex-lg-justify-between{justify-content:space-between !important}.flex-lg-justify-around{justify-content:space-around !important}.flex-lg-items-start{align-items:flex-start !important}.flex-lg-items-end{align-items:flex-end !important}.flex-lg-items-center{align-items:center !important}.flex-lg-items-baseline{align-items:baseline !important}.flex-lg-items-stretch{align-items:stretch !important}.flex-lg-content-start{align-content:flex-start !important}.flex-lg-content-end{align-content:flex-end !important}.flex-lg-content-center{align-content:center !important}.flex-lg-content-between{align-content:space-between !important}.flex-lg-content-around{align-content:space-around !important}.flex-lg-content-stretch{align-content:stretch !important}.flex-lg-auto{flex:1 1 auto !important}.flex-lg-shrink-0{flex-shrink:0 !important}.flex-lg-self-auto{align-self:auto !important}.flex-lg-self-start{align-self:flex-start !important}.flex-lg-self-end{align-self:flex-end !important}.flex-lg-self-center{align-self:center !important}.flex-lg-self-baseline{align-self:baseline !important}.flex-lg-self-stretch{align-self:stretch !important}.flex-lg-item-equal{flex-grow:1;flex-basis:0}}@media (min-width: 1280px){.flex-xl-row{flex-direction:row !important}.flex-xl-row-reverse{flex-direction:row-reverse !important}.flex-xl-column{flex-direction:column !important}.flex-xl-wrap{flex-wrap:wrap !important}.flex-xl-nowrap{flex-wrap:nowrap !important}.flex-xl-justify-start{justify-content:flex-start !important}.flex-xl-justify-end{justify-content:flex-end !important}.flex-xl-justify-center{justify-content:center !important}.flex-xl-justify-between{justify-content:space-between !important}.flex-xl-justify-around{justify-content:space-around !important}.flex-xl-items-start{align-items:flex-start !important}.flex-xl-items-end{align-items:flex-end !important}.flex-xl-items-center{align-items:center !important}.flex-xl-items-baseline{align-items:baseline !important}.flex-xl-items-stretch{align-items:stretch !important}.flex-xl-content-start{align-content:flex-start !important}.flex-xl-content-end{align-content:flex-end !important}.flex-xl-content-center{align-content:center !important}.flex-xl-content-between{align-content:space-between !important}.flex-xl-content-around{align-content:space-around !important}.flex-xl-content-stretch{align-content:stretch !important}.flex-xl-auto{flex:1 1 auto !important}.flex-xl-shrink-0{flex-shrink:0 !important}.flex-xl-self-auto{align-self:auto !important}.flex-xl-self-start{align-self:flex-start !important}.flex-xl-self-end{align-self:flex-end !important}.flex-xl-self-center{align-self:center !important}.flex-xl-self-baseline{align-self:baseline !important}.flex-xl-self-stretch{align-self:stretch !important}.flex-xl-item-equal{flex-grow:1;flex-basis:0}}.position-static{position:static !important}.position-relative{position:relative !important}.position-absolute{position:absolute !important}.position-fixed{position:fixed !important}.top-0{top:0 !important}.right-0{right:0 !important}.bottom-0{bottom:0 !important}.left-0{left:0 !important}.v-align-middle{vertical-align:middle !important}.v-align-top{vertical-align:top !important}.v-align-bottom{vertical-align:bottom !important}.v-align-text-top{vertical-align:text-top !important}.v-align-text-bottom{vertical-align:text-bottom !important}.v-align-baseline{vertical-align:baseline !important}.overflow-hidden{overflow:hidden !important}.overflow-scroll{overflow:scroll !important}.overflow-auto{overflow:auto !important}.clearfix::before{display:table;content:""}.clearfix::after{display:table;clear:both;content:""}.float-right{float:right !important}.float-left{float:left !important}.float-none{float:none !important}@media (min-width: 544px){.float-sm-left{float:left !important}.float-sm-right{float:right !important}.float-sm-none{float:none !important}}@media (min-width: 768px){.float-md-left{float:left !important}.float-md-right{float:right !important}.float-md-none{float:none !important}}@media (min-width: 1012px){.float-lg-left{float:left !important}.float-lg-right{float:right !important}.float-lg-none{float:none !important}}@media (min-width: 1280px){.float-xl-left{float:left !important}.float-xl-right{float:right !important}.float-xl-none{float:none !important}}.width-fit{max-width:100% !important}.width-full{width:100% !important}.height-fit{max-height:100% !important}.height-full{height:100% !important}.min-width-0{min-width:0 !important}.direction-rtl{direction:rtl !important}.direction-ltr{direction:ltr !important}@media (min-width: 544px){.direction-sm-rtl{direction:rtl !important}.direction-sm-ltr{direction:ltr !important}}@media (min-width: 768px){.direction-md-rtl{direction:rtl !important}.direction-md-ltr{direction:ltr !important}}@media (min-width: 1012px){.direction-lg-rtl{direction:rtl !important}.direction-lg-ltr{direction:ltr !important}}@media (min-width: 1280px){.direction-xl-rtl{direction:rtl !important}.direction-xl-ltr{direction:ltr !important}}.m-0{margin:0 !important}.mt-0{margin-top:0 !important}.mr-0{margin-right:0 !important}.mb-0{margin-bottom:0 !important}.ml-0{margin-left:0 !important}.mt-n0{margin-top:-0 !important}.mr-n0{margin-right:-0 !important}.mb-n0{margin-bottom:-0 !important}.ml-n0{margin-left:-0 !important}.mx-0{margin-right:0 !important;margin-left:0 !important}.my-0{margin-top:0 !important;margin-bottom:0 !important}.m-1{margin:4px !important}.mt-1{margin-top:4px !important}.mr-1{margin-right:4px !important}.mb-1{margin-bottom:4px !important}.ml-1{margin-left:4px !important}.mt-n1{margin-top:-4px !important}.mr-n1{margin-right:-4px !important}.mb-n1{margin-bottom:-4px !important}.ml-n1{margin-left:-4px !important}.mx-1{margin-right:4px !important;margin-left:4px !important}.my-1{margin-top:4px !important;margin-bottom:4px !important}.m-2{margin:8px !important}.mt-2{margin-top:8px !important}.mr-2{margin-right:8px !important}.mb-2{margin-bottom:8px !important}.ml-2{margin-left:8px !important}.mt-n2{margin-top:-8px !important}.mr-n2{margin-right:-8px !important}.mb-n2{margin-bottom:-8px !important}.ml-n2{margin-left:-8px !important}.mx-2{margin-right:8px !important;margin-left:8px !important}.my-2{margin-top:8px !important;margin-bottom:8px !important}.m-3{margin:16px !important}.mt-3{margin-top:16px !important}.mr-3{margin-right:16px !important}.mb-3{margin-bottom:16px !important}.ml-3{margin-left:16px !important}.mt-n3{margin-top:-16px !important}.mr-n3{margin-right:-16px !important}.mb-n3{margin-bottom:-16px !important}.ml-n3{margin-left:-16px !important}.mx-3{margin-right:16px !important;margin-left:16px !important}.my-3{margin-top:16px !important;margin-bottom:16px !important}.m-4{margin:24px !important}.mt-4{margin-top:24px !important}.mr-4{margin-right:24px !important}.mb-4{margin-bottom:24px !important}.ml-4{margin-left:24px !important}.mt-n4{margin-top:-24px !important}.mr-n4{margin-right:-24px !important}.mb-n4{margin-bottom:-24px !important}.ml-n4{margin-left:-24px !important}.mx-4{margin-right:24px !important;margin-left:24px !important}.my-4{margin-top:24px !important;margin-bottom:24px !important}.m-5{margin:32px !important}.mt-5{margin-top:32px !important}.mr-5{margin-right:32px !important}.mb-5{margin-bottom:32px !important}.ml-5{margin-left:32px !important}.mt-n5{margin-top:-32px !important}.mr-n5{margin-right:-32px !important}.mb-n5{margin-bottom:-32px !important}.ml-n5{margin-left:-32px !important}.mx-5{margin-right:32px !important;margin-left:32px !important}.my-5{margin-top:32px !important;margin-bottom:32px !important}.m-6{margin:40px !important}.mt-6{margin-top:40px !important}.mr-6{margin-right:40px !important}.mb-6{margin-bottom:40px !important}.ml-6{margin-left:40px !important}.mt-n6{margin-top:-40px !important}.mr-n6{margin-right:-40px !important}.mb-n6{margin-bottom:-40px !important}.ml-n6{margin-left:-40px !important}.mx-6{margin-right:40px !important;margin-left:40px !important}.my-6{margin-top:40px !important;margin-bottom:40px !important}.mx-auto{margin-right:auto !important;margin-left:auto !important}@media (min-width: 544px){.m-sm-0{margin:0 !important}.mt-sm-0{margin-top:0 !important}.mr-sm-0{margin-right:0 !important}.mb-sm-0{margin-bottom:0 !important}.ml-sm-0{margin-left:0 !important}.mt-sm-n0{margin-top:-0 !important}.mr-sm-n0{margin-right:-0 !important}.mb-sm-n0{margin-bottom:-0 !important}.ml-sm-n0{margin-left:-0 !important}.mx-sm-0{margin-right:0 !important;margin-left:0 !important}.my-sm-0{margin-top:0 !important;margin-bottom:0 !important}}@media (min-width: 544px){.m-sm-1{margin:4px !important}.mt-sm-1{margin-top:4px !important}.mr-sm-1{margin-right:4px !important}.mb-sm-1{margin-bottom:4px !important}.ml-sm-1{margin-left:4px !important}.mt-sm-n1{margin-top:-4px !important}.mr-sm-n1{margin-right:-4px !important}.mb-sm-n1{margin-bottom:-4px !important}.ml-sm-n1{margin-left:-4px !important}.mx-sm-1{margin-right:4px !important;margin-left:4px !important}.my-sm-1{margin-top:4px !important;margin-bottom:4px !important}}@media (min-width: 544px){.m-sm-2{margin:8px !important}.mt-sm-2{margin-top:8px !important}.mr-sm-2{margin-right:8px !important}.mb-sm-2{margin-bottom:8px !important}.ml-sm-2{margin-left:8px !important}.mt-sm-n2{margin-top:-8px !important}.mr-sm-n2{margin-right:-8px !important}.mb-sm-n2{margin-bottom:-8px !important}.ml-sm-n2{margin-left:-8px !important}.mx-sm-2{margin-right:8px !important;margin-left:8px !important}.my-sm-2{margin-top:8px !important;margin-bottom:8px !important}}@media (min-width: 544px){.m-sm-3{margin:16px !important}.mt-sm-3{margin-top:16px !important}.mr-sm-3{margin-right:16px !important}.mb-sm-3{margin-bottom:16px !important}.ml-sm-3{margin-left:16px !important}.mt-sm-n3{margin-top:-16px !important}.mr-sm-n3{margin-right:-16px !important}.mb-sm-n3{margin-bottom:-16px !important}.ml-sm-n3{margin-left:-16px !important}.mx-sm-3{margin-right:16px !important;margin-left:16px !important}.my-sm-3{margin-top:16px !important;margin-bottom:16px !important}}@media (min-width: 544px){.m-sm-4{margin:24px !important}.mt-sm-4{margin-top:24px !important}.mr-sm-4{margin-right:24px !important}.mb-sm-4{margin-bottom:24px !important}.ml-sm-4{margin-left:24px !important}.mt-sm-n4{margin-top:-24px !important}.mr-sm-n4{margin-right:-24px !important}.mb-sm-n4{margin-bottom:-24px !important}.ml-sm-n4{margin-left:-24px !important}.mx-sm-4{margin-right:24px !important;margin-left:24px !important}.my-sm-4{margin-top:24px !important;margin-bottom:24px !important}}@media (min-width: 544px){.m-sm-5{margin:32px !important}.mt-sm-5{margin-top:32px !important}.mr-sm-5{margin-right:32px !important}.mb-sm-5{margin-bottom:32px !important}.ml-sm-5{margin-left:32px !important}.mt-sm-n5{margin-top:-32px !important}.mr-sm-n5{margin-right:-32px !important}.mb-sm-n5{margin-bottom:-32px !important}.ml-sm-n5{margin-left:-32px !important}.mx-sm-5{margin-right:32px !important;margin-left:32px !important}.my-sm-5{margin-top:32px !important;margin-bottom:32px !important}}@media (min-width: 544px){.m-sm-6{margin:40px !important}.mt-sm-6{margin-top:40px !important}.mr-sm-6{margin-right:40px !important}.mb-sm-6{margin-bottom:40px !important}.ml-sm-6{margin-left:40px !important}.mt-sm-n6{margin-top:-40px !important}.mr-sm-n6{margin-right:-40px !important}.mb-sm-n6{margin-bottom:-40px !important}.ml-sm-n6{margin-left:-40px !important}.mx-sm-6{margin-right:40px !important;margin-left:40px !important}.my-sm-6{margin-top:40px !important;margin-bottom:40px !important}}@media (min-width: 768px){.m-md-0{margin:0 !important}.mt-md-0{margin-top:0 !important}.mr-md-0{margin-right:0 !important}.mb-md-0{margin-bottom:0 !important}.ml-md-0{margin-left:0 !important}.mt-md-n0{margin-top:-0 !important}.mr-md-n0{margin-right:-0 !important}.mb-md-n0{margin-bottom:-0 !important}.ml-md-n0{margin-left:-0 !important}.mx-md-0{margin-right:0 !important;margin-left:0 !important}.my-md-0{margin-top:0 !important;margin-bottom:0 !important}}@media (min-width: 768px){.m-md-1{margin:4px !important}.mt-md-1{margin-top:4px !important}.mr-md-1{margin-right:4px !important}.mb-md-1{margin-bottom:4px !important}.ml-md-1{margin-left:4px !important}.mt-md-n1{margin-top:-4px !important}.mr-md-n1{margin-right:-4px !important}.mb-md-n1{margin-bottom:-4px !important}.ml-md-n1{margin-left:-4px !important}.mx-md-1{margin-right:4px !important;margin-left:4px !important}.my-md-1{margin-top:4px !important;margin-bottom:4px !important}}@media (min-width: 768px){.m-md-2{margin:8px !important}.mt-md-2{margin-top:8px !important}.mr-md-2{margin-right:8px !important}.mb-md-2{margin-bottom:8px !important}.ml-md-2{margin-left:8px !important}.mt-md-n2{margin-top:-8px !important}.mr-md-n2{margin-right:-8px !important}.mb-md-n2{margin-bottom:-8px !important}.ml-md-n2{margin-left:-8px !important}.mx-md-2{margin-right:8px !important;margin-left:8px !important}.my-md-2{margin-top:8px !important;margin-bottom:8px !important}}@media (min-width: 768px){.m-md-3{margin:16px !important}.mt-md-3{margin-top:16px !important}.mr-md-3{margin-right:16px !important}.mb-md-3{margin-bottom:16px !important}.ml-md-3{margin-left:16px !important}.mt-md-n3{margin-top:-16px !important}.mr-md-n3{margin-right:-16px !important}.mb-md-n3{margin-bottom:-16px !important}.ml-md-n3{margin-left:-16px !important}.mx-md-3{margin-right:16px !important;margin-left:16px !important}.my-md-3{margin-top:16px !important;margin-bottom:16px !important}}@media (min-width: 768px){.m-md-4{margin:24px !important}.mt-md-4{margin-top:24px !important}.mr-md-4{margin-right:24px !important}.mb-md-4{margin-bottom:24px !important}.ml-md-4{margin-left:24px !important}.mt-md-n4{margin-top:-24px !important}.mr-md-n4{margin-right:-24px !important}.mb-md-n4{margin-bottom:-24px !important}.ml-md-n4{margin-left:-24px !important}.mx-md-4{margin-right:24px !important;margin-left:24px !important}.my-md-4{margin-top:24px !important;margin-bottom:24px !important}}@media (min-width: 768px){.m-md-5{margin:32px !important}.mt-md-5{margin-top:32px !important}.mr-md-5{margin-right:32px !important}.mb-md-5{margin-bottom:32px !important}.ml-md-5{margin-left:32px !important}.mt-md-n5{margin-top:-32px !important}.mr-md-n5{margin-right:-32px !important}.mb-md-n5{margin-bottom:-32px !important}.ml-md-n5{margin-left:-32px !important}.mx-md-5{margin-right:32px !important;margin-left:32px !important}.my-md-5{margin-top:32px !important;margin-bottom:32px !important}}@media (min-width: 768px){.m-md-6{margin:40px !important}.mt-md-6{margin-top:40px !important}.mr-md-6{margin-right:40px !important}.mb-md-6{margin-bottom:40px !important}.ml-md-6{margin-left:40px !important}.mt-md-n6{margin-top:-40px !important}.mr-md-n6{margin-right:-40px !important}.mb-md-n6{margin-bottom:-40px !important}.ml-md-n6{margin-left:-40px !important}.mx-md-6{margin-right:40px !important;margin-left:40px !important}.my-md-6{margin-top:40px !important;margin-bottom:40px !important}}@media (min-width: 1012px){.m-lg-0{margin:0 !important}.mt-lg-0{margin-top:0 !important}.mr-lg-0{margin-right:0 !important}.mb-lg-0{margin-bottom:0 !important}.ml-lg-0{margin-left:0 !important}.mt-lg-n0{margin-top:-0 !important}.mr-lg-n0{margin-right:-0 !important}.mb-lg-n0{margin-bottom:-0 !important}.ml-lg-n0{margin-left:-0 !important}.mx-lg-0{margin-right:0 !important;margin-left:0 !important}.my-lg-0{margin-top:0 !important;margin-bottom:0 !important}}@media (min-width: 1012px){.m-lg-1{margin:4px !important}.mt-lg-1{margin-top:4px !important}.mr-lg-1{margin-right:4px !important}.mb-lg-1{margin-bottom:4px !important}.ml-lg-1{margin-left:4px !important}.mt-lg-n1{margin-top:-4px !important}.mr-lg-n1{margin-right:-4px !important}.mb-lg-n1{margin-bottom:-4px !important}.ml-lg-n1{margin-left:-4px !important}.mx-lg-1{margin-right:4px !important;margin-left:4px !important}.my-lg-1{margin-top:4px !important;margin-bottom:4px !important}}@media (min-width: 1012px){.m-lg-2{margin:8px !important}.mt-lg-2{margin-top:8px !important}.mr-lg-2{margin-right:8px !important}.mb-lg-2{margin-bottom:8px !important}.ml-lg-2{margin-left:8px !important}.mt-lg-n2{margin-top:-8px !important}.mr-lg-n2{margin-right:-8px !important}.mb-lg-n2{margin-bottom:-8px !important}.ml-lg-n2{margin-left:-8px !important}.mx-lg-2{margin-right:8px !important;margin-left:8px !important}.my-lg-2{margin-top:8px !important;margin-bottom:8px !important}}@media (min-width: 1012px){.m-lg-3{margin:16px !important}.mt-lg-3{margin-top:16px !important}.mr-lg-3{margin-right:16px !important}.mb-lg-3{margin-bottom:16px !important}.ml-lg-3{margin-left:16px !important}.mt-lg-n3{margin-top:-16px !important}.mr-lg-n3{margin-right:-16px !important}.mb-lg-n3{margin-bottom:-16px !important}.ml-lg-n3{margin-left:-16px !important}.mx-lg-3{margin-right:16px !important;margin-left:16px !important}.my-lg-3{margin-top:16px !important;margin-bottom:16px !important}}@media (min-width: 1012px){.m-lg-4{margin:24px !important}.mt-lg-4{margin-top:24px !important}.mr-lg-4{margin-right:24px !important}.mb-lg-4{margin-bottom:24px !important}.ml-lg-4{margin-left:24px !important}.mt-lg-n4{margin-top:-24px !important}.mr-lg-n4{margin-right:-24px !important}.mb-lg-n4{margin-bottom:-24px !important}.ml-lg-n4{margin-left:-24px !important}.mx-lg-4{margin-right:24px !important;margin-left:24px !important}.my-lg-4{margin-top:24px !important;margin-bottom:24px !important}}@media (min-width: 1012px){.m-lg-5{margin:32px !important}.mt-lg-5{margin-top:32px !important}.mr-lg-5{margin-right:32px !important}.mb-lg-5{margin-bottom:32px !important}.ml-lg-5{margin-left:32px !important}.mt-lg-n5{margin-top:-32px !important}.mr-lg-n5{margin-right:-32px !important}.mb-lg-n5{margin-bottom:-32px !important}.ml-lg-n5{margin-left:-32px !important}.mx-lg-5{margin-right:32px !important;margin-left:32px !important}.my-lg-5{margin-top:32px !important;margin-bottom:32px !important}}@media (min-width: 1012px){.m-lg-6{margin:40px !important}.mt-lg-6{margin-top:40px !important}.mr-lg-6{margin-right:40px !important}.mb-lg-6{margin-bottom:40px !important}.ml-lg-6{margin-left:40px !important}.mt-lg-n6{margin-top:-40px !important}.mr-lg-n6{margin-right:-40px !important}.mb-lg-n6{margin-bottom:-40px !important}.ml-lg-n6{margin-left:-40px !important}.mx-lg-6{margin-right:40px !important;margin-left:40px !important}.my-lg-6{margin-top:40px !important;margin-bottom:40px !important}}@media (min-width: 1280px){.m-xl-0{margin:0 !important}.mt-xl-0{margin-top:0 !important}.mr-xl-0{margin-right:0 !important}.mb-xl-0{margin-bottom:0 !important}.ml-xl-0{margin-left:0 !important}.mt-xl-n0{margin-top:-0 !important}.mr-xl-n0{margin-right:-0 !important}.mb-xl-n0{margin-bottom:-0 !important}.ml-xl-n0{margin-left:-0 !important}.mx-xl-0{margin-right:0 !important;margin-left:0 !important}.my-xl-0{margin-top:0 !important;margin-bottom:0 !important}}@media (min-width: 1280px){.m-xl-1{margin:4px !important}.mt-xl-1{margin-top:4px !important}.mr-xl-1{margin-right:4px !important}.mb-xl-1{margin-bottom:4px !important}.ml-xl-1{margin-left:4px !important}.mt-xl-n1{margin-top:-4px !important}.mr-xl-n1{margin-right:-4px !important}.mb-xl-n1{margin-bottom:-4px !important}.ml-xl-n1{margin-left:-4px !important}.mx-xl-1{margin-right:4px !important;margin-left:4px !important}.my-xl-1{margin-top:4px !important;margin-bottom:4px !important}}@media (min-width: 1280px){.m-xl-2{margin:8px !important}.mt-xl-2{margin-top:8px !important}.mr-xl-2{margin-right:8px !important}.mb-xl-2{margin-bottom:8px !important}.ml-xl-2{margin-left:8px !important}.mt-xl-n2{margin-top:-8px !important}.mr-xl-n2{margin-right:-8px !important}.mb-xl-n2{margin-bottom:-8px !important}.ml-xl-n2{margin-left:-8px !important}.mx-xl-2{margin-right:8px !important;margin-left:8px !important}.my-xl-2{margin-top:8px !important;margin-bottom:8px !important}}@media (min-width: 1280px){.m-xl-3{margin:16px !important}.mt-xl-3{margin-top:16px !important}.mr-xl-3{margin-right:16px !important}.mb-xl-3{margin-bottom:16px !important}.ml-xl-3{margin-left:16px !important}.mt-xl-n3{margin-top:-16px !important}.mr-xl-n3{margin-right:-16px !important}.mb-xl-n3{margin-bottom:-16px !important}.ml-xl-n3{margin-left:-16px !important}.mx-xl-3{margin-right:16px !important;margin-left:16px !important}.my-xl-3{margin-top:16px !important;margin-bottom:16px !important}}@media (min-width: 1280px){.m-xl-4{margin:24px !important}.mt-xl-4{margin-top:24px !important}.mr-xl-4{margin-right:24px !important}.mb-xl-4{margin-bottom:24px !important}.ml-xl-4{margin-left:24px !important}.mt-xl-n4{margin-top:-24px !important}.mr-xl-n4{margin-right:-24px !important}.mb-xl-n4{margin-bottom:-24px !important}.ml-xl-n4{margin-left:-24px !important}.mx-xl-4{margin-right:24px !important;margin-left:24px !important}.my-xl-4{margin-top:24px !important;margin-bottom:24px !important}}@media (min-width: 1280px){.m-xl-5{margin:32px !important}.mt-xl-5{margin-top:32px !important}.mr-xl-5{margin-right:32px !important}.mb-xl-5{margin-bottom:32px !important}.ml-xl-5{margin-left:32px !important}.mt-xl-n5{margin-top:-32px !important}.mr-xl-n5{margin-right:-32px !important}.mb-xl-n5{margin-bottom:-32px !important}.ml-xl-n5{margin-left:-32px !important}.mx-xl-5{margin-right:32px !important;margin-left:32px !important}.my-xl-5{margin-top:32px !important;margin-bottom:32px !important}}@media (min-width: 1280px){.m-xl-6{margin:40px !important}.mt-xl-6{margin-top:40px !important}.mr-xl-6{margin-right:40px !important}.mb-xl-6{margin-bottom:40px !important}.ml-xl-6{margin-left:40px !important}.mt-xl-n6{margin-top:-40px !important}.mr-xl-n6{margin-right:-40px !important}.mb-xl-n6{margin-bottom:-40px !important}.ml-xl-n6{margin-left:-40px !important}.mx-xl-6{margin-right:40px !important;margin-left:40px !important}.my-xl-6{margin-top:40px !important;margin-bottom:40px !important}}.p-0{padding:0 !important}.pt-0{padding-top:0 !important}.pr-0{padding-right:0 !important}.pb-0{padding-bottom:0 !important}.pl-0{padding-left:0 !important}.px-0{padding-right:0 !important;padding-left:0 !important}.py-0{padding-top:0 !important;padding-bottom:0 !important}.p-1{padding:4px !important}.pt-1{padding-top:4px !important}.pr-1{padding-right:4px !important}.pb-1{padding-bottom:4px !important}.pl-1{padding-left:4px !important}.px-1{padding-right:4px !important;padding-left:4px !important}.py-1{padding-top:4px !important;padding-bottom:4px !important}.p-2{padding:8px !important}.pt-2{padding-top:8px !important}.pr-2{padding-right:8px !important}.pb-2{padding-bottom:8px !important}.pl-2{padding-left:8px !important}.px-2{padding-right:8px !important;padding-left:8px !important}.py-2{padding-top:8px !important;padding-bottom:8px !important}.p-3{padding:16px !important}.pt-3{padding-top:16px !important}.pr-3{padding-right:16px !important}.pb-3{padding-bottom:16px !important}.pl-3{padding-left:16px !important}.px-3{padding-right:16px !important;padding-left:16px !important}.py-3{padding-top:16px !important;padding-bottom:16px !important}.p-4{padding:24px !important}.pt-4{padding-top:24px !important}.pr-4{padding-right:24px !important}.pb-4{padding-bottom:24px !important}.pl-4{padding-left:24px !important}.px-4{padding-right:24px !important;padding-left:24px !important}.py-4{padding-top:24px !important;padding-bottom:24px !important}.p-5{padding:32px !important}.pt-5{padding-top:32px !important}.pr-5{padding-right:32px !important}.pb-5{padding-bottom:32px !important}.pl-5{padding-left:32px !important}.px-5{padding-right:32px !important;padding-left:32px !important}.py-5{padding-top:32px !important;padding-bottom:32px !important}.p-6{padding:40px !important}.pt-6{padding-top:40px !important}.pr-6{padding-right:40px !important}.pb-6{padding-bottom:40px !important}.pl-6{padding-left:40px !important}.px-6{padding-right:40px !important;padding-left:40px !important}.py-6{padding-top:40px !important;padding-bottom:40px !important}@media (min-width: 544px){.p-sm-0{padding:0 !important}.pt-sm-0{padding-top:0 !important}.pr-sm-0{padding-right:0 !important}.pb-sm-0{padding-bottom:0 !important}.pl-sm-0{padding-left:0 !important}.px-sm-0{padding-right:0 !important;padding-left:0 !important}.py-sm-0{padding-top:0 !important;padding-bottom:0 !important}}@media (min-width: 544px){.p-sm-1{padding:4px !important}.pt-sm-1{padding-top:4px !important}.pr-sm-1{padding-right:4px !important}.pb-sm-1{padding-bottom:4px !important}.pl-sm-1{padding-left:4px !important}.px-sm-1{padding-right:4px !important;padding-left:4px !important}.py-sm-1{padding-top:4px !important;padding-bottom:4px !important}}@media (min-width: 544px){.p-sm-2{padding:8px !important}.pt-sm-2{padding-top:8px !important}.pr-sm-2{padding-right:8px !important}.pb-sm-2{padding-bottom:8px !important}.pl-sm-2{padding-left:8px !important}.px-sm-2{padding-right:8px !important;padding-left:8px !important}.py-sm-2{padding-top:8px !important;padding-bottom:8px !important}}@media (min-width: 544px){.p-sm-3{padding:16px !important}.pt-sm-3{padding-top:16px !important}.pr-sm-3{padding-right:16px !important}.pb-sm-3{padding-bottom:16px !important}.pl-sm-3{padding-left:16px !important}.px-sm-3{padding-right:16px !important;padding-left:16px !important}.py-sm-3{padding-top:16px !important;padding-bottom:16px !important}}@media (min-width: 544px){.p-sm-4{padding:24px !important}.pt-sm-4{padding-top:24px !important}.pr-sm-4{padding-right:24px !important}.pb-sm-4{padding-bottom:24px !important}.pl-sm-4{padding-left:24px !important}.px-sm-4{padding-right:24px !important;padding-left:24px !important}.py-sm-4{padding-top:24px !important;padding-bottom:24px !important}}@media (min-width: 544px){.p-sm-5{padding:32px !important}.pt-sm-5{padding-top:32px !important}.pr-sm-5{padding-right:32px !important}.pb-sm-5{padding-bottom:32px !important}.pl-sm-5{padding-left:32px !important}.px-sm-5{padding-right:32px !important;padding-left:32px !important}.py-sm-5{padding-top:32px !important;padding-bottom:32px !important}}@media (min-width: 544px){.p-sm-6{padding:40px !important}.pt-sm-6{padding-top:40px !important}.pr-sm-6{padding-right:40px !important}.pb-sm-6{padding-bottom:40px !important}.pl-sm-6{padding-left:40px !important}.px-sm-6{padding-right:40px !important;padding-left:40px !important}.py-sm-6{padding-top:40px !important;padding-bottom:40px !important}}@media (min-width: 768px){.p-md-0{padding:0 !important}.pt-md-0{padding-top:0 !important}.pr-md-0{padding-right:0 !important}.pb-md-0{padding-bottom:0 !important}.pl-md-0{padding-left:0 !important}.px-md-0{padding-right:0 !important;padding-left:0 !important}.py-md-0{padding-top:0 !important;padding-bottom:0 !important}}@media (min-width: 768px){.p-md-1{padding:4px !important}.pt-md-1{padding-top:4px !important}.pr-md-1{padding-right:4px !important}.pb-md-1{padding-bottom:4px !important}.pl-md-1{padding-left:4px !important}.px-md-1{padding-right:4px !important;padding-left:4px !important}.py-md-1{padding-top:4px !important;padding-bottom:4px !important}}@media (min-width: 768px){.p-md-2{padding:8px !important}.pt-md-2{padding-top:8px !important}.pr-md-2{padding-right:8px !important}.pb-md-2{padding-bottom:8px !important}.pl-md-2{padding-left:8px !important}.px-md-2{padding-right:8px !important;padding-left:8px !important}.py-md-2{padding-top:8px !important;padding-bottom:8px !important}}@media (min-width: 768px){.p-md-3{padding:16px !important}.pt-md-3{padding-top:16px !important}.pr-md-3{padding-right:16px !important}.pb-md-3{padding-bottom:16px !important}.pl-md-3{padding-left:16px !important}.px-md-3{padding-right:16px !important;padding-left:16px !important}.py-md-3{padding-top:16px !important;padding-bottom:16px !important}}@media (min-width: 768px){.p-md-4{padding:24px !important}.pt-md-4{padding-top:24px !important}.pr-md-4{padding-right:24px !important}.pb-md-4{padding-bottom:24px !important}.pl-md-4{padding-left:24px !important}.px-md-4{padding-right:24px !important;padding-left:24px !important}.py-md-4{padding-top:24px !important;padding-bottom:24px !important}}@media (min-width: 768px){.p-md-5{padding:32px !important}.pt-md-5{padding-top:32px !important}.pr-md-5{padding-right:32px !important}.pb-md-5{padding-bottom:32px !important}.pl-md-5{padding-left:32px !important}.px-md-5{padding-right:32px !important;padding-left:32px !important}.py-md-5{padding-top:32px !important;padding-bottom:32px !important}}@media (min-width: 768px){.p-md-6{padding:40px !important}.pt-md-6{padding-top:40px !important}.pr-md-6{padding-right:40px !important}.pb-md-6{padding-bottom:40px !important}.pl-md-6{padding-left:40px !important}.px-md-6{padding-right:40px !important;padding-left:40px !important}.py-md-6{padding-top:40px !important;padding-bottom:40px !important}}@media (min-width: 1012px){.p-lg-0{padding:0 !important}.pt-lg-0{padding-top:0 !important}.pr-lg-0{padding-right:0 !important}.pb-lg-0{padding-bottom:0 !important}.pl-lg-0{padding-left:0 !important}.px-lg-0{padding-right:0 !important;padding-left:0 !important}.py-lg-0{padding-top:0 !important;padding-bottom:0 !important}}@media (min-width: 1012px){.p-lg-1{padding:4px !important}.pt-lg-1{padding-top:4px !important}.pr-lg-1{padding-right:4px !important}.pb-lg-1{padding-bottom:4px !important}.pl-lg-1{padding-left:4px !important}.px-lg-1{padding-right:4px !important;padding-left:4px !important}.py-lg-1{padding-top:4px !important;padding-bottom:4px !important}}@media (min-width: 1012px){.p-lg-2{padding:8px !important}.pt-lg-2{padding-top:8px !important}.pr-lg-2{padding-right:8px !important}.pb-lg-2{padding-bottom:8px !important}.pl-lg-2{padding-left:8px !important}.px-lg-2{padding-right:8px !important;padding-left:8px !important}.py-lg-2{padding-top:8px !important;padding-bottom:8px !important}}@media (min-width: 1012px){.p-lg-3{padding:16px !important}.pt-lg-3{padding-top:16px !important}.pr-lg-3{padding-right:16px !important}.pb-lg-3{padding-bottom:16px !important}.pl-lg-3{padding-left:16px !important}.px-lg-3{padding-right:16px !important;padding-left:16px !important}.py-lg-3{padding-top:16px !important;padding-bottom:16px !important}}@media (min-width: 1012px){.p-lg-4{padding:24px !important}.pt-lg-4{padding-top:24px !important}.pr-lg-4{padding-right:24px !important}.pb-lg-4{padding-bottom:24px !important}.pl-lg-4{padding-left:24px !important}.px-lg-4{padding-right:24px !important;padding-left:24px !important}.py-lg-4{padding-top:24px !important;padding-bottom:24px !important}}@media (min-width: 1012px){.p-lg-5{padding:32px !important}.pt-lg-5{padding-top:32px !important}.pr-lg-5{padding-right:32px !important}.pb-lg-5{padding-bottom:32px !important}.pl-lg-5{padding-left:32px !important}.px-lg-5{padding-right:32px !important;padding-left:32px !important}.py-lg-5{padding-top:32px !important;padding-bottom:32px !important}}@media (min-width: 1012px){.p-lg-6{padding:40px !important}.pt-lg-6{padding-top:40px !important}.pr-lg-6{padding-right:40px !important}.pb-lg-6{padding-bottom:40px !important}.pl-lg-6{padding-left:40px !important}.px-lg-6{padding-right:40px !important;padding-left:40px !important}.py-lg-6{padding-top:40px !important;padding-bottom:40px !important}}@media (min-width: 1280px){.p-xl-0{padding:0 !important}.pt-xl-0{padding-top:0 !important}.pr-xl-0{padding-right:0 !important}.pb-xl-0{padding-bottom:0 !important}.pl-xl-0{padding-left:0 !important}.px-xl-0{padding-right:0 !important;padding-left:0 !important}.py-xl-0{padding-top:0 !important;padding-bottom:0 !important}}@media (min-width: 1280px){.p-xl-1{padding:4px !important}.pt-xl-1{padding-top:4px !important}.pr-xl-1{padding-right:4px !important}.pb-xl-1{padding-bottom:4px !important}.pl-xl-1{padding-left:4px !important}.px-xl-1{padding-right:4px !important;padding-left:4px !important}.py-xl-1{padding-top:4px !important;padding-bottom:4px !important}}@media (min-width: 1280px){.p-xl-2{padding:8px !important}.pt-xl-2{padding-top:8px !important}.pr-xl-2{padding-right:8px !important}.pb-xl-2{padding-bottom:8px !important}.pl-xl-2{padding-left:8px !important}.px-xl-2{padding-right:8px !important;padding-left:8px !important}.py-xl-2{padding-top:8px !important;padding-bottom:8px !important}}@media (min-width: 1280px){.p-xl-3{padding:16px !important}.pt-xl-3{padding-top:16px !important}.pr-xl-3{padding-right:16px !important}.pb-xl-3{padding-bottom:16px !important}.pl-xl-3{padding-left:16px !important}.px-xl-3{padding-right:16px !important;padding-left:16px !important}.py-xl-3{padding-top:16px !important;padding-bottom:16px !important}}@media (min-width: 1280px){.p-xl-4{padding:24px !important}.pt-xl-4{padding-top:24px !important}.pr-xl-4{padding-right:24px !important}.pb-xl-4{padding-bottom:24px !important}.pl-xl-4{padding-left:24px !important}.px-xl-4{padding-right:24px !important;padding-left:24px !important}.py-xl-4{padding-top:24px !important;padding-bottom:24px !important}}@media (min-width: 1280px){.p-xl-5{padding:32px !important}.pt-xl-5{padding-top:32px !important}.pr-xl-5{padding-right:32px !important}.pb-xl-5{padding-bottom:32px !important}.pl-xl-5{padding-left:32px !important}.px-xl-5{padding-right:32px !important;padding-left:32px !important}.py-xl-5{padding-top:32px !important;padding-bottom:32px !important}}@media (min-width: 1280px){.p-xl-6{padding:40px !important}.pt-xl-6{padding-top:40px !important}.pr-xl-6{padding-right:40px !important}.pb-xl-6{padding-bottom:40px !important}.pl-xl-6{padding-left:40px !important}.px-xl-6{padding-right:40px !important;padding-left:40px !important}.py-xl-6{padding-top:40px !important;padding-bottom:40px !important}}.p-responsive{padding-right:16px !important;padding-left:16px !important}@media (min-width: 544px){.p-responsive{padding-right:40px !important;padding-left:40px !important}}@media (min-width: 1012px){.p-responsive{padding-right:16px !important;padding-left:16px !important}}.h1{font-size:26px !important}@media (min-width: 768px){.h1{font-size:32px !important}}.h2{font-size:22px !important}@media (min-width: 768px){.h2{font-size:24px !important}}.h3{font-size:18px !important}@media (min-width: 768px){.h3{font-size:20px !important}}.h4{font-size:16px !important}.h5{font-size:14px !important}.h6{font-size:12px !important}.h1,.h2,.h3,.h4,.h5,.h6{font-weight:600 !important}.f1{font-size:26px !important}@media (min-width: 768px){.f1{font-size:32px !important}}.f2{font-size:22px !important}@media (min-width: 768px){.f2{font-size:24px !important}}.f3{font-size:18px !important}@media (min-width: 768px){.f3{font-size:20px !important}}.f4{font-size:16px !important}@media (min-width: 768px){.f4{font-size:16px !important}}.f5{font-size:14px !important}.f6{font-size:12px !important}.f00-light{font-size:40px !important;font-weight:300 !important}@media (min-width: 768px){.f00-light{font-size:48px !important}}.f0-light{font-size:32px !important;font-weight:300 !important}@media (min-width: 768px){.f0-light{font-size:40px !important}}.f1-light{font-size:26px !important;font-weight:300 !important}@media (min-width: 768px){.f1-light{font-size:32px !important}}.f2-light{font-size:22px !important;font-weight:300 !important}@media (min-width: 768px){.f2-light{font-size:24px !important}}.f3-light{font-size:18px !important;font-weight:300 !important}@media (min-width: 768px){.f3-light{font-size:20px !important}}.text-small{font-size:12px !important}.lead{margin-bottom:30px;font-size:20px;font-weight:300;color:#586069}.lh-condensed-ultra{line-height:1 !important}.lh-condensed{line-height:1.25 !important}.lh-default{line-height:1.5 !important}.text-right{text-align:right !important}.text-left{text-align:left !important}.text-center{text-align:center !important}@media (min-width: 544px){.text-sm-right{text-align:right !important}.text-sm-left{text-align:left !important}.text-sm-center{text-align:center !important}}@media (min-width: 768px){.text-md-right{text-align:right !important}.text-md-left{text-align:left !important}.text-md-center{text-align:center !important}}@media (min-width: 1012px){.text-lg-right{text-align:right !important}.text-lg-left{text-align:left !important}.text-lg-center{text-align:center !important}}@media (min-width: 1280px){.text-xl-right{text-align:right !important}.text-xl-left{text-align:left !important}.text-xl-center{text-align:center !important}}.text-normal{font-weight:400 !important}.text-bold{font-weight:600 !important}.text-italic{font-style:italic !important}.text-uppercase{text-transform:uppercase !important}.no-underline{text-decoration:none !important}.no-wrap{white-space:nowrap !important}.ws-normal{white-space:normal !important}.wb-break-all{word-break:break-all !important}.text-emphasized{font-weight:600;color:#24292e}.list-style-none{list-style:none !important}.text-shadow-dark{text-shadow:0 1px 1px rgba(27,31,35,0.25),0 1px 25px rgba(27,31,35,0.75)}.text-shadow-light{text-shadow:0 1px 0 rgba(255,255,255,0.5)}.text-mono{font-family:"SFMono-Regular",Consolas,"Liberation Mono",Menlo,Courier,monospace}.v-hidden{visibility:hidden !important}.v-visible{visibility:visible !important}.d-table{display:table !important}.d-table-cell{display:table-cell !important}.table-fixed{table-layout:fixed !important}.d-block{display:block !important}.d-inline{display:inline !important}.d-inline-block{display:inline-block !important}.d-flex{display:flex !important}.d-inline-flex{display:inline-flex !important}.d-none{display:none !important}@media (min-width: 544px){.d-sm-table{display:table !important}.d-sm-table-cell{display:table-cell !important}.d-sm-block{display:block !important}.d-sm-inline{display:inline !important}.d-sm-inline-block{display:inline-block !important}.d-sm-flex{display:flex !important}.d-sm-inline-flex{display:inline-flex !important}.d-sm-none{display:none !important}}@media (min-width: 768px){.d-md-table{display:table !important}.d-md-table-cell{display:table-cell !important}.d-md-block{display:block !important}.d-md-inline{display:inline !important}.d-md-inline-block{display:inline-block !important}.d-md-flex{display:flex !important}.d-md-inline-flex{display:inline-flex !important}.d-md-none{display:none !important}}@media (min-width: 1012px){.d-lg-table{display:table !important}.d-lg-table-cell{display:table-cell !important}.d-lg-block{display:block !important}.d-lg-inline{display:inline !important}.d-lg-inline-block{display:inline-block !important}.d-lg-flex{display:flex !important}.d-lg-inline-flex{display:inline-flex !important}.d-lg-none{display:none !important}}@media (min-width: 1280px){.d-xl-table{display:table !important}.d-xl-table-cell{display:table-cell !important}.d-xl-block{display:block !important}.d-xl-inline{display:inline !important}.d-xl-inline-block{display:inline-block !important}.d-xl-flex{display:flex !important}.d-xl-inline-flex{display:inline-flex !important}.d-xl-none{display:none !important}}@media (max-width: 544px){.hide-sm{display:none !important}}@media (min-width: 544px) and (max-width: 768px){.hide-md{display:none !important}}@media (min-width: 768px) and (max-width: 1012px){.hide-lg{display:none !important}}@media (min-width: 1012px){.hide-xl{display:none !important}}.sr-only{position:absolute;width:1px;height:1px;padding:0;overflow:hidden;clip:rect(0, 0, 0, 0);word-wrap:normal;border:0}.show-on-focus{position:absolute;width:1px;height:1px;margin:0;overflow:hidden;clip:rect(1px, 1px, 1px, 1px)}.show-on-focus:focus{z-index:20;width:auto;height:auto;clip:auto}
/*# sourceMappingURL=frameworks-d1a05e466ba23154906d6cc001cd4853.css.map */
`)
	compressedBigFile = `
H4sIAAAAAAAC/9y9e5PjNpIv+v98Cm45OrbLFtUU9SiVFOPYdl/bMzv27I49njO+Xt8IigQluvlakqpS
WUf3s9/Ai0TiSanbvhtnvWOXiF8CiUw8MoEE8Ic3n/7LH7xPva+z7k/HnfdNtj903lMwXUxn+PO7qn5p
yLfX8b0XBrPQ8/F/HhjBxPtzGU8x8pssRmWLEu9YJqjxvv3z373Xh66r282bN/usOxx307gq3tRNVqCG
ffHbl7KLTn53QAXy96hETdRVzZtdXu3eFFHboebNN39+9+Vfv//y/g/ep2+mde7H57jKq2bzySp6mD8k
F/JtNsH/aT387ycOCIJlHC8JAJF0VPa06SKMZySpLTKBuJ1xSLgIH0NEqcuu/xquV/OIfH3PvyUP82jx
SDMjWdVJK2RZt4hlTT827D/kYxwj8WfbSD+jvjLzMF2FpJAnCimeeRqar8KAMrA78o+7+SwJ1+RjlvGP
aZTu0ni7i+L3+6Y6lomvouPQiRZqHIebzQ6lVYPOcVV2qOw2d//Pt3cXWMdzWpWd/4xwS9rsqjzZqvIs
cl7uw3wZByn9eJjQ/3hUEfRXa8xP0HmRUVTbveRok3VRnsVbVbnFzsycgEqgXDVCSVOE0oAVPYMNRgcP
0jRdUHgMNanPfJcwOWWDglbpOtVlLoohaYwVFLrBrm9py/UqWLHmvOcfH5ePSUQzjKsmP3fo1PkJiqsm
6rKq3JBOn2cl2krt9V2VoG+zpqkYF2lUZPnLpqjKqq2jGG0PlK15ENQnRrzLo/j9NskaFJPM864RM/Jx
Oe25jpIkK/ebRX3yAjHdqxvUpwbeoj4B6jZuqjzfRY2fZnmOmomYuD92HeIpZ0Wwz4esQxeVoD3vqiZB
jU/Gys2sPnltlWeJ90mSJDp1PuB/tiQ7n8hhU1bPTVQr9SyPxQ41QnXmuLresj5ti6z0n7OkO2xCLDui
kijP9uWGcME18fj46CqIVqKImveoOQs6MIL89rjrcnQeigDQ+Ni0VcNlkqNUFAnVLpBXWZVoS6sCNZlk
T0B1KK7KJGperAW0Wf6Emss0Lvw06hgznoY/WmJ07CrGzibw/iUr6qrporIT1Lb55AE9yBlKvNGv7flX
PysTdNrMJLyPJae2qGa/i16HwSRcLidhMAmmy/ttVGYF7Va7PCvfe7NpsGq9tkN1+3p272VlmpWkHcaF
T7FIKEhXrbFZapoqrvm/vUcvaRMVqPUI+Tl4db4sg1dqdbomKts6alDZXWYBhhEuu2h3TrK2zqOXTVbi
du3v8ip+v5XHkaw8oCbrQGtqjjnuYXXVZgQT7doqP3ZoS/QebGkbCrZdVW/8Je4Ku6rrqmLjk35RPaEm
zavnzSFLElSqWRua0SdxHJM8A55fsFV4ILVr/QSl0THvPPzrgKJE6EZHHea/j1XX957gMSCQEu2jLnvq
vyeLBflOCx2+h4/hZShoQnLvmqrcK+M8gaFCnQdJAtGkcSC/sHyz96g7NNVxf1CwRI8sUVPJ9+jluWr6
ifMhWGtAUVcVfcVmjxoEGwEZZrZaaDAJSgerL9UAnqImi3Y58vu5M1hGNtx8oiZ2L/WgtPVSQx5XRSFY
i9Ey0ICwUMt+Zo1mMyNm4DbV5lSgbpi2l0ttQ4vyLM0G+elhu2OWd1lvHs8DnWx2TRS/R90w6D9oQF3U
12z2EGgV3jXZ7ig0/yDW9aIGTC5KOmm81hwQMTp6a4vykpVPUZ4l8SGCSeJAXhW4x5V7Pi6w3h/ykeEC
R36vraNSzKCIuviQlXtJXsEucFGWVWkgjkJgS/VlYHEL8xSdTZbLyWwZTPBsMr8HZFGMBxI6/vZUYgaf
oHUapimw2/pBr0E5GaDkIVWcKVULidpb556G/hYn2iJq9lnJ5ezP8ajNvtGxnX5iNhDHkW/MesRTzbY6
drhm1JpQmIY8Zb8iXcWAVYILYHOBOK2J+Tz1xiSwIg/6z1dYnupkx+2K1ZZPpLieBm7O4pwY9PryT1xj
/ZcXpo+Lnv9zP/WxyVYglLM66bKS69xzxjO+mIXAirQgLVYBrTm2kcU2wqU412R2hlZyU0Q5aF9aA+YJ
NV0WRzmzvLuq1jVnXSWxFV5bVb0QOxbWNrBOqWknfdYVJPR0tSzJwOmL1mWE8k6TA7U6N2zotefA6uxt
Nn5R/eq3KKcunt2QtGb0MfK4lRnqiDIBYOsItDeoduKV8vElSrJjS+ROfjPXR1S3UORWdJyZgUy/4WGs
/0Kb3SYAzl7doC02wkhleaMmzZdxyalpRfkvrsRwaxn8n7I22+Vo6z+j3fus87uo9g/Z/pDjjFXJ9TjC
OjG0ys7Ps33UHRvUbsjy0ak7RvnWiQBawFUj0h0qumtQ9N7Hv2VpkHQqEgJiMlHUau00GsdD04cCJdPn
LNmjTjPvCPLmjv5Uaj9+0+WklsO6SNPlkuWSoLM4C2pG4okyDWpmIc2ndmJYmdhVJ5wP5pktAPq7CnJe
oKg9NkgjSNrwyejK2mOgGBekmWV51r3oPDjm86pZ11VWkkHrCZVdq4qDMUUXizh120VdFl8Mrr3Cydas
ybmcSdJEe01GrBMB1tIqPrYo8dxs6KjpOIagWZc84n+0pbhpH5JFKtnITdW2hyhr+ODXf1DavDy6Ku3o
c2wFj0NpoS5eKRPO8scVfWWp/39UHTtCKGqIe3DWrSRrFox7vwH/D/sNi3u6fFU1MfLpLMXXIJl1TMan
fytQkkVe3WRld7Ys3Bl7UL88RMZl/xDF7zdR2hF/le4m/Ou/XmQ3ibdSPBmcJRMJKCHJorzauwdwIQ95
Bpwt+yF5OkOFN12jQhmiwOSp4cDvqlryJoV1JoSYGaajpHBOjGESpdY6psReVtbH7ixYitBHMpgbfE0Z
FZJVoF3A15W7O3ZdVZ4HG+UheCUNvs0eGf0vef1cmsLVnCbKF3FUO/MthqVsiROoH9ZRCQjYd/yZLaYu
Hl5dQbiPaka30pHN3eXNZ9cQDuXNp0sdIclZ675oXB7VozHkSAeComp1XgHrWvVpKyyIazkfbB8th6IH
JtgafW86qasR+uVcshNj2aXRtEZhdE3X+B9NFahZhVklQ5hGFNCFXuq6gpCLplcQOn8ZvOK+FrNttsOa
HrDpZ9YCpPH17r/CWbL7ryCIAvqvcJZEd9YcrEJAZbTLUaIvxZR3XNUvdNBoid4mdgjRn8v1lxcanCLC
JZj9W0nmi0VsWEwYMvMb9ISaFl2T6QjheAbWSe1D7RCnCM+UBxWaPpPGz8oWNR1KNOrJ+0TR5siKaI82
xyZ/nURdtCE/39TlfruLWrRaTLJ/fPEf3z0Hf/l6X719+/btX7//4fDlD3v857f4X+/evf0R/ydJvv4x
wh++/iH/8m//+NufZ+G3z8d3/3yZf/buf738af31F/sfv/73/Y/v37794v/evf3b45ff/PPYPGOKL/79
ux+WXzbv/32/3//xj/ei6dNrhbYQD8tPTG9QjaJuQ//j6yWSoByZBMLSfid5LP7ysgvfrF5+DKE8fvhu
/5+/rv7SPjz/7XeQR3w4lu+BWZqmaYoCM9hvu6jpjAbO2kKJysRmVRlIq7JEcXdOszzn3G3brqneow0m
Yn+zNZmZtifkmnoilJqR1kqu18hCaa2kkRRUEqG0r+F6ja6o4TgVJ2niqvpt6l+ghV0yN7YNfb5xledR
3aLEp+s0SlzT6+l0en83gvKsH97X691WajJaizcVVvceg1dbEPyxhWuIC8N4zxnCk55nWMsFGwnGOMDw
f2AcINnsY7RkuVE096XRR47mUmg9bTyPlIloOfp9qMqYvEBYDdseG08nRdokAf7HlYGwPAfit4SWNFuR
OCL8H1du1mAbi1iV4B3qtU9GMK+uibhpDIspIwlvpv4fwiposPE6XqfRR9fJLXW8uXr/Byvhuh15GoXr
ykeOG3AGaNJ+C7YO3VVvQRDn3fdffVuVlf8d2h/zqLmbeO+qsq3yqJ14d99kO0TL9jDobuJ9i8q8wphj
k6Fm4g0xoMN0NwvxDCeNvcBxmy61jKrBNzw+3AQuW1xrOVhcj0Zll3UvIL6KrZYIoVXsiyx5stQmBdzq
S+EBUx9cDI/P1hfTdlUT7dF5JJgEK30wSzwe2FDKscat8CMUY1Ujj+768HJYyH2CuijLW2WhOc1OKOE7
srPg6eCRcEy6MMGXgbZk0TWtmoIuv+ZRh/752l8Gr+77pYXHx8dtEZ14618HTwe2PrtYrPHUXcVdFlfl
eVRwAq5QV9Vb4hbEx6ZBZfcOV+iyq5KXaUa374rq2CLvp6bK0R/v6KrB3c8bsl00UXEUYExuj0URNS80
3bg5yZdo4yiPX88CLDDfm9Wn+8s0zXLkoyTrqsbHFYgaFJ2FfUNu0izrk7eoT0SPZBjxlOGpH53UwYmO
TXxoGkamBpFhiQtyCKiVd/JhzbDLEGUlavy6QU8ZevamXbQroye/i3btmTUMn9pg5D/+jDhidkLxRx8S
Tuo6WyqeQeDK7fMpn/43ada0nR8fsjwRXDFi6/Hs5mPY6/cPxQ62CnSsTDtU1LjJM+V62C0oss6PGxR1
aDKGgrSNiEy9bd8DSNMfVWB7qJ7JpvnkCvZY+iiSvIrIjhkD+EW7H0VXVn58iMo9aq8mJVGPIhX08jRZ
NAibBuO1IOGv04FELLE/ojirzvRVsWhMIhipL4lqvLYkwut11dd+hJ4E7HU6EghH6EdAg+pNRrFv0Y0A
HqkXgWK8TgSiq/WxKavu9ZDDvVsrKoVWNyb9S4LQSsatYCWXEWpWaJzKVijG60QhVTQzigq2sXGdy0Vs
7zXXSBQSjOs7Nr27ia7sFGbpO0WplmQq3S1YXV4jxKsjcwpZRzRS1DrSUc1WR3hlywXl2PRlFLKUwwj5
ShRO0Ur4kVKVqMa3YInwOoF69VSUHre0yYkDYa1ipV17Z+e/+nOtFpApCESgBYsfNDONb8uiKVgYO3WO
FsxgT/Ms7oTFpc8sodCmGE9NFFm6SxIhf5YRW+y+lmBDwwUmxmS2FWPY8vdn5Oww26a5s5xY1+dPN6Ec
TPioTBjk3G+dDeEpQNra3GkdjNkXWZLkyAkjXDBxcDdY4MPNBq3BEDbmri0LThtTO4J9sDAhss94sMHp
4q5dL1xwJtWsLflDoRPuHzE8rZrCx02o9qZJE+335FDgrjrxvkmPltZRnHUvWDhGgs2hekJ9nGxRPSEI
xiMms//OfX7TlSVHnyz4sFMmtP0vZJ7J32mG8sSn5StdgkYAyruPJBB0tniczJYPk9lqOQnuxYxhZdj2
pgqQZTbUK9SB9SJ40EHVein7p8NVBVGMDlXOwsQsyH62Y4N+ey6iE9vke1iR5qNAvOkX1cmbolMdlQnl
EcwqfTy7mTbJ2iJrh5mMRU3CuUlPzMsdpnwdH2oAHVlnpCMo/ezN29FlfAw+HXWmq5eXKZ5wMZdR7le1
zjUiTSaPdiiXU3jOLduSkUwfQiinUUqiFJZEo+M9Y25mCm0BrPBji/iem8+MjrNftL7wfYNyVIinZcQ0
RrMlu066BM23y/RX1FRxntW7KmoSenC23wceTg316/SrVbKS49OkIOt+GVSXuaesSdPak2AnP26q2pgw
LVH7zEcX/LdP118B5LlFPeS5RToIarHJ11Q17vgTiXhI4dnwwO0B+BM2T1HyM8OSBCPXPfinJOoin1u2
bV51PwuFa/Mi7UjBKEcD+1DQwfoMVLLejj2L7YAoTG5l7KPUjOhXTcODR2p740uOvO1DfdlOh3q8B/+t
Mo51oYub7QfsJGoPKPFYJIYuCFicX5X82eK8+/ijdC6AL+qvggDOkORwNT1eLRZ2iMok1xSjAZktWjh6
C7fc6PaLyHbRxKObRv2RCe9ft/r4FXH2e1jhfwBrJbWABgFYu6I3LZ/PwpbWANd2SzxSDseJ5RJMJM/K
CWQNTyx26T9JTJEfVw3CPw9dV2/esEijKYs9yqo/eJ/i1O9QjiIYp5TT0KWpGu205nFRLMrpzZtP/8Wj
24TZr2gat633tJjOpjPvf5OcWBCU9789IeSpRHGVR+0bSPfpm0NX5GBfvY3K1m9Rk6Wk35JtO1xPP0p+
ObZsKu9PXepSyX4e32YKLlHTZXGOJlGbJWjCNi0nabaPIzK74j+PDZqkVYUNbXaxSBFl5aRA5XFSRk8T
/ZTGtvf6r3nWdn7WoeISHZOsmsRR+RS1k7qp9g1q28lTlqBKu1lJCcj66E+4ITdV3v58b7CpeH5naYdz
F7UkluLCzZHJT3Ro+hmOuJH9jG+0oWEZk4gZvGwg6AO/ot2u+anLuhz9LEUZEj7NwRdAazqEl1Rdh5Kt
C3DZTTSXvvCZTJu4I7bwJUlLzYUwh5lwXCdERW8YTFcPqPCCi/76ok/SdDAdguDSFlGeCzmtg1eX9rib
tMdaPA60fAXWNQLN4QuTZtvjjg8JfjANl6i44MzJ5T/BFP/Miv7qDr6fj+me9qRxbZqq6u7P8q1AZEvp
/S6Z1A2atFFR668rM0StoOJCuxDvczNUeIugPl0OphOymgOv/DQnNYon5PTWhM7Ak37rm+xvy2e+L1Xd
UdtfuQJIzOxsKooWQoOGhlmGyI0h8CDl/dS91EJMwIT9blCLuuFne9wVWXf385k3dmzMRE1UxmhDSVmm
LHyKGM5+VpaomUglmBGsTDOAc6Ei1MYxxEVyzgaaJiv3Clv6ZM6TPrVnCCb3MRHEziF92/uClPN3dOou
xNdtUadxX+MA/9O3AS8c7kjZBNP5Enfb6Sqk/33APSNHexoXrbWgwFk/Plx22InfDv4wiLkINOfHLn1L
7RtbdOyqC5NCfEDx+111ElpPlGTV3c8GrgbNMDhdkiR6Zc2LKNVv66xk7uTECK2OHYTy5RyRRXp6Vt9+
ceWISrht6Fdp2qJu44f1Sc5gKJd+8WOcRy5xaUQPo7+OE9I5RSHUx84X1jzOUJ/DuspyIZCR3cdjjX0W
LhFjr92KQ8/lU73CLtpRayKeCHXeZKG5n4IaNCI55i9HfvvSdqiYfIE9z2+j+Hvy86uq7CZ336N9hbwf
/nw3+RPKnxCeTSZvmyzKJ4OBNbl7i7PxSBSU92VR/ZLdDZTqh+9fil2V34lj/0JapJ9NlzDGUnudZ3oR
rnYlLrcu3OzCDRDzNW3aqX4VBJdDM5k2x7zfTehHitmyPnmB7RIn8TSwHPGknqZIUhSi5eXQDGvXuNx+
URUOJcMyPaFgS7uUgK7gSvgcRc1mV3UHkZSk9QM5bttkOGIf+PGHDf/j0iWT7nCWh3n5nIYcQMUsZR7E
Jq97smRqs1Y1Kn++//xT8osRQPNVjLU9zCaHcHKYTw6LyWE5Oaz4ei67tghcIRRA62we8m0oUduhaL8t
dIi5iAh0iMUZ7m6piOVZavkKYnWWonllRG2rKtlHkY3IR+zQYB+BXpvYGz3HfFLl/bUEzD20yrHKvSqf
HPG/z8RVITYAudhvk1fPWPdVEZWXY+4dByj+o+JfKvbFQB/l9SG6JAlYnw8uXTcht7NYw6dviE+Esr7U
vQWqF8D2Ny6/X/rTRZ5SJsiqqX6DUHcYapYsk0jqmCQU8Yvq5PtxVSbktNEZjsDhUgaQ5Wi+N8vHgXV9
YidgdGAy5YyDUsd5bL5d6TM5TYW/FWo+XvtrEh+6UuYZUy2b6tnEChkqq2OrFUe4MBcBCYnrKw0nWiCQ
YbgwoCTxGXEuyZEKcKnhH55vzEsU0gCSJALUMCNqoCG7M3BMwZfu4+5nIbL1DJowdTpI64a3f82GAF5N
6K2YRM+gyX1BVol2dFZ1QmpoPkQ7Qwu0Hsg2edR2fpXSq1Cli+5mck6Q01BJFusY8nqIehHET8YzUoQ8
7gobEsrxYYH9pnpmYc6cfUGk6mVlBlWEFlWIVYCC+ihS4f/1/WPZoCiZ9An0N7XHD1FSPZO7SgMv8Jht
6WVli7ohpzJ6yvaRuGvV55000Z57AtotIe4e7Jtox7+Rv7mHMePtTFfSNGtJEfus3JtKVcvZZeVeLAv/
tpTRYiNrlyM/PlQtKnWTDTtT5c5if6ja7mzq6qNzkOt6yBJ0FsIH+lS6TrFvohclSzcXnHyXH9Eo8lm6
TlOBnDgbpPSNIWxALZLS4CLNNLQcdlvU6+Hieo9sr9+f+9zEa32Z96T1iwABj0pwu1MXfZOTFCFNR9pR
yD3gsEtRiKMiHOGZk0lKug2XUWAhnuHsEa/RjKuIpIOZ2yTq7ahMyEQwCokH5VFAJjoLNonKvYLg4VkC
wrOP2i5CdbYyE1JhcvFfL1LSnl5QnmMpGSPfeqS1GAGHu6GjA7oMInLKGxoxK4MxubxMd3gaiZtjsSNb
OPozZSDuiC4zap6XEGZp1mGlzLmzL10qh3MDDh35wg/dkd41LAO8uVOzFU8yQRdMhvaHnzfSXUWM367U
7L9rRcKlTYQdwhhRjRUGBE8ccY38JAeKxqnJYRnGrXrDzr4uCED5YL7xRXtZTG8Wi6m07rPglYf/ZQgz
Cx8m89lkvpwE0/BeXnshOztbw7onrZ/8UV4cxSr0wCNA4lFPpo9lEAhrow9LSmX1Y8mNaAT1rjqWnXC3
Pnk5h848zBCTJSpetzhUf3ZPMjSs9Tm0Qknh+cqAfkyyllzGNcG/pvzXWbqxWafVwGNZsI1QkoHmdlB2
o9HQZVwrnyhN5+lqq+SA2Y6I6ZFkqOxe+7N1kKD9xGO2mhe8mniM2nsMXt0L1abcMSNnMIIDYgIH0xAV
tMHNJ7MgnISzBbsYvxc5zcBkvaAV2qWzq1gOCJuUZUJNWNb3HzLCgclFbh7z5b1ZGRO68vg5kb6G+UcU
o3Sr1Ziz1K0gTeJCEHnO6I7SnItVbMicU1PLG0qaryaL2WSxIpeNXtFQ3IyHgO++afp1k9FVXGHBSSk2
XEcPi+U1yp4vkmC5psqm1EP75GUK7ZR/GtFel+EkDNaT9ZpdyCpmOLTbPkNT+w1Xj+kivKZKYRrvlnNW
JUL9Qe13KfEuNOOeeV1z7jWmqdPDY7qY39I6PqBV9xUArbuvgq6V8zt1yb2604elrqU/LpJ5FP4GLV2Z
oMLHXbh4NOxHUcpjUw+3Dem7yGoxj3bBNe3pAS2X8QNtT5Ra7CKkSNBDyJcRHWQWribr5WT2+Cj2EJqf
2EFofqb+sUznD9H6qvqsFyhe0vpQ6o/TPyjnoHtQ1vW9g+pKo6DZPI4Wv3vnoNxLfYPyf2vX2M0e4+Th
o3YNwpDSM1brebxb23oG8dqs/SKYr2ZxfE07ClYPj9y0odRCvyArKkKvIIs67j6xmszC2SRchkKXGBZn
hKxM3SGYL+P4qukimD8s0gWvBqb+KN2BcC12BsK2tisYXOpgsYzj2e/dEQjfsBsQzm/tBOvZLkDLj9kJ
6HqN1AXYupmlC7ClG74CsgsX89nWtLj6sUx8VurG3fLDYD7BVqU4GTBisECo771qdVxcJ2ixWAaUa0rN
mv2Yxi0ypqhCkjjoBPSTvhtABenruVuGQRj/3n2CVQT2ClYVbb8Auvxd3AN+SGBEd9iaHWCezaBTp/cf
PNwDUnGk5p/EFsC+6ZuAVAvjTCXVzSQ8BtXw11dQx6g+sed4SNawfsPAxAvnY4Rage3YdYGFpAvYYnlF
tE1WcJ3vt7YAB9Mq0NLQMp+z7uDHWCjnNK+ibkNvZjZsQwa2TVi6quO3xXCovT55M+lgeyitDtPtfUyY
R+ThBv4ixgPu+jO6VqfG8cElPRKEQGPNaPg7PRsoBHyxU3I6zJQuuGpWZPsdGedq9VKqVaAvyosmU5Tn
Wd1mrcqj9m0CEHnrLesTeZ/ZHgcFWFn1b19/slgskqV++8qwKAyvYkYhthX015TNTCrwIj7oKDUfuSjJ
Z9FVvEtmxkL4MKaWQlMcazOzNdkhaas4i3K1R7iatHMdHi5euqRtjVwyr3D3rxoaAzpsHZjEe4gC4IoD
35g09VunKvXZdviTm4xV/F46WSxEhAuPj8dooCLbudY9k0Azbrg3cnU7Jr/z3ogpZGQTfPiuhbirbXmg
uUfCSQp8AgomjXK9njyuJrOAmXRwM+Ay5VdaksMMn/PIV2U7TwtTQn6tWC4mnszuDpGOWHfl1+BQu+2C
S9pFByJ3CLIAHR983BN91v81Eb7tOng7w5IE8bBkurOqOdirTOzgqnH1bkeQ4cgLHC2RRnOFSbp7DrIU
WZrZg9KcYxgoa7Bl5XrR9Q/4kc8i8Kt4H4TMqam4z+D3iRuWVk3hWakoz59dhxmRLa31Z1eCRmRMBPfZ
dRhNtuIF8cMN+YBCmKylFLEBG/L9kPZMihjas6GE36R5WyQ1tofjLATfyaQoKkcWz2fDcB/NCmJO3hhB
mVj9zFrC9b1rqOBn12FGZOvqXXrQiIwdvUuLGd27/o5nqf/Y/YLiDs5cIAlG87DQSJTn3IB7NToARc3W
7zfpaGaPj6+GM4uaa0b6m0PI5S3qPcHkgh92/ntCf7HDqcLTyvOF4HCt6pO3Vi5m+6jGvRJ7UVbsL+0a
N706YF2fPGoLX3GuQXqgUV1uw3mE9clTF5GW91B6UzZ3it824jcq16n6SbuOQp2vD+Jv4orIsASImtsF
DLa6XKb0KCTBRvrYXRb9C4EbY5Rsml422D8wnbHkTw1sRqLMaYylthBbe7geWjsJOJ0Lv1nw/XzMAg7N
PB8uNFxoHOXVgLT7fBzVHwb62IebeAGHLEG9S3XsKnJ7fe89kEtv4q4lST5O41G9xkcpt4ZTebo3o3XP
T9O2yA4yw/NnZKSB5ywNAynIxCOjoYeKaf+AuvEdSHHsC4VL72GMm3T/y054/FKwUiATRNz0VDL/9vPE
jCEHt38WF2BYxZfs9IFPpDGq/tOy6qQbloY5w7KMswj66ydo9J2UMXcuJa9Si/G7JtvvUbMhCSjx/l/v
U08PnYzOwcEOX/KkU+YZtJz+8wg3tkde4cUyGk+4Zk+jzMCbB1SfWgov6WjjtUUL8xYhq8qQX1bsTQu9
5EBMT4Y9bZVjPF56IS3SpyMfLIgbFENIP3bRScu2XBUwkQDKtQP9wVTjApCy1kPYYZd/j9AyhF+hapFQ
FBsZEDXJYBGDjprLfrx4jpoyK/fwiKE47pH2IgdBM9U/zJdxkJqHJ2AkPSZBtDSOW4wPrx4Er4S0A2Bk
MDdZezA0YHDzhcYAZfIJ1U9YMP/1KNfVOzb567sr3//8Av98+8M78g7ou1+fvpx9hT/8X1998e0/vvzh
12/hP395+/WbOaGYf/f93/Nv36aLd2+i77/5E/70n1/mX/7tH98twv+e//Xbvy3e/vnt/su//b188yaM
v0m++0tdf//1r1/8+PzrP/759Gv51//Vrt8t1++yY9v+57//sj6s/vq8r5rmL98lf386/uPxabZ6m5bB
2x+/+o8vHqIfSZlf5l/9/f33x78V797d3Xu9qeypZrFu/xFKjN0ixFvYh62siiYuMdXo7sNZvsaVQn4q
jnmX1Tn6GVwSwtxWbKOJ9tkH22tCxvqSxSLAhbPyrKHcA8vMZv6W+YL2VdCu5a6+NVvNpuyt9rORatoe
qka4t5decW1HM2y4tGPzqtyfRXtVQPKLSXQCEjt6GASScgO5UG1WjNEDP2QAM1EYT7p+DPMCbyWl0vlV
mRFFzDTNo65DJUpwXppZHN4jHsISRGp4lF5GekniHRac2QU3DGTANGt9cgM7DBbSA9tjHKO2PYMIbBX6
GbURxesNQB0a9N/HrCECoBKTjxj1m8Aw4qefJz+9g42J8TWB1zXjWsFPWZlkcdRVcOvCsfkIWGdXhgqX
Hy916bqi+I25OjS2TUrT3rXNLKf5sPqnx7yXhZTV1qiyKZ5zvSmbeSdqiiJH+gUleqI+kbYq1bXrrx4N
trIXwQe6BemG4nth68FoYU1Z1Npc42tccYpfsVc0gunvvzED+JU6egmaM2DJGnJZ0Lo8VIwtIyMrMsJ4
6Wl//Uvwirq5ZLrvlbrc8lVy4cZA1Vfvu7J3x81JakqKlzveoAx3/caIUszlDFrN0t5GXE1glI5H6I9p
B5zwFBo05XVl4/VssvLhYVj5hgxi5zsVI12jIB2vdYlPT60pmkuFzrrG+Usekniw7mq2niXamicxCuRT
xCgyFa9tLnLVcZaXESrVkvPC6bQ6WCezB3lYDNXVRcmTx3kMEw6wJufGNaAkF1j/PEkk+y2ja3pk7YTf
qj25hmaYwK4iYzKUjiYTw/QGjtXDksT/e0P+bt8wkbVvqrir4qjz2QdyRHu6z9K7+8ttdXYUXFTJMUft
m+iX6PSGUU3rcn9leVxYVxRGSFhRbMW/KvMXr40bhEovKhPvNXf0cLNM0FMWI7/OTij3ycLxxgvvJyoR
BhNPcDQFQV9dxlXgBpEpDs923uwxTOpsDDJM6vp0f/4f3kOUE9/8uoHfs6PMw9+7m/xbePode8pQ2mXa
dlFHrrHVuwE+2KNZCW/gCFeYDDFJcm7qyw0sYFDCcen104u41GgC95nTxXA95/AQLtno6Z0Ld8YnF99p
lOUoGcc2xZq4thag40hXTeaA0mryC0mUPHn3ELQqrCiO7SP+l2+/Cr9a0q5Cdme0e9hgD5BwTOJ2tBe2
iwiPHq7Lo6w8jwqSvJAngvyoTPykqep+MflBWkxWPDKwxLuSLRHLcST1/uR5vI5T8SafwBHW44o3AlXy
piymciJ/5yO1/J0aklLspA4iGaUQkxV75dqKqr6Qm7coQ56BUZOaW3ZDMFlU0NfFTLqLEnLLsELJE8yk
ybHOcSdABFdGhZqJCjFn11WVv8vUKrDvZkIWzK6thZBmzgAVdfeikJKvdsHVqCmytqX7Mhr5CenmjBpE
/O2qefH7NTI5Mw3GkCEgpJOan5VpdXZsy5roPOkeZMIyH0clLulnwiFqTc21iMpjlNObreNDVbVIt17E
lpAX0pIQvN5oTeKGYFz5sEYXBMFMWxyP59IkkRMRjphuT1sFOj4NEVVglZbcX8mu8UYlCfzu16WlXXX1
rkf+uA44O6I/PWQZAAOZARpYhIbgYXEY/bhxQ4aSpe0P5WyViQrOTNBjX0SPKE1pcyYRkv0V58MnJYO+
3E/ixzQNaHXIjPHcZB3ymSGiW+JnD5DRmL2qKFDZ0Sn5Cix/pF+zqS7u4a7p9pE5F/brbJpKR9CSX2xr
AF4fSp7ZM9MDQU0sQP5iG5cpWJPvN1lYfc0lsgKHqNuRDAgvxnFKE0dsvDJWxdCBxV07YdEm4Lt6/R6T
OLSt69OWPpa04bYB2PbvL8cV91fIZdxJ9oSVV5NBCNwtPZhyPF1Sc18Bx2baQM7aqbAXRng4hNMopse9
wF4nXK9fO2OD6ik6URvV9rKaZn2rJ/TYZfsgmFRI7j0oZUOVX/ZC35wUKIqszFgACQlxIfFcSnSGdicQ
+wpZH8gIH39kYcBCkjQcqiJQY/vM27x8aT60QNWjMAFA80Od0jzu4nwiJskvSA7RzRcdrI94Nmx/Wao9
HFDQMQAPMJC7uG48nzsmb9C6/NkQsqnjfDj3oGVcPBYh8Q3md/vcPyJjeV9BeHibB4+vB8+PVY5GPgiU
NAijpxwRNSVgrwiZGh4FLxL5PdjrWPTz/VkMtyDjy3U5nHIxh/AKMeXHomzl9hLIhi5/vpqgR4mUIa8S
KKbRnNdlbM2E+crv9wAv0woPCYesSVgjmc+nc/J/ry7T7rmiaS1LXK2mK/x/D68oYVodm+7QB4ywr4co
T9m3JR7sukODOJbn9MDBaZYOOdAwkmNDv3LsGn+mQU2C30Sna1pfwfpwasyf8WynvJ74a9ivAE15DfHn
uVg5/GEhyYl9XvLwnxkgX4lywB8e+AdY+hoKmJM/itIivAeceVj8jFfqEZY/C8UQHc0BgOViQQ4AYHBb
GGTTFibxtIUqobYwCaktTHJqC1VUbWGSVluYBNYWqsxwtQxiw0kGyeEkIDzb8QlMUCQG6RWJSXpFokqv
SEzSKxKT9IpElV6RmKRXJCbpFYkqPVwtg/RwkkF6OMkpPTpWM/Hle4P48r1JfPleFV++N4kv35vEl+9V
8eV7k/jyvUl8+V4VH66WQXw4ySA+nOQWH5momPhOpnHtZBzaTprR7WQc4E7GMe6kGeZOxpHuZBzsTprx
7mQe8k7mUe8kD3yX6f7YdfLWua/s2/h0bZ6CP/8pzqO2/fSPdzjPu5+lHWuyGyeebgFzK0zkWQov6kBO
1sra2MDHQOTgaG1haK3nR/ZLGTvCSy+MH/qGi0Tl4Ic8EWNiSEq0zU281OIq9flt8VtosC1uU6JI9xvo
sbhVlcVvok3bZMlKLpLr1Fkkv4U6i+Q2dYp0H1+dRXKjOgXC31qd/ezNiiZu2BX6zPe/hT7z/W36FOk+
vj7z/Y36FAh/c31yc4IVTZziK/R5yn8LfZ7y2/Qp0n18fZ7yG/UpEH5UfU7pO7n+DCwDDZYXSw5BsmAP
svQ5SKcOPU1ZgBTBNmTpS5AuWIgsfQXv7AmGlAeYovC8BukrhedHkP4g8DwLoDQUpmdQXIMVaTFGGCnx
la2yJk6zXdzEezZJnLjRdqETf9oud+JYm0RPPGy79ImrbVcA8blNOqDOt10N1AvXa8JmSDBy4nhbVUE8
cLsqiCtuUgXxye2qIM65XRXESzepgrjrdlUQv92uCuLAm1RBPXm7KqhLP14VvRHA6IkXb9UFceftuiB+
vUkXxMG364J4+nZdEJffpAvi+9t1QRYB7LogqwEmXdBlAbsu6PrAFbrgEzijJ0sCVl2QtQG7LsgigUkX
ZLXArguybGDXBVk/MOmCLCTYdUFWFOy6IEsLJl3QNQa7Luhig0EX0wKVR2m/fbYEb2maLvS89lleXJLp
jju4MK4cH3e+QtrnbbjuDt6qedWzndqs5eMTmlxFSvW+vP7Z5Q97AFRXhOFoh4X2qntc+SNzPfnw+JN8
iyq8V8n8tpR4GljN1nI2jGuWyZLdn8rWBcnOOt9k0t0TPl+FwaNQ4rBFLwTUmnbrxSco4e2mQn78umph
016NtRbw5E9+YEokAkeIAEn0FHVRo57rFe9mGGQa5ajplGu0ebQowR0QDNs0dEn3W+L6o5K223XNgbrz
dJmuRw4DrAKWVi0BR9yQOW6U4Bn+Zt0dBMHoX8lf2oZLOjjzfIZ2oLRJCvC7aNdq3nIW0+13+K4N7x46
WsC4oUgTPD7UVfNic//Wv8D+MHi5nqdTLmk3TnXenB9+H8rhdzELX2i0zJgXbRkROnVNZJV3H5sFRhnd
hddSCK5Ywuf6SCXcACMAHP+4LieTYz3oNn6a5WS9KGu7s+4Jb4CYtkWU5x7/RO0J+dIyIUos1FynIWZX
/xKdfH7tpZipY0yC54ZHZCgmjHmD4QKreK3JBHvtUnxYmAamG27hsfVA8rHPht/UPuLGbd2596Fy19kf
KEIxSkEGSiceLVdvKl4VT+da9ToGEb+LGoMhglsav0aOiT0UTqr7M/07BKYHTC7T9iiO9jxL0mVo0ohw
HA68IhqHkvhUZ8JhV4EBy0zMtMMySfNje5BqEPSp7guvwfu5ixtMCcU/4aaCwMJngB81Fk1I7a/UFz7R
cfxK61nIQLjhWvdVU+SQJkVcbke97aJ9xQXU6KNcGS7mqFpEt71uQLNsUdTEB03LUecWQEEv4OQxAqES
VTYP1JlRk4H/nCWojxrQFRNr75jEg8QjNn7Y6rw0hIN3MlTHQnethVww6canzhPf22Wvhtwac2ouQGqY
OgS7QNaC4Be2myHD4C4EFmvhn0ntw9iTrWSets3cFvpqqBW7zov4DEWVRLnPrnzlNZxfQcvDYMLAWENI
Re5mJk2UthD2tsZAiydydpP3WQ3EvEx/4GeS/ho99VNKmqPT9pdj22XpCz/PsCFZ+TvUPSNUut03MWPy
TD/MXeeIABJgDM5W4BofYSPP5ozMpkudJQQ6IqxHqHM5NIzx7qImjHUEtvp7SqgXQgpnL3RPw9ZDUYuM
XHgwgdn7Rt70cPmCYoV47MKQtlp8dcac6w1csUYtN1PcunxU6tGepo2BJhXAaWelNEttPuxOzTMuezPz
Zh4N8tViSOvzW5Sn/SITLCA95rl8eMgmHHbUBGKGkPvxffoyraN9VtI26rZGRfAVFulA5kUT8VdbRyX4
MNaefODrEsqNRrqlK3B9sqEtM+Nq7HX5v/ObRKbVej7yihKG51kkaRsTJbMRHO3Z3m5EAsbE4yoyX4a0
j2d7Ak7YWA7FRo0dmTEVSuWooIcpQhKrZOOP9ZPZM5jA2tfoW7Wv5jA9PjYNPdiofmSue0/5AQ6ImPs+
qmFxwzNWEkwjsqn2gSs+T7p3IdhlpixHJIyLwrprqK6xkKVXzX6AmpEnsGs4qt5VVd5ldY39b/U475Da
32RpufaQ/B+42b4fB6dLVHjkrUYyrrExzmP/mc3q05vZdOn5UV3nyG9f2g4Vky/yrHz/bRR/T35+VZXd
5O57tK+Q98Of7yZ/QvkTwgPe5G2TRfmkjcrWb1GTpZO7tzgb7x0WtPdlUf2S3Q2U6ofvX4pdld/1YyMd
i4uq6g6Y9/a4o/csRWWXRXkWtSgR259quJkXtYTD3/QDMafICyzkW46G4OFyz6eC56pJfDzGb3YNit77
+DeYAeoGWW8jjLqueR01WeST45v3YMSe7WZpONc9BDJcSiw2BOMuGWwJM9gSlNsTmQRZ6da7FPurFFfa
ZfeBz397j17SJipQ6zGO2ZXM57SpCuGW5a7qf8wuoHqkLw93Faop7HpFIYH6lloanqQSkdFWS8NSoN0C
lvK07Ssqs4L89MuoQBtYfyE1OTKq6awVPpNnK4oqQZu0ap6jJhETu6zAzll6LImluMFGv5+VYq4Iczld
tKIw/bJiCUapKhBFUgPCLGcVY8nGKHkFQlUgVzKAdSQ3ZQ+PEuuKHiBm9gaMVv/ioWKBqtVl1WoLaJ95
nnTzJnjFVpGXw+XbOIX4FbAIjahavR5wIWx8wHmRE4ZDKdyjfpC9ZX+lGEZgeLhoq3cWjjHyoBhd8K1W
CGrcLmy52iakb1jPkB+hpuLF5uyTpiBdSzQ0cyheIkilUPFYpypebJOaZVt+gGzLsbLVNtreUxtmRPJX
HnXon6+X5C1+AS7JnAia70HJl8nj2mty/VGbqyDeJcyTVvsBxiSIoiVOhVm2aLhJFeYpthIanDWaYaQw
zIT+QNiS5SFzTL0PM8t0QYCiZjqtiYAQqiSASghsGbM6iIt/5kIgeKkFE1VYOCbpPcPMe5SuiTBmyhgQ
ohhM2YtIpVrDiM/YoCZSf+9odOJrIcx4oleFKp/FL8MRffI0wSjL0Sf3qA/OUh7VLdq0qI6aqEN6lp3d
WYssbYOLtt/7SovXZvw8mgWkjhs8TlS+DJWFjAb/FdyfR+iNvgUBB7oui9+/6CdKlmS28jQ5WbnQWQpx
2/pdcyzjqEPgh99FzR51E/DR00Gcb9KEuj3/8Tv4unsCAdv0hRRcqemvqKniPKt3VdQkftb6bJHZWTN3
Jlflsbmh3I2xGHAzRxAE0nEabH76aZRgg/ssGfnss866n42w3/3qKBUwTaO2O2uymwdB0Yr+FecoeCW4
Vbg7AcdqyLs6dlruq2N3G/sS79WxMzIfTOcq75ghwC1gPgDMH+shYaurxbHeGsr9IBdLEg5xP2ZqVY41
UMN0rbcecAXvJR3pkcE9qH5SPeubHk74TSqelUolCROjWIeKnC5t0mDV3DfVs39iI3nwStYxTf4taqqo
N5gugYIZZ10Fz+oTrttDk5Xv/ZOsG/79N1KNtlnKfPe89ZwLfMdRrhvO+Hc937Oljbf4uMtif4d+zVDz
OpiGEy+YeMF0Np94s+nyHrDGixebidBISPrrAFMZOwvFzPoGVB/zFsn1IR91lQltNcHTLFixyTr2WKlP
gsg2WZlmZdYhsU60fNDu55eZZXybzgHrGm3w73ptQG33eQSvzpKQ5snr2cTD/39/WeqTpziV/YvJ3JLJ
ZUqmUtJnz8J2dE/ikTbO7ehdFL9Poxj5yjOqW3OSWAYPG5R1Pw3C5f1lSg1n8UZKtvvBVirFuXzwwcUz
I6No6I6y+GMcHXUCz2pMhJMSW+bim9/jqF7G1mx7A0v0ntgzeAKJ3R2rq3p+RGfdJpAJ7OeikHmc1BrN
0lRHssf+goSeL5JguTaitSVEYbyLYq3Sh8ry3SVyR7sBq+efvLGio6iPTZ3LElqlizCeaTWL8rx6Puse
ztHXN3rRMkRDN40kSdS8P+sC4PVai+L3xEiBJPIdvrPlvY484N02MHTTHkEPQRg75oBjqx3mrjhAeZyq
qfMNSLokAXBk8wYlQsEsGkyLmqlXIGtxoYSTbxyIsyYemgwDLYNXUhX4bcPixcP0guWZ5oLl2b2B3MeO
+bFQc1nqclkas8mxn6Xh5cpsyGEEbWbYZ8Puv+buaGNuZVUi+UJoiN37xF/Wv2cpQ+lgZ9gK14F531Qz
n6VrNX+hbyoUNL5KR3E2RSZrszdyRLbKVRIy/KrckNc6tGhjCUmcpmgh00hvpViG4L04/Boe3pIp+Hiq
g8/niR5uKSNd7mYyER/iFbhumOdwcxnLNFBbBm6/CCswycjV3PLjMtSe7QGvZ+sgQfuJ2ldWy3vl6/09
MBmGvOnDIZvhbV89jEZYBcErL1RWVsgiFek35s5CMM1w4oMeYVQhQvsFgYF6JDz0Y8iNdDZzD2NA3AfM
DZ+uwzVRue8rGQWLWRCYULAONDhSxbJmZW5LBEVHL8OQRRAsJBgGCGuKQ2UiPO23C9brQOlOEijZRTuJ
9Rw7pA7x9xjdoTM90qUoCByXL/U+cNt04otjR4ae8r2lWgPIkZ/hXF2CuijLW7Kkm0cvP1U1Kn/+vD0W
RdS8qCEhaXZCCTs1LR1V6k9P82iRtfxcqhQ5BV7UHMJWQBC0xB4RtIFHXuwjeBVdHouwY0eidbGPSf5I
sgZR/7ypnoFsOc5v0BNqWqTB8ySVrr+SGJDQryr6uYnqc//XBv9LBdGFdQFGP6hAHnHbdlFjCFYmSWZK
VCbGIGczFY2SUgjpZzMZCwU+WwOFzeQR0bSBmiaqxFmHipYJiG7tkS9W4VAaLBqFApnLYEIRSUwCoQS7
qEXk7nqRhH8016RBXXwANOybrm0SEYHqj2kdnG4QgbttcBogBlfL4ES8ZUAqR7vgxKxd6GhNrWIQjCjN
ntYkz+jYVTAOX8WwZdPgLPySfDqagvKUZigE7RuyxFBRhwRrUyChGLQ34LWqI2igN+EEgQEuNV5CYG67
rAaisAmFUdLk6BH672OUUzmSZbzZlvy9i9qs3QS2+8toocXowZ9Cbxr/yZWn10wBbTFuFmiL8RNBW3zA
XCAQXzcdCIRXzggC5Y2TgpDDTfNCW9w0NfRk42eHnuSKCaKnuXKOEGp1xTRBb+29aaYQSK+ZLASyq+YL
ge6mKUOgv2HWAHK6auJoi3FzRzF++iiumkGKWyaR4tp5pLh6Kilumk2K6ycU1jNcc4rtJkYCLJLRkwqF
3jSpkIuXr5lUimTcpFIk4yeVIvmASUUgvm5SEQivnFQEyhsnFSGHmyaVIrlpUunJxk8qPckVk0pPc+Wk
ItTqikmF3h1+06QikF4zqQhkV00qAt1Nk4pAf8OkAuR01aRSJKMmlSIZPalg6PhJhaOvmlQ40ehJhROM
n1Q4xXWTylCb8ZMK6xm3TCr9nbIEme9HzyoUetOsQq5/v2ZWyffjZpV8P35WyfcfMKsIxNfNKgLhlbOK
QHnjrCLkcNOsku9vmlV6svGzSk9yxazS01w5qwi1umJWoS8Y3DSrCKTXzCoC2VWzikB306wi0N8wqwA5
XTWr5PtRs0q+Hz2rYOj4WYWjr5pVONHoWYUTjJ9VOMV1s8pQm/GzCusZN80q/HZsgjzlo2cVCr1pViGP
UFwzq5zycbPKKR8/q5zyD5hVBOLrZhWB8MpZRaC8cVYRcrhpVjnlN80qPdn4WaUnuWJW6WmunFWEWl0x
q9B3VG6aVQTSa2YVgeyqWUWgu2lWEehvmFWAnK6aVU75qFnllI+eVTB0/KzC0VfNKpxo9KzCCcbPKpzi
ulllqM34WYX1DOesMuUxAD59GVx+KRzk3GP5jSPqHSR6PL9yQr2EQo8n8QhSeAIMMyGhnmqMJw/u1Ed1
9uGcmjhOFsCpidx8YodE6QVNZ+21TVqCrqrP6nE5LZRFfUto+lWfN7mMRFMA+24m0hclJOkZ5C1WZlHX
avlBQp8G55+l04Z6bBs3VZ4PWPpbjyWjQI9UxgByXViancY8S91Dr7h3jFwcxiL8hRuCpV6IMSQmf7hn
TIMgYaXDi88jn4+ktG0xooC2GMUp2X80smJfKMf0RTKClyIZxQtZtr6KF2F9hYh9P4KZfD9OhfurmRnM
cpzBKR/BDLbIRzBDjGQzM1PCgZ9m0iFVGO/NQMc8PxsQ9NKfPh9+B6YJhnMyYXq5+MF5kBEcYXsnwm+6
/Cw4Il1uwOVdI+DyrhnZb4YccLcYWRjuZrbybJ1jyAW3/ZEF4r50bYF9DxCEtB8vz/0NJfJmPmSDW/HI
EnGvsJY4xVN1f2E/aFJdn6I55lE0Q6rODCh2Q7rWEihyHz6dpJReguJ9pfxSYsBXOChlFnyFhxIyIQNO
tlpuLey/mIW3tYqlf3lrI73UiEUyE/NU0pshnTKrIHYDgpWuQOSnvzRclIANX8NHKTHiazgpZVZ8DS8l
ZEaFnOx13lqr8mIT6NYhqP4BNOWpz0J4LhPnu1YFFEKm16p4Qkk6a1U48AU2DRclYMPX8FGG6tOoiqJk
VnwNLyVkRoWc7HXeWqvyYhPo1iGo/hk69anYQni4lJzMXKkimp+VK6xlCc3lJ9xWqoDmZ/meYkVbgBVf
x0s519wipehrrjwapOGnnJ+Vi6tkjdmrvrVX6MUq2q1LYv37gMqTtVhWC3AbqGYEWkjP52gGoIX8todm
/IGPFOo4KQErvo6XcqF56lfRmsyOr+OnhAxpMCdH1bf2Cr1YRbt1Sax/snEzD1VZLcWsVUAzACjrKmQ3
QBgPKkZ+N1LHSQlY8XW8lBIzvo6bUmbH1/FTQoY0mJOj6lt7hV6sot26JNa/orlZBKqsVmCODFRJraQZ
OFDltJLNjkCVEnzKU8dJuYKGh4aXciVbHhpuypViemj4KVeS7RGoWrNXfWuv0ItVtFuXxFjoPGBAWqQB
DMgLOBYHrsDemNkvEBL1roEIMHgHIsTkIIgYvY/QFk43AUBMngIAGZ0FgDL4C/aq210Gm1xtXoNTlTaf
QnyP3OBWiBCjZyGCzM6F8vq5niO3iwFAZi8DwCyOBsAZfQ2XIFzuhl3Wdo/DqWSbPyI+KW9wSUSI0SsR
QWbHRHnAXs+R2z0BILOHAmAWJwXgjH6KSxAuV8Uua7u34lSy1ZcR0o3ujIgxezQiyuLUiDCjX4Ol7nZt
AMri3QCczcEBQLOP45SH081xSN3h6Tg1bvWDhHSjKyRizN6QiLI4RCLM6BNh0bvdIoCyeEYAZ3OOANDs
Hznl4XSRHFJ3eElOjVt9KCHd6EaJGLMnJaIszpQIM/pTWPRulwqgLF4VwNkcKwA0+1ZOeTjdK4fUHR6W
U+NW/0tIN7pgIsbshYkoiyMmwoy+GBa92x0DKItHBnA2pwwAzX6ZUx5O18whdYd3ZtusKvwisThXQqLe
uRIBBudKhJicKxGjd66KxOlcAYjJuQIgo3MFUAbnyl51u3Nlk+u1zhVQpc25EpJNzpUIMTpXIsjsXIko
k3OFxex0rgDI7FwBmMW5Ajijc+UShMu5ssv6eucKKNnmXAnJJudKhBidKxFkdq5ElMm5wgJ3OlcAZHau
AMziXAGc0blyCcLlXNllfb1zBZRsda6EdKNzJWLMzpWIsjhXIszoXGGpu50rgLI4VwBnc64A0OxcOeXh
dK4cUr/BuQIatzpXQrrRuRIxZudKRFmcKxFmdK6w6N3OFUBZnCuAszlXAGh2rpzycDpXDqnf4FwBjVud
KyHd6FyJGLNzJaIszpUIMzpXWPRu5wqgLM4VwNmcKwA0O1dOeTidK4fUb3CugMatzpWQbnSuRIzZuRJR
FudKhBmdKyx6t3MFUBbnCuBszhUAmp0rpzyczpVD6jc4V31gXuHne4t3JSTqvSsRYPCuRIjJuxIxeu8q
3zu9KwAxeVcAZPSuAMrgXdmrbveubHK91ruCurS5V0Kyyb0SIUb3SgSZ3SsRZXKvsJyd7hUAmd0rALO4
VwBndK9cgnC5V3ZZX+9eQS3b/Csh2eRfiRCjfyWCzP6ViDL5V1jiTv8KgMz+FYBZ/CuAM/pXLkG4/Cu7
rK/3r6CWrQ6WkG50sESM2cESURYHS4QZHSwsdreDBVAWBwvgbA4WAJodLKc8nA6WQ+o3OFhQ5VYPS0g3
elgixuxhiSiLhyXCjB4Wlr3bwwIoi4cFcDYPCwDNHpZTHk4PyyH1GzwsqHKriyWkG10sEWN2sUSUxcUS
YUYXC8ve7WIBlMXFAjibiwWAZhfLKQ+ni+WQ+g0uFlS51ccS0o0+logx+1giyuJjiTCjj4Vl7/axAMri
YwGczccCQLOP5ZSH08dySP0WH4sfRSr8U27xsYREvY8lAgw+lggx+Vgn5xGiU+70sQDE5GMBkNHHOo04
TmSvut3Hssn1ah8L6NLmYwnJJh9LhBh9LBFk9rFOI84fnfIRPhYAmX0sALP4WKdRZ5FcgnD5WHZZ3+Bj
AS3bfCwh2eRjiRCjjyWCzD7WacThpVM+wscCILOPBWAWH+s06iCTSxAuH8su6xt8LKBlq48lpBt9LBFj
9rFElMXHOo05+XTKx/hYAGXxsQDO5mOdxp2CcsrD6WM5pH6LjwVUbvWxhHSjjyVizD6WiLL4WKcxx6ZO
+RgfC6AsPhbA2Xys07gjVE55OH0sh9Rv8bGAyq0+lpBu9LFEjNnHElEWH+s05szVKR/jYwGUxccCOJuP
dRp3/sopD6eP5ZD6LT4WULnVxxLSjT6WiDH7WCLK4mOdxhzYOuVjfCyAsvhYAGfzsU7jDm855eH0sRxS
d/lY09oPznWUJFm5l8z9uhuSNE5U3QjJOheq3gkArQNV5wJC43DUJ2sRWxvpi4X5rZ0rf9ZLRB5W625I
1LoidSMA9I5IvRMgBjekzgWM1lSvT46CtnbyF2tFti7+/LCX0VqVUQiyXqsyCiXW16qMQllGa1VGIZTR
WpWRvaCtnfzFWpGtiz9/3stIMcnqbkjVm8F1IyAMRnC9EzAmE7jOBZDeQKxPrrK2jgxe7NXZOpn0F72w
Qk2vW4DcQ023W0gVCDX9biELK9R0vAUUVqjpeY6yto4MXuzV2TqZ9Je9sBQzoO6GVL3pVTcCwmB41TsB
YzK76lwA6Y2S+uQqa+vI4MVena2TSX81DOeBKqwVHAYDVVgreZwNVGGtlBE9UIW1kob0QBWWo6ytI4MX
e3W2DiYth0RqejTYaCqIqXprASAMBgPAmGwGANKbDY6yHJaDtS5W48EpQatpIaabrAuAMRoYAGW2MQDM
ZGY4S3RaGo56OYwNp0ytpoiYbrJGAMZokACU2SYBMJNZ4izRaZk46uUwTpwytZsuIsBovQCQ2YABMIsN
A3BGM8ZdqNuScdXOZcw4hWs3dUSA0doBILPBA2AWmwfgjGaPu1C35eOqncv4cQrXbhqJAKN1BEBmAwnA
LDYSwBnNJHehbkvJVTuXseQUrt2UEgFGawqAzAYVgFlsKoAzmlXuQt2Wlat2LuPKFiVe07OBRutKTNVb
VwBhsK4AxmRdAZDeunKU5bCurHW52roCErRaV2K6yboCGKN1BVBm6wrATNaVs0SndeWo1w3WFZCp1boS
003WFcAYrSuAMltXAGayrpwlOq0rR71usK6ATO3WlQgwWlcAZLauAMxiXQGc0bpyF+q2rly1u8W6AsK1
W1ciwGhdAZDZugIwi3UFcEbryl2o27py1e4W6woI125diQCjdQVAZusKwCzWFcAZrSt3oW7rylW7W6wr
IFy7dSUCjNYVAJmtKwCzWFcAZ7Su3IW6rStX7W6xrvoA0ZoeDjKaV2Kq3rwCCIN5BTAm8wqA9OaVoyyH
eWWty9XmFRSh1b4S0032FcAY7SuAMttXAGayr5wlOu0rR71usK+gUK0GlphuMrAAxmhgAZTZwAIwk4Hl
LNFpYDnqdYOBBYVqt7BEgNHCAiCzhQVgFgsL4IwWlrtQt4Xlqt0tFhaUrt3EEgFGEwuAzCYWgFlMLIAz
mljuQt0mlqt2t5hYULp2G0sEGG0sADLbWABmsbEAzmhjuQt121iu2t1iY0Hp2o0sEWA0sgDIbGQBmMXI
AjijkeUu1G1kuWp3k5HFI8RqejrAaGSJqXojCyAMRhbAmIyskzuwyFGWw8iy1uV6IwuI0GpkiekmIwtg
jEYWQJmNrNOYSCRniU4jy1GvW4wsIFSrkSWmm4wsgDEaWQBlNrJOY0KXnCU6jSxHvW4xsoBQ7UaWCDAa
WQBkNrIAzGJknUbFOrkLdRtZrtrdZGQB6dqNLBFgNLIAyGxkAZjFyDqNCo5yF+o2sly1u8nIAtK1G1ki
wGhkAZDZyAIwi5F1GhVN5S7UbWS5aneTkQWkazeyRIDRyAIgs5EFYBYj6zQq/MpdqNvIctXOaWRNa79B
bV2VLXl59/Zxyrqbayziqgo77O2PU4/L9DA7p1XZ+W32K9qE7mryZVVApzTj6SEUsw3HZwvoFkq2cyF5
th6frUgXKs3isBCzleekw1JMlcfUw0pMlceEw2wyPYST6WE+mR4Wk+lhOekpnul7n6tAeqz7Ro2kdo2k
N2oktWskvVEjqV0jqU0jtmxtdJdpalVlalVlGgR+Th+Y7UFyjxb1Ood6tTGty3mt8K6C5Nnl1uJd9cKl
zxSQ1DZvLX3mqhZpvGrpH6fumow1bVwBSS391tLVjNWuQN4Vb4soz22tM0dRIh/mC+rTFuYs87mNq7xq
Np8s16tg9XiZ5gc/rsoElS1K/GPeNdE5z0rUP18MixTAEDYNlzIyQWl0zDsJB2GkpvQNZ/InfR9dfciZ
JJIHoQWY8io0SYtR2aFGxNEvIydzJvvRTPHnxJ18tcUo1myNhxD2z4G7eePPizt5K5KbeesNFKqi/Xht
7kcyl+9vZ44b0oSyfy7czRx/ftzJ3CkfxxxFl1VTRDmwBRaSLUBwuypPrBYDQWVdlGcxGyK6lxxt6BcV
eaxr1MRRiyiXXROVbVo1xaZPADRl5R/LBDW451KKBMVVE5Enp+X30zH6uYnq8/Mh65Df1lGMNmWFP8EH
1FtefQjEnyBw5+8aFL338fj3XDUJ/bnpP6r1Q0V9iPCQp0iND3fhInwM0WWaZy0TFn0TfvitVox220OU
VM9+EjXvz8KHTeDN6hP5X7PfRa/Dh8l8NpkvJ8E0XN5PaGq41CQ/LO9h1vnQJkHeASNdLif8f8G0Jy6q
sqKVTaMiy182d99/9W1VVv53aH/Mo+Zu8q4q2yqP2sndN9kOUe15GHI3+RaVeTV5Vx2bDDUTnBXRxmX6
5B+yJEHl+Slrs12WZ93Lhn4BknnySXqORBz7BJ8u97sIw5KsrfPoZUN+6RB+jPIcwsgnqBDyOc1OKDnT
v/PopTp2G/JJynaXV/H7PkfyS0JkJWngHEJ/ajFSZuJHCZ/m6NTj8A99fgAmfJPQpIlymNw+bS/249nG
LXkOGiV8AncLtS1GybWHjRdtWzilO+Q6UsBtYZexbTZO8LTpFnKRXCXkYkzLLcY13uL69lu4m3BxbSsu
kuuF3JsVCZ7/3VLmoJFSzvcjpJzvR0m5h42Xcr53SnnIdaSU8/0NUub2UYINGbeUOWiklE/5CCmf8lFS
7mHjpXzKnVIech0p5VM+VsrRSR6VD1mC/La4SkOU1ovKBGTZr3LhLIvkhvFLybLvcSTPfH9bdyXEp9xM
PG0bvyrzl3NdtRkxKKNdW+XHDm1pXrP6tOXeYn3a9lvg2+oJNWlePTODZBvnWb1pUNy9DiYe+//7LTEY
sfHJTMvtrmoS1GyCy7Q9VM9+VfppFR/bseXzWxEtxc/q08QD/7qXCtvQIn/1szJBp00YsLKiY1fxwsjf
JFP81+UPbz79xGurYxOjb6O6zsr9D99988e0iQr0XDXvWz+ZRcESLVarXRTOZ8vFY7BKVnEcBLM4WayX
82ncttMiqr1P3/zh/wsAAP//uR4jxH7CAQA=
`
)
