package rbac_test

import (
	"io/ioutil"
	"os"

	"github.com/google/go-cmp/cmp"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"

	rbacv1 "k8s.io/api/rbac/v1"

	"sigs.k8s.io/controller-tools/pkg/genall"
	"sigs.k8s.io/controller-tools/pkg/loader"
	"sigs.k8s.io/controller-tools/pkg/markers"
	"sigs.k8s.io/controller-tools/pkg/rbac"
	"sigs.k8s.io/yaml"
)

var _ = Describe("ClusterRole generated by the RBAC Generator", func() {
	// run this test multiple times to make sure the Rule order is stable.
	const stableTestCount = 5
	for i := 0; i < stableTestCount; i++ {
		It("should match the expected result", func() {
			By("switching into testdata to appease go modules")
			cwd, err := os.Getwd()
			Expect(err).NotTo(HaveOccurred())
			Expect(os.Chdir("./testdata")).To(Succeed()) // go modules are directory-sensitive
			defer func() { Expect(os.Chdir(cwd)).To(Succeed()) }()

			By("loading the roots")
			pkgs, err := loader.LoadRoots(".")
			Expect(err).NotTo(HaveOccurred())

			By("registering RBAC rule marker")
			reg := &markers.Registry{}
			Expect(reg.Register(rbac.RuleDefinition)).To(Succeed())

			By("creating GenerationContext")
			ctx := &genall.GenerationContext{
				Collector: &markers.Collector{Registry: reg},
				Roots:     pkgs,
			}

			By("generating a ClusterRole")
			clusterRole, err := rbac.GenerateClusterRole(ctx, "manager-role")
			Expect(err).NotTo(HaveOccurred())

			By("loading the desired YAML")
			expectedFile, err := ioutil.ReadFile("role.yaml")
			Expect(err).NotTo(HaveOccurred())

			By("parsing the desired YAML")
			var role rbacv1.ClusterRole
			Expect(yaml.Unmarshal(expectedFile, &role)).To(Succeed())

			By("comparing the two")
			Expect(*clusterRole).To(Equal(role), "type not as expected, check pkg/rbac/testdata/README.md for more details.\n\nDiff:\n\n%s", cmp.Diff(*clusterRole, role))
		})
	}
})
