/*
Copyright 2019 The Jetstack cert-manager contributors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package controller

import (
	"context"
	"fmt"
	"sync"
	"time"

	"k8s.io/apimachinery/pkg/util/wait"
	"k8s.io/client-go/tools/cache"
	"k8s.io/client-go/util/workqueue"

	logf "github.com/jetstack/cert-manager/pkg/logs"
)

type runDurationFunc struct {
	fn       func(context.Context)
	duration time.Duration
}

type queueingController interface {
	Register(*Context) (workqueue.RateLimitingInterface, []cache.InformerSynced, []RunFunc, error)
	ProcessItem(ctx context.Context, key string) error
}

type controller struct {
	// ctx is the root golang context for the controller
	ctx context.Context

	// the function that should be called when an item is popped
	// off the workqueue
	syncHandler func(ctx context.Context, key string) error

	// mustSync is a slice of informers that must have synced before
	// this controller can start
	mustSync []cache.InformerSynced

	// additionalInformers is a list of informer 'Run' functions that must be
	// called before starting this controller
	additionalInformers []RunFunc

	// a set of functions that should be called every duration.
	runDurationFuncs []runDurationFunc

	// queue is a reference to the queue used to enqueue resources
	// to be processed
	queue workqueue.RateLimitingInterface
}

type RunFunc func(stopCh <-chan struct{})

// Run starts the controller loop
func (c *controller) Run(workers int, stopCh <-chan struct{}) error {
	ctx, cancel := context.WithCancel(c.ctx)
	defer cancel()
	log := logf.FromContext(ctx)

	log.Info("starting control loop")
	// wait for all the informer caches we depend on are synced
	if !cache.WaitForCacheSync(stopCh, c.mustSync...) {
		return fmt.Errorf("error waiting for informer caches to sync")
	}

	var wg sync.WaitGroup
	for i := 0; i < workers; i++ {
		wg.Add(1)
		// TODO (@munnerz): make time.Second duration configurable
		go wait.Until(func() {
			defer wg.Done()
			c.worker(ctx)
		}, time.Second, stopCh)
	}

	for _, f := range c.runDurationFuncs {
		go wait.Until(func() { f.fn(ctx) }, f.duration, stopCh)
	}

	<-stopCh
	log.Info("shutting down queue as workqueue signaled shutdown")
	c.queue.ShutDown()
	log.V(logf.DebugLevel).Info("waiting for workers to exit...")
	wg.Wait()
	log.V(logf.DebugLevel).Info("workers exited")
	return nil
}

// AdditionalInformers is a list of additional informer 'Run' functions
// that will be started when the shared informer factories 'Start' function
// is called.
func (c *controller) AdditionalInformers() []RunFunc {
	return c.additionalInformers
}

func (b *controller) worker(ctx context.Context) {
	log := logf.FromContext(b.ctx)

	log.V(logf.DebugLevel).Info("starting worker")
	for {
		obj, shutdown := b.queue.Get()
		if shutdown {
			break
		}

		var key string
		// use an inlined function so we can use defer
		func() {
			defer b.queue.Done(obj)
			var ok bool
			if key, ok = obj.(string); !ok {
				return
			}
			log := log.WithValues("key", key)
			log.Info("syncing item")
			if err := b.syncHandler(ctx, key); err != nil {
				log.Error(err, "re-queuing item  due to error processing")
				b.queue.AddRateLimited(obj)
				return
			}
			log.Info("finished processing work item")
			b.queue.Forget(obj)
		}()
	}
	log.V(logf.DebugLevel).Info("exiting worker loop")
}
