'use strict';

exports.__esModule = true;
exports.default = tokenize;
var SINGLE_QUOTE = 39;
var DOUBLE_QUOTE = 34;
var BACKSLASH = 92;
var SLASH = 47;
var NEWLINE = 10;
var SPACE = 32;
var FEED = 12;
var TAB = 9;
var CR = 13;
var OPEN_PARENTHESES = 40;
var CLOSE_PARENTHESES = 41;
var OPEN_CURLY = 123;
var CLOSE_CURLY = 125;
var SEMICOLON = 59;
var ASTERICK = 42;
var COLON = 58;
var AT = 64;
var COMMA = 44;

var RE_AT_END = /[ \n\t\r\f\{\(\)'"\\;/]/g;
var RE_NEW_LINE = /[\r\f\n]/g;
var RE_WORD_END = /[ \n\t\r\f\(\)\{\}:;@!'"\\,]|\/(?=\*)/g;
var RE_BAD_BRACKET = /.[\\\/\("'\n]/;

function tokenize(input) {
    var tokens = [];
    var css = input.css.valueOf();

    var code = void 0,
        next = void 0,
        quote = void 0,
        lines = void 0,
        last = void 0,
        content = void 0,
        escape = void 0,
        nextLine = void 0,
        nextOffset = void 0,
        escaped = void 0,
        escapePos = void 0,
        prev = void 0,
        n = void 0;

    var length = css.length;
    var offset = -1;
    var line = 1;
    var pos = 0;

    function unclosed(what) {
        throw input.error('Unclosed ' + what, line, pos - offset);
    }

    while (pos < length) {
        code = css.charCodeAt(pos);

        if (code === NEWLINE || code === FEED || code === CR && css.charCodeAt(pos + 1) !== NEWLINE) {
            offset = pos;
            line += 1;
        }

        switch (code) {
            case CR:
                if (css.charCodeAt(pos + 1) === NEWLINE) {
                    offset = pos;
                    line += 1;
                    pos += 1;
                    tokens.push(['newline', '\r\n', line - 1]);
                } else {
                    tokens.push(['newline', '\r', line - 1]);
                }
                break;

            case FEED:
            case NEWLINE:
                tokens.push(['newline', css.slice(pos, pos + 1), line - 1]);
                break;

            case SPACE:
            case TAB:
                next = pos;
                do {
                    next += 1;
                    code = css.charCodeAt(next);
                } while (code === SPACE || code === TAB);

                tokens.push(['space', css.slice(pos, next)]);
                pos = next - 1;
                break;

            case OPEN_CURLY:
                tokens.push(['{', '{', line, pos - offset]);
                break;

            case CLOSE_CURLY:
                tokens.push(['}', '}', line, pos - offset]);
                break;

            case COLON:
                tokens.push([':', ':', line, pos - offset]);
                break;

            case SEMICOLON:
                tokens.push([';', ';', line, pos - offset]);
                break;

            case COMMA:
                tokens.push([',', ',', line, pos - offset]);
                break;

            case OPEN_PARENTHESES:
                prev = tokens.length ? tokens[tokens.length - 1][1] : '';
                n = css.charCodeAt(pos + 1);
                if (prev === 'url' && n !== SINGLE_QUOTE && n !== DOUBLE_QUOTE && n !== SPACE && n !== NEWLINE && n !== TAB && n !== FEED && n !== CR) {
                    next = pos;
                    do {
                        escaped = false;
                        next = css.indexOf(')', next + 1);
                        if (next === -1) unclosed('bracket');
                        escapePos = next;
                        while (css.charCodeAt(escapePos - 1) === BACKSLASH) {
                            escapePos -= 1;
                            escaped = !escaped;
                        }
                    } while (escaped);

                    tokens.push(['brackets', css.slice(pos, next + 1), line, pos - offset, line, next - offset]);
                    pos = next;
                } else {
                    next = css.indexOf(')', pos + 1);
                    content = css.slice(pos, next + 1);

                    if (next === -1 || RE_BAD_BRACKET.test(content)) {
                        tokens.push(['(', '(', line, pos - offset]);
                    } else {
                        tokens.push(['brackets', content, line, pos - offset, line, next - offset]);
                        pos = next;
                    }
                }

                break;

            case CLOSE_PARENTHESES:
                tokens.push([')', ')', line, pos - offset]);
                break;

            case SINGLE_QUOTE:
            case DOUBLE_QUOTE:
                quote = code === SINGLE_QUOTE ? '\'' : '"';
                next = pos;
                do {
                    escaped = false;
                    next = css.indexOf(quote, next + 1);
                    if (next === -1) unclosed('quote');
                    escapePos = next;
                    while (css.charCodeAt(escapePos - 1) === BACKSLASH) {
                        escapePos -= 1;
                        escaped = !escaped;
                    }
                } while (escaped);

                content = css.slice(pos, next + 1);
                lines = content.split('\n');
                last = lines.length - 1;

                if (last > 0) {
                    nextLine = line + last;
                    nextOffset = next - lines[last].length;
                } else {
                    nextLine = line;
                    nextOffset = offset;
                }

                tokens.push(['string', css.slice(pos, next + 1), line, pos - offset, nextLine, next - nextOffset]);

                offset = nextOffset;
                line = nextLine;
                pos = next;
                break;

            case AT:
                RE_AT_END.lastIndex = pos + 1;
                RE_AT_END.test(css);
                if (RE_AT_END.lastIndex === 0) {
                    next = css.length - 1;
                } else {
                    next = RE_AT_END.lastIndex - 2;
                }
                tokens.push(['at-word', css.slice(pos, next + 1), line, pos - offset, line, next - offset]);
                pos = next;
                break;

            case BACKSLASH:
                next = pos;
                escape = true;

                nextLine = line;

                while (css.charCodeAt(next + 1) === BACKSLASH) {
                    next += 1;
                    escape = !escape;
                }
                code = css.charCodeAt(next + 1);
                if (escape) {
                    if (code === CR && css.charCodeAt(next + 2) === NEWLINE) {
                        next += 2;
                        nextLine += 1;
                        nextOffset = next;
                    } else if (code === CR || code === NEWLINE || code === FEED) {
                        next += 1;
                        nextLine += 1;
                        nextOffset = next;
                    } else {
                        next += 1;
                    }
                }
                tokens.push(['word', css.slice(pos, next + 1), line, pos - offset, line, next - offset]);
                if (nextLine !== line) {
                    line = nextLine;
                    offset = nextOffset;
                }
                pos = next;
                break;

            default:
                n = css.charCodeAt(pos + 1);

                if (code === SLASH && n === ASTERICK) {
                    next = css.indexOf('*/', pos + 2) + 1;
                    if (next === 0) unclosed('comment');

                    content = css.slice(pos, next + 1);
                    lines = content.split('\n');
                    last = lines.length - 1;

                    if (last > 0) {
                        nextLine = line + last;
                        nextOffset = next - lines[last].length;
                    } else {
                        nextLine = line;
                        nextOffset = offset;
                    }

                    tokens.push(['comment', content, line, pos - offset, nextLine, next - nextOffset]);

                    offset = nextOffset;
                    line = nextLine;
                    pos = next;
                } else if (code === SLASH && n === SLASH) {
                    RE_NEW_LINE.lastIndex = pos + 1;
                    RE_NEW_LINE.test(css);
                    if (RE_NEW_LINE.lastIndex === 0) {
                        next = css.length - 1;
                    } else {
                        next = RE_NEW_LINE.lastIndex - 2;
                    }

                    content = css.slice(pos, next + 1);

                    tokens.push(['comment', content, line, pos - offset, line, next - offset, 'inline']);

                    pos = next;
                } else {
                    RE_WORD_END.lastIndex = pos + 1;
                    RE_WORD_END.test(css);
                    if (RE_WORD_END.lastIndex === 0) {
                        next = css.length - 1;
                    } else {
                        next = RE_WORD_END.lastIndex - 2;
                    }

                    tokens.push(['word', css.slice(pos, next + 1), line, pos - offset, line, next - offset]);
                    pos = next;
                }

                break;
        }

        pos++;
    }

    return tokens;
}
module.exports = exports['default'];
//# sourceMappingURL=data:application/json;base64,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