/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2018, 2019. All Rights Reserved.
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * US Government Users Restricted Rights - Use, duplication or disclosure
 * restricted by GSA ADP Schedule Contract with IBM Corp.
 *******************************************************************************/
import R from 'ramda'
import React from 'react'
import {
  getAge,
  getClusterCount,
  getSearchLink
} from '../../lib/client/resource-helper'
import { Link } from 'react-router-dom'
import config from '../../lib/shared/config'
import { validator } from './validators/hcm-application-validator'
import msgs from '../../nls/platform.properties'
import ChannelLabels from '../components/common/ChannelLabels'

export default {
  defaultSortField: 'name',
  uriKey: 'name',
  primaryKey: 'name',
  secondaryKey: 'namespace',
  validator,
  tableKeys: [
    {
      msgKey: 'table.header.name',
      resourceKey: 'name',
      transformFunction: createApplicationLink
    },
    {
      msgKey: 'table.header.namespace',
      resourceKey: 'namespace'
    },
    {
      msgKey: 'table.header.clusters',
      tooltipKey: 'table.header.application.clusters.tooltip',
      resourceKey: 'clusterCount',
      transformFunction: createClustersLink
    },
    {
      msgKey: 'table.header.resource',
      tooltipKey: 'table.header.application.resource.tooltip',
      resourceKey: 'hubChannels',
      transformFunction: getChannels
    },
    {
      msgKey: 'table.header.timeWindow',
      tooltipKey: 'table.header.application.timeWindow.tooltip',
      resourceKey: 'hubSubscriptions',
      transformFunction: getTimeWindow
    },
    {
      msgKey: 'table.header.created',
      resourceKey: 'created',
      transformFunction: getAge
    }
  ],
  tableActions: [
    {
      key: 'table.actions.applications.view',
      link: {
        url: item =>
          `/multicloud/applications/${encodeURIComponent(
            item.namespace
          )}/${encodeURIComponent(item.name)}`
      }
    },
    {
      key: 'table.actions.applications.edit',
      link: {
        url: item =>
          `/multicloud/applications/${encodeURIComponent(
            item.namespace
          )}/${encodeURIComponent(item.name)}/edit`,
        state: { cancelBack: true }
      }
    },
    {
      key: 'table.actions.applications.search',
      link: {
        url: item =>
          getSearchLink({
            properties: {
              name: item.name,
              namespace: item.namespace,
              kind: 'application',
              apigroup: 'app.k8s.io'
            }
          })
      }
    },
    {
      key: 'table.actions.applications.remove',
      modal: true,
      delete: true
    }
  ],
  detailKeys: {
    title: 'application.details',
    headerRows: ['type', 'detail'],
    rows: [
      {
        cells: [
          {
            resourceKey: 'description.title.name',
            type: 'i18n'
          },
          {
            resourceKey: 'name'
          }
        ]
      },
      {
        cells: [
          {
            resourceKey: 'description.title.namespace',
            type: 'i18n'
          },
          {
            resourceKey: 'namespace'
          }
        ]
      },
      {
        cells: [
          {
            resourceKey: 'description.title.created',
            type: 'i18n'
          },
          {
            resourceKey: 'created',
            transformFunction: getAge
          }
        ]
      },
      {
        cells: [
          {
            resourceKey: 'description.title.labels',
            type: 'i18n'
          },
          {
            resourceKey: 'labels'
          }
        ]
      }
    ]
  }
}

export function createApplicationLink(item = {}, ...param) {
  const { name, namespace = 'default' } = item
  if (param[2]) {
    return item.name
  }
  const link = `${config.contextPath}/${encodeURIComponent(
    namespace
  )}/${encodeURIComponent(name)}`
  return <Link to={link}>{name}</Link>
}

export function createClustersLink(item = {}, locale = '') {
  const clusterCount = R.path(['clusterCount'], item) || {}
  const localPlacement = (R.path(['hubSubscriptions'], item) || []).some(
    sub => sub.localPlacement
  )
  return getClusterCount({
    locale,
    remoteCount: clusterCount.remoteCount,
    localPlacement: localPlacement || clusterCount.localCount,
    name: item.name,
    namespace: item.namespace,
    kind: 'application',
    apigroup: 'app.k8s.io'
  })
}

export function getChannels(item = {}, locale = '') {
  return (
    <ChannelLabels
      channels={(R.path(['hubChannels'], item) || []).map(ch => ({
        type: ch['ch.type'],
        pathname: ch['ch.pathname'],
        gitBranch: ch['sub._gitbranch'],
        gitPath: ch['sub._gitpath']
      }))}
      locale={locale}
    />
  )
}

export function getTimeWindow(item = {}, locale = '') {
  // Check for 'active' or 'blocked' subscription, ignoring 'none'
  return (R.path(['hubSubscriptions'], item) || []).some(sub =>
    ['active', 'blocked'].includes(sub.timeWindow)
  )
    ? msgs.get('table.cell.timeWindow.yes', locale)
    : ''
}
