"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var wrapConnectorHooks_1 = require("./wrapConnectorHooks");
var isRef_1 = require("./utils/isRef");
var shallowEqual = require('shallowequal');
var SourceConnector = /** @class */ (function () {
    function SourceConnector(backend) {
        var _this = this;
        this.backend = backend;
        this.hooks = wrapConnectorHooks_1.default({
            dragSource: function (node, options) {
                _this.dragSourceOptions = options || null;
                if (isRef_1.isRef(node)) {
                    _this.dragSourceRef = node;
                }
                else {
                    _this.dragSourceNode = node;
                }
                _this.reconnectDragSource();
            },
            dragPreview: function (node, options) {
                _this.dragPreviewOptions = options || null;
                if (isRef_1.isRef(node)) {
                    _this.dragPreviewRef = node;
                }
                else {
                    _this.dragPreviewNode = node;
                }
                _this.reconnectDragPreview();
            },
        });
        this.handlerId = null;
        // The drop target may either be attached via ref or connect function
        this.dragSourceRef = null;
        this.dragSourceOptionsInternal = null;
        // The drag preview may either be attached via ref or connect function
        this.dragPreviewRef = null;
        this.dragPreviewOptionsInternal = null;
        this.lastConnectedHandlerId = null;
        this.lastConnectedDragSource = null;
        this.lastConnectedDragSourceOptions = null;
        this.lastConnectedDragPreview = null;
        this.lastConnectedDragPreviewOptions = null;
    }
    SourceConnector.prototype.receiveHandlerId = function (newHandlerId) {
        if (this.handlerId === newHandlerId) {
            return;
        }
        this.handlerId = newHandlerId;
        this.reconnect();
    };
    Object.defineProperty(SourceConnector.prototype, "connectTarget", {
        get: function () {
            return this.dragSource;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(SourceConnector.prototype, "dragSourceOptions", {
        get: function () {
            return this.dragSourceOptionsInternal;
        },
        set: function (options) {
            this.dragSourceOptionsInternal = options;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(SourceConnector.prototype, "dragPreviewOptions", {
        get: function () {
            return this.dragPreviewOptionsInternal;
        },
        set: function (options) {
            this.dragPreviewOptionsInternal = options;
        },
        enumerable: true,
        configurable: true
    });
    SourceConnector.prototype.reconnect = function () {
        this.reconnectDragSource();
        this.reconnectDragPreview();
    };
    SourceConnector.prototype.reconnectDragSource = function () {
        // if nothing has changed then don't resubscribe
        var didChange = this.didHandlerIdChange() ||
            this.didConnectedDragSourceChange() ||
            this.didDragSourceOptionsChange();
        if (didChange) {
            this.disconnectDragSource();
        }
        var dragSource = this.dragSource;
        if (!this.handlerId) {
            return;
        }
        if (!dragSource) {
            this.lastConnectedDragSource = dragSource;
            return;
        }
        if (didChange) {
            this.lastConnectedHandlerId = this.handlerId;
            this.lastConnectedDragSource = dragSource;
            this.lastConnectedDragSourceOptions = this.dragSourceOptions;
            this.dragSourceUnsubscribe = this.backend.connectDragSource(this.handlerId, dragSource, this.dragSourceOptions);
        }
    };
    SourceConnector.prototype.reconnectDragPreview = function () {
        // if nothing has changed then don't resubscribe
        var didChange = this.didHandlerIdChange() ||
            this.didConnectedDragPreviewChange() ||
            this.didDragPreviewOptionsChange();
        if (didChange) {
            this.disconnectDragPreview();
        }
        var dragPreview = this.dragPreview;
        if (!this.handlerId || !dragPreview) {
            return;
        }
        if (didChange) {
            this.lastConnectedHandlerId = this.handlerId;
            this.lastConnectedDragPreview = dragPreview;
            this.lastConnectedDragPreviewOptions = this.dragPreviewOptions;
            this.dragPreviewUnsubscribe = this.backend.connectDragPreview(this.handlerId, dragPreview, this.dragPreviewOptions);
        }
    };
    SourceConnector.prototype.didHandlerIdChange = function () {
        return this.lastConnectedHandlerId !== this.handlerId;
    };
    SourceConnector.prototype.didConnectedDragSourceChange = function () {
        return this.lastConnectedDragSource !== this.dragSource;
    };
    SourceConnector.prototype.didConnectedDragPreviewChange = function () {
        return this.lastConnectedDragPreview !== this.dragPreview;
    };
    SourceConnector.prototype.didDragSourceOptionsChange = function () {
        return !shallowEqual(this.lastConnectedDragSourceOptions, this.dragSourceOptions);
    };
    SourceConnector.prototype.didDragPreviewOptionsChange = function () {
        return !shallowEqual(this.lastConnectedDragPreviewOptions, this.dragPreviewOptions);
    };
    SourceConnector.prototype.disconnectDragSource = function () {
        if (this.dragSourceUnsubscribe) {
            this.dragSourceUnsubscribe();
            this.dragSourceUnsubscribe = undefined;
            this.dragPreviewNode = null;
            this.dragPreviewRef = null;
        }
    };
    SourceConnector.prototype.disconnectDragPreview = function () {
        if (this.dragPreviewUnsubscribe) {
            this.dragPreviewUnsubscribe();
            this.dragPreviewUnsubscribe = undefined;
            this.dragPreviewNode = null;
            this.dragPreviewRef = null;
        }
    };
    Object.defineProperty(SourceConnector.prototype, "dragSource", {
        get: function () {
            return (this.dragSourceNode || (this.dragSourceRef && this.dragSourceRef.current));
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(SourceConnector.prototype, "dragPreview", {
        get: function () {
            return (this.dragPreviewNode ||
                (this.dragPreviewRef && this.dragPreviewRef.current));
        },
        enumerable: true,
        configurable: true
    });
    return SourceConnector;
}());
exports.default = SourceConnector;
