---
layout: "guides"
page_title: "Tokens and Leases - Guides"
sidebar_current: "guides-identity-lease"
description: |-
  Tokens are the core method for authentication within Vault. For every
  authentication token and dynamic secret, Vault creates a lease
  containing information such as duration, renewability, and more.
  Understanding the lifecycle of leases means understanding the lifecycle of
  tokens in some sense.
---

# Tokens and Leases

Almost everything in Vault has an associated lease, and when the lease is
expired, the secret is revoked. Tokens are not an exception. Every non-root
token has a time-to-live (TTL) associated with it. When a token expires and it's
not renewed, the token is automatically revoked.

## Lease Hierarchy

When a new token or secret is created, it is a child of the creator. If the
parent is revoked or expires, so do all its children regardless of their own
leases. A child may be a token, secret, or authentication created by a parent. A
parent is almost always a **token**.

Suppose a hierarchy exists with respective TTL as follows:

```
    b519c6aa... (3h)
        6a2cf3e7... (4h)
        1d3fd4b2... (1h)
            794b6f2f... (2h)
```

In this scenario, the lease ID of `1d3fd4b2..` will expire in an hour. If a
token or secret with a lease is not renewed before the lease expires, it will be
revoked by the Vault server. When it's revoked, it takes its child
(`794b6f2f...`) although the child has one more hour before it expires. Then,
two hours later, `b519c6aa...` will be revoked and takes its child
(`6a2cf3e7...`) with it.


## Reference Material

- The [Validation](/guides/secret-mgmt/dynamic-secrets.html#validation) section of the
[Secret as a Service](/guides/secret-mgmt/dynamic-secrets.html) guide demonstrates lease
renewal and revocation
- [Tokens documentation](/docs/concepts/tokens.html)
- [Token Auth Method (API)](/api/auth/token/index.html)
- [Lease, Renew, and Revoke](/docs/concepts/lease.html)

## Estimated Time to Complete

10 minutes

## Personas

The end-to-end scenario described in this guide involves one persona:

- **`admin`** with privileged permissions to create and manage tokens

See the [policy requirements](#policy-requirements) section for details.

## Challenge

Consider the following scenarios often encountered outside of Vault:

- There is no **break glass** procedure available for revoking
access to credentials in the event of a breach
- Credentials for external systems (e.g. AWS, MySQL) are shared
- Need temporal access to a database in a specific scenario

## Solution

Vault has built-in support for secret revocation. Vault can revoke not only
a single secret, but also a tree of secrets. For example, Vault can revoke all
secrets read by a specific **user** or all secrets of a specific **type**.
Revocation assists in key rolling as well as locking down systems in the case of
an intrusion.

If a user or machine needs a temporal access to Vault, you can set a short TTL
or a number of uses to a token so the token is automatically revoked at the
end of its life.

This also allows for organizations to plan and train for various
"break glass" procedures.

## Prerequisites

To perform the tasks described in this guide, you need to have a Vault
environment.  Refer to the [Getting
Started](/intro/getting-started/install.html) guide to install Vault. Make sure
that your Vault server has been [initialized and
unsealed](/intro/getting-started/deploy.html).


### Policy requirements

-> **NOTE:** For the purpose of this guide, you can use the **`root`** token to work
with Vault. However, it is recommended that root tokens are used for just
enough initial setup or in emergencies. As a best practice, use tokens with
an appropriate set of policies based on your role in the organization.

To perform all tasks demonstrated in this guide, your policy must include the
following permissions:

```shell
# List available auth method - Step 1
path "sys/auth" {
  capabilities = [ "read" ]
}

# Read default token configuration
path "sys/auth/token/tune" {
  capabilities = [ "read", "sudo" ]
}

# Create and manage tokens (renew, lookup, revoke, etc.)
path "auth/token/*" {
  capabilities = [ "create", "read", "update", "delete", "list", "sudo" ]
}

# For Advanced Features - list available secret engines
path "sys/mounts" {
  capabilities = [ "read" ]
}

# For Advanced Features - tune the database secret engine TTL
path "sys/mounts/database/tune" {
  capabilities = [ "update" ]
}
```

If you are not familiar with policies, complete the
[policies](/guides/identity/policies.html) guide.


## Steps

Tokens are the core method for authentication within Vault. Tokens can be used
directly or dynamically generated by the auth methods. Regardless, the clients
need valid tokens to interact with Vault.

This guide demonstrates the lifecycle of tokens.

1. [Read token auth method configuration](#step1)
1. [Create short-lived tokens](#step2)
1. [Create tokens with use limit](#step3)
1. [Periodic tokens](#step4)
1. [Orphan tokens](#step5)
1. [Revoke tokens](#step6)


### <a name="step1"></a>Step 1: Read token auth method configuration

When you create leases with no specific TTL values, the default value applies
to the lease.

```shell
$ vault auth list

Path       Type      Accessor                Default TTL  Max TTL  Replication Behavior  Description
approle/   approle   auth_approle_53f0fb08   system       system   replicated
github/    github    auth_github_b770f4c8    system       system   replicated
token/     token     auth_token_2ad69043     system       system   replicated            token based credentials
userpass/  userpass  auth_userpass_d326d2f9  system       system   replicated
```

The system max TTL is **32 days**, but you can override it to be longer or
shorter in Vault's configuration file.

Another option is to tune the mount configuration to override the system
defaults by calling the **`/sys/mounts/<PATH>/tune`** endpoint (e.g.
`/sys/mounts/database/tune`). For the auth method system configuration, call
**`/sys/auth/<METHOD>/tune`** endpoint.

NOTE: Refer to the [Advanced Features](#advanced-features) section for tuning
the system configuration.

#### CLI command

Read the default TTL settings for **token** auth method:

```shell
$ vault read sys/auth/token/tune

Key              	Value
---              	-----
default_lease_ttl	2764800
force_no_cache   	false
max_lease_ttl    	2764800
```

#### API call using cURL

Use `/sys/auth/token/tune` endpoint to read the default TTL settings for **token** auth
method:

```shell
$ curl --header "X-Vault-Token: <TOKEN>" \
     --request GET \
     <VAULT_ADDRESS>/v1/sys/auth/token/tune
```

Where `<TOKEN>` is your valid token with read permission on the
`sys/auth/token/tune` path.

**Example:**

```shell
$ curl --header "X-Vault-Token: ..." --request GET \
      https://vault.rocks/v1/sys/auth/token/tune | jq
{
  "default_lease_ttl": 2764800,
  "max_lease_ttl": 2764800,
  "force_no_cache": false,
  "request_id": "630fd49d-f704-540f-0641-41516087654f",
  "lease_id": "",
  "renewable": false,
  "lease_duration": 0,
  "data": {
    "default_lease_ttl": 2764800,
    "force_no_cache": false,
    "max_lease_ttl": 2764800
  },
  "wrap_info": null,
  "warnings": null,
  "auth": null
}
```

-> **NOTE:** The returned TTL value is in seconds (2764800 seconds = 32 days).

### <a name="step2"></a>Step 2: Create short-lived tokens

Create a new token with TTL of 30 seconds which means that the token gets
automatically revoked after 30 seconds.

#### CLI command

To view optional parameters to create tokens:

```shell
$ vault token create -help
```

There are a number of parameters you can set.  To specify the token TTL, pass
the value using `-ttl` parameter.

**Example:**

```shell
# Create a token with TTL of 30 seconds
$ vault token create -ttl=30s
Key            	Value
---            	-----
token          	7544266f-3ec9-81a6--data504-e258b89de862
token_accessor 	59aae2f1-2e97-6ebb-f925-8a97cf5a9942
token_duration 	30s
token_renewable	true
token_policies 	[admin]

# Test the new token
$ VAULT_TOKEN=3b2b1285-844b-4b40-6afa-623f39c1b738 vault token lookup
Key             	Value
---             	-----
accessor        	2b2b5b83-7f22-fecd-03f0-4e25bf64da11
creation_time   	1515702047
creation_ttl    	30
display_name    	token
expire_time     	2018-01-11T20:21:17.900969673Z
explicit_max_ttl	0
id              	3b2b1285-844b-4b40-6afa-623f39c1b738
issue_time      	2018-01-11T20:20:47.90096937Z
meta            	<nil>
num_uses        	0
orphan          	false
path            	auth/token/create
policies        	[admin]
renewable       	true
ttl             	8
```

**NOTE:** The `vault token lookup` command returns the token's properties.
In this example, it shows that this token has 8 more seconds before it expires.

When you execute a Vault command using the new token immediately following its
creation, it should work. Wait for 30 seconds and try again. It returns
**`Code: 403. Errors:`** which indicates a forbidden API call due to expired
token usage.


You can **renew** the token's TTL as long as the token has not expired.

```shell
$ vault token renew <TOKEN>
```

If you want to renew and extend the token's TTL, pass the desired extension:

```shell
$ vault token renew <TOKEN> <EXTENSION>
```

The extension value can be an integer number of seconds (e.g. 3600) or a string
duration (e.g. "1h").


#### API call using cURL

Use the `auth/token/create` endpoint to create a new token. There are a number of
optional [parameters](/api/auth/token/index.html#create-token) that you can pass
in the request payload.

**Example:**

The following example sets the `ttl` parameter.

```shell
# Create a new token with TTl of 30 seconds
$ curl --header "X-Vault-Token: ..." --request POST \
     --data '{"ttl": "30s"}' \
     https://vault.rocks/v1/auth/token/create | jq
{
  ...
 "auth": {
   "client_token": "f7d88963-1aba-64d7-11a0-9282ae7681d0",
   "accessor": "c0a40d94-b814-e46f-7e56-ee18fccdf1b6",
   "policies": [
     "admin"
   ],
   "metadata": null,
   "lease_duration": 30,
   "renewable": true
 }
}

# Pass the returned token (`client_token`) in the `X-Vault-Token` header to test
$ curl --header "X-Vault-Token: f7d88963-1aba-64d7-11a0-9282ae7681d0" \
     --request GET \
     https://vault.rocks/v1/auth/token/lookup-self | jq
{
  ...
 "data": {
   "accessor": "a54fea3f-6c09--data288-ede5-53288569f988",
   "creation_time": 1515702669,
   "creation_ttl": 30,
    ...
   "renewable": true,
   "ttl": 14
 },
 ...
}
```

When you invoke the API using the new token immediately following its
creation, it should work. Wait for 30 seconds and try again. It returns
**`Code: 403. Errors:`** which indicates a forbidden API call due to expired
token usage.

#### Renew the token:

```shell
$ curl --header "X-Vault-Token: ..." --request POST \
       https://vault.rocks/v1/auth/token/renew/<TOKEN> | jq

# Renew token with 1 hour extension
$ curl --header "X-Vault-Token: ..." --request POST \
       --data '{"increment": "3600"}' \
       https://vault.rocks/v1/auth/token/renew/<TOKEN> | jq
```

-> **NOTE:** Tokens can be renewed as long as its life hasn't reached its max
TTL. For example, if the token's TTL is 1 hour and max TTL is 24 hours, you can
renew the token up to 24 hours from its creation time. Once 24 hours has passed from
the token's creation time, the token is revoked by Vault. For long running
processes, this may introduce complexity. In such case, use [periodic tokens](#step4).


### <a name="step3"></a>Step 3: Create tokens with use limit

In addition to TTL and max TTL, tokens may be limited to a number of uses. Use
limit tokens expire at the end of their last use regardless of their remaining
TTLs. On the same note, use limit tokens expire at the end of their TTLs
regardless of their remaining uses.

To create tokens with a use limit, set the number of uses when you
create them.

#### CLI command

Create a token with the `-use-limit` property argument.

**Example:**

```shell
$ vault token create -policy=default -use-limit=2

Key            	Value
---            	-----
token          	bd39178e-176e-cc91-3930-94f7b0194de5
token_accessor 	a230f5ab-b59f--datab0b-855d-36ea4319b58e
token_duration 	768h0m0s
token_renewable	true
token_policies 	[default]
```

This creates a token with the _default_ policy and a use limit of 2.

#### Verification

```shell
$ VAULT_TOKEN=bd39178e-176e-cc91-3930-94f7b0194de5 vault token lookup

Key             	Value
---             	-----
accessor        	a230f5ab-b59f--datab0b-855d-36ea4319b58e
creation_time   	1515710251
creation_ttl    	2764800
display_name    	token
expire_time     	2018-02-12T22:37:31.715486503Z
explicit_max_ttl	0
id              	bd39178e-176e-cc91-3930-94f7b0194de5
issue_time      	2018-01-11T22:37:31.715486221Z
meta            	<nil>
num_uses        	1
orphan          	false
path            	auth/token/create
policies        	[default]
renewable       	true
ttl             	2764769


$ VAULT_TOKEN=bd39178e-176e-cc91-3930-94f7b0194de5 vault write cubbyhole/token \
    value=bd39178e-176e-cc91-3930-94f7b0194de5

Success! Data written to: cubbyhole/token


$ VAULT_TOKEN=bd39178e-176e-cc91-3930-94f7b0194de5 vault read cubbyhole/token
Error reading cubbyhole/token: Error making API request.

URL: GET http://127.0.0.1:8200/v1/cubbyhole/token
Code: 403. Errors:

* permission denied
```

The first command read the token's properties and then wrote a value to the cubbyhole
secret engine. This exhausted the use limit of 2 for this token.  Therefore,
the attempt to read the secret from the cubbyhole failed.


#### API call using cURL

Set the `num_uses` property in the request payload.

```shell
$ curl --header "X-Vault-Token: ..." --request POST  \
       --data '{ "policies": ["default"], "num_uses":2 }' \
       https://vault.rocks/v1/auth/token/create | jq
{
  "request_id": "0e98ff80-2825-7f50-6522-b6f95d596ef4",
  "lease_id": "",
  "renewable": false,
  "lease_duration": 0,
  "data": null,
  "wrap_info": null,
  "warnings": null,
  "auth": {
    "client_token": "d9c2f2e5-6b8a-4021-476c-ebd3f166d668",
    "accessor": "4dd5ef0d-8515-c3ae-ea49-016c3e9eb968",
    "policies": [
      "default"
    ],
    "metadata": null,
    "lease_duration": 2764800,
    "renewable": true
  }
}
```

This creates a token with the _default_ policy and a use limit of 2.

#### Verification

```text
$ curl --header "X-Vault-Token: d9c2f2e5-6b8a-4021-476c-ebd3f166d668" \
       --request GET \
       https://vault.rocks/v1/auth/token/lookup-self | jq
{
  "request_id": "77be1321-c0ca-e099-6f92-4ad87133b044",
  "lease_id": "",
  "renewable": false,
  "lease_duration": 0,
  "data": {
    "accessor": "4dd5ef0d-8515-c3ae-ea49-016c3e9eb968",
    "creation_time": 1515711922,
    "creation_ttl": 2764800,
    "display_name": "token",
    "expire_time": "2018-02-12T23:05:22.746137253Z",
    "explicit_max_ttl": 0,
    "id": "d9c2f2e5-6b8a-4021-476c-ebd3f166d668",
    "issue_time": "2018-01-11T23:05:22.746136892Z",
    "meta": null,
    "num_uses": 1,
    ...
}

$ curl --header "X-Vault-Token: d9c2f2e5-6b8a-4021-476c-ebd3f166d668" \
       --request POST \
       --data '{ "value": "d9c2f2e5-6b8a-4021-476c-ebd3f166d668" }' \
       https://vault.rocks/v1/cubbyhole/token


$ curl --header "X-Vault-Token: d9c2f2e5-6b8a-4021-476c-ebd3f166d668" \
       --request GET \
       https://vault.rocks/v1/cubbyhole/token | jq
{
  "errors": [
    "permission denied"
  ]
}
```

The first command read the token's properties and then wrote a value to the cubbyhole
secret engine. This exhausted the use limit of 2 for this token.  Therefore,
the attempt to read the secret from the cubbyhole failed.



### <a name="step4"></a>Step 4: Periodic tokens

**Root** or **sudo** users have the ability to generate **periodic tokens**.
Periodic tokens have a TTL, but no max TTL; therefore, they may live for an
infinite duration of time so long as they are renewed within their TTL. This
is useful for long-running services that cannot handle regenerating a token.

#### CLI command

First, create a token role with a specific `period`.  When you set `period`,
tokens created for this role will have no max TTL. Instead, the `period` becomes
the token renewal period. This value can be an integer value in seconds (e.g.
2764800) or a string duration (e.g. 72h).

```shell
$ vault write auth/token/roles/<ROLE_NAME> allowed_policies="<POLICY_NAMES>" period=<RENEWAL_PERIOD>
```

**Example:**

```shell
$ vault write auth/token/roles/zabbix allowed_policies="default" period="24h"
```

Now, generate a token:

```shell
$ vault token create -role=zabbix

Key            	Value
---            	-----
token          	de91ebba-20ad-18ba-fa43-08e1932de301
token_accessor 	1f8abad0-c1db-9399-15ee--datad4b6230386c
token_duration 	24h0m0s
token_renewable	true
token_policies 	[default]
```


#### API call using cURL

First, create a token role by setting `period`.  When you set `period`, tokens
created for this role will have no max TTL. Instead, the `period` becomes the
token renewal period. This value can be an integer value in seconds (e.g.
2764800) or a string duration (e.g. 72h).

**Example:**

```shell
$ curl --header "X-Vault-Token: ..." --request POST \
       --data @payload.json \
       https://vault.rocks/v1/auth/token/roles/zabbix

$ cat payload.json
{
	"allowed_policies": [
		"default"
	],
	"period": "24h"
}
```

This creates a token role named `zabbix` with `default` policies attached.
Its renewal period is set to 24 hours.

Now, generate a token:

```plaintext
$ curl --header "X-Vault-Token: ..." --request POST \
     https://vault.rocks/v1/auth/token/create/zabbix | jq
{
  ...
  "auth": {
    "client_token": "a59c0d41-8df7-ba8e-477e-9bfb394f28a0",
    "accessor": "c2023006-ce8d-532b-136f-330223ccf464",
    "policies": [
      "default"
    ],
    "metadata": null,
    "lease_duration": 86400,
    "renewable": true,
    "entity_id": ""
  }
```

Generated tokens are renewable indefinitely as long as they are renewed
before the lease duration expires. The token renew command was covered in
[Step 2](#step2).


#### Additional Note: Periodic Tokens with AppRole

It probably makes better sense to create **AppRole** periodic tokens since we
are talking about long-running apps that need to be able to renew their token
indefinitely.

-> For more details about AppRole, read the [AppRole Pull
Authentication](/guides/identity/authentication.html) guide.

To create AppRole periodic tokens, create your AppRole role with
`period` specified.

**Example:**

```plaintext
$ vault write auth/approle/role/jenkins policies="jenkins" period="72h"
```

Or

```plaintext
$ curl --header "X-Vault-Token:..." --request POST \
       --data @payload.json \
       https://vault.rocks/v1/auth/approle/role/jenkins

$ cat payload.json
{
  "allowed_policies": [
		"jenkins"
	],
  "period": "72h"  
}
```



### <a name="step5"></a>Step 5: Orphan tokens

**Root** or **sudo users** have the ability to generate **orphan** tokens. Orphan tokens
are **not** children of their parent; therefore, orphan tokens do not expire when their
parent does.


**NOTE:** Orphan tokens still expire when their own max TTL is reached.

#### CLI command

```shell
$ vault token create -orphan
```

#### API call using cURL

```shell
$ curl --header "X-Vault-Token:..." --request POST \
       --data '{ "no_parent": true }' \
       https://vault.rocks/v1/auth/token/create-orphan | jq
```


### <a name="step6"></a>Step 6: Revoke tokens and leases

Revoking a token and all its children.

#### CLI command

To revoke a specific token:

```shell
$ vault token revoke <TOKEN>
```

To revoke all leases under a specific path:

```shell
$ vault lease revoke -prefix <PATH>
```

**Example:**

```shell
# Revoke a specific token
$ vault token revoke eeaf890e-4b0f-a687-4190-c75b1d6d70bc

# Revoke all leases for database auth method
$ vault lease revoke -prefix database/creds

# Revoke all tokens
$ vault lease revoke -prefix auth/token/create
```


#### API call using cURL

To revoke a specific token, call `/auth/token/revoke` endpoint.  If you want to revoke tokens/secrets under a specific path, call `/sys/leases/revoke-prefix/<PATH>`.


**Example:**

```shell
# Revoke a specific token
$ curl --header "X-Vault-Token:..." --request POST \
       --data '{ "token": "eeaf890e-4b0f-a687-4190-c75b1d6d70bc" }' \
       https://vault.rocks/v1/auth/token/revoke

# Revoke all secrets for database auth method
$ curl --header "X-Vault-Token:..." --request POST \
       https://vault.rocks/v1/sys/leases/revoke-prefix/database/creds

# Revoke all tokens
$ curl --header "X-Vault-Token:..." --request POST \
       https://vault.rocks/v1/sys/leases/revoke-prefix/auth/token/create
```


## Advanced Features

It is important to understand lease configuration to avoid having your
secret leases expire earlier than you expected.

#### 1. Determine the TTLs specific to the mount

```shell
$ vault secrets list

Path        Type       Accessor            Plugin  Default TTL  Max TTL  Force No Cache  Replication Behavior  Seal Wrap  Description
cubbyhole/  cubbyhole  cubbyhole_36021b8e  n/a     n/a          n/a      false           local                 false      per-token private secret storage
database/   database   database_e21b9b4f   n/a     system       system   false           replicated            false
identity/   identity   identity_035fe03b   n/a     n/a          n/a      false           replicated            false      identity store
pki/        pki        pki_9ae09eb3        n/a     system       system   false           replicated            false
secret/     kv         kv_2e59ba96         n/a     system       system   false           replicated            false      key/value secret storage
ssh/        ssh        ssh_ea06b9bb        n/a     system       system   false           replicated            false
sys/        system     system_f5b5ecac     n/a     n/a          n/a      false           replicated            false      system endpoints used for control, policy and debugging
transit/    transit    transit_07fc2df9    n/a     system       system   false           replicated            false
```

Notice the **Default TTL** and **Max TTL** columns.

#### 2. Tune the system TTLs

Override the global defaults by specifying `default_lease_ttl` and
`max_lease_ttl` to meet your requirements.

**Example:**

The following example assumes that you have a database secret engine configured.

```shell
$ vault write sys/mounts/database/tune default_lease_ttl="8640"
```

Or

```shell
$ curl --header "X-Vault-Token:..." --request POST \
       --data '{ "max_lease_ttl": 129600}' \
       https://vault.rocks/v1/sys/mounts/database/tune
```



#### 3. Check the role specific TTLs

Depending on the auth method, there may be more specific TTLs configured (e.g.
roles, groups, users) as you have done so in [Step 4](#step4).

```shell
$ vault read auth/token/roles/zabbix

Key                	Value
---                	-----
allowed_policies   	[default]
disallowed_policies	[]
explicit_max_ttl   	0
name               	zabbix
orphan             	false
path_suffix
period             	86400
renewable          	true
```


## Next steps

Now that you have learned the lifecycle of tokens and leases, read the [AppRole Pull
Authentication](/guides/identity/authentication.html) guide to learn how to generate
tokens for apps or machines.
