"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _path = require("path");

var _arrify = _interopRequireDefault(require("arrify"));

var _micromatch = require("micromatch");

var _options = require("./options");

var _linter = _interopRequireDefault(require("./linter"));

var _utils = require("./utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// @ts-ignore

/** @typedef {import('webpack').Compiler} Compiler */

/** @typedef {import('./options').Options} Options */
const ESLINT_PLUGIN = 'ESLintWebpackPlugin';
let counter = 0;

class ESLintWebpackPlugin {
  /**
   * @param {Options} options
   */
  constructor(options = {}) {
    this.options = (0, _options.getOptions)(options);
    this.run = this.run.bind(this);
  }
  /**
   * @param {Compiler} compiler
   * @returns {void}
   */


  apply(compiler) {
    // Generate key for each compilation,
    // this differentiates one from the other when being cached.
    this.key = compiler.name || `${ESLINT_PLUGIN}_${counter += 1}`; // If `lintDirtyModulesOnly` is disabled,
    // execute the linter on the build

    if (!this.options.lintDirtyModulesOnly) {
      compiler.hooks.run.tapPromise(ESLINT_PLUGIN, this.run);
    } // TODO: Figure out want `compiler.watching` is and how to use it in Webpack5.
    // From my testing of compiler.watch() ... compiler.watching is always
    // undefined (webpack 4 doesn't define it either) I'm leaving it out
    // for now.


    let isFirstRun = this.options.lintDirtyModulesOnly;
    compiler.hooks.watchRun.tapPromise(ESLINT_PLUGIN, c => {
      if (isFirstRun) {
        isFirstRun = false;
        return Promise.resolve();
      }

      return this.run(c);
    });
  }
  /**
   * @param {Compiler} compiler
   */


  async run(compiler) {
    // Do not re-hook
    if ( // @ts-ignore
    compiler.hooks.thisCompilation.taps.find( // @ts-ignore
    ({
      name
    }) => name === ESLINT_PLUGIN)) {
      return;
    }

    const options = { ...this.options,
      exclude: (0, _utils.parseFiles)(this.options.exclude || 'node_modules', this.getContext(compiler)),
      extensions: (0, _arrify.default)(this.options.extensions),
      files: (0, _utils.parseFiles)(this.options.files || '', this.getContext(compiler))
    };
    const wanted = (0, _utils.parseFoldersToGlobs)(options.files, options.extensions);
    const exclude = (0, _utils.parseFoldersToGlobs)(options.exclude, []);
    compiler.hooks.thisCompilation.tap(ESLINT_PLUGIN, compilation => {
      /** @type {import('./linter').Linter} */
      let lint;
      /** @type {import('./linter').Reporter} */

      let report;

      try {
        ({
          lint,
          report
        } = (0, _linter.default)(this.key, options, compilation));
      } catch (e) {
        compilation.errors.push(e);
        return;
      } // Gather Files to lint


      compilation.hooks.finishModules.tap(ESLINT_PLUGIN, modules => {
        /** @type {string[]} */
        const files = []; // @ts-ignore

        for (const {
          resource
        } of modules) {
          if (resource) {
            const [file] = resource.split('?');

            if (file && !files.includes(file) && (0, _micromatch.isMatch)(file, wanted) && !(0, _micromatch.isMatch)(file, exclude)) {
              files.push(file);
            }
          }
        }

        if (files.length > 0) {
          lint(files);
        }
      }); // await and interpret results

      compilation.hooks.additionalAssets.tapPromise(ESLINT_PLUGIN, processResults);

      async function processResults() {
        const {
          errors,
          warnings,
          generateReportAsset
        } = await report();

        if (warnings && !options.failOnWarning) {
          // @ts-ignore
          compilation.warnings.push(warnings);
        } else if (warnings && options.failOnWarning) {
          // @ts-ignore
          compilation.errors.push(warnings);
        }

        if (errors && options.failOnError) {
          // @ts-ignore
          compilation.errors.push(errors);
        }

        if (generateReportAsset) {
          await generateReportAsset(compilation);
        }
      }
    });
  }
  /**
   *
   * @param {Compiler} compiler
   * @returns {string}
   */


  getContext(compiler) {
    if (!this.options.context) {
      return String(compiler.options.context);
    }

    if (!(0, _path.isAbsolute)(this.options.context)) {
      return (0, _path.join)(String(compiler.options.context), this.options.context);
    }

    return this.options.context;
  }

}

var _default = ESLintWebpackPlugin;
exports.default = _default;